package com.allendowney.thinkdast;

import java.io.IOException;
import java.util.LinkedList;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Queue;

import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;

import redis.clients.jedis.Jedis;


public class WikiCrawler {
   // przechowuje punkt startowy
   @SuppressWarnings("unused")
   private final String source;

   // indeks, w którym zapisywane są wyniki
   private JedisIndex index;

   // kolejka ciągów URL wskazujących strony, które mają zostać zindeksowane
   private Queue<String> queue = new LinkedList<String>();

   // pobieracz wykorzystywany do pobrania stron z serwisu Wikipedia
   final static WikiFetcher wf = new WikiFetcher();

   /**
    * Konstruktor.
    *
    * @param source
    * @param index
    */
   public WikiCrawler(String source, JedisIndex index) {
      this.source = source;
      this.index = index;
      queue.offer(source);
   }

   /**
    * Zwraca liczbę ciągów URL w kolejce.
    *
    * @return
    */
   public int queueSize() {
      return queue.size();
   }

   /**
    * Pobiera URL z kolejki i indeksuje go.
    * @param testing
    *
    * @return URL indeksowanej strony.
    * @throws IOException
    */
   public String crawl(boolean testing) throws IOException {
      if (queue.isEmpty()) {
         return null;
      }
      String url = queue.poll();
      System.out.println("Pełzanie po " + url);

      if (testing==false && index.isIndexed(url)) {
         System.out.println("Już zindeksowana.");
         return null;
      }
      
      Elements paragraphs;
      if (testing) {
         paragraphs = wf.readWikipedia(url);
      } else {
         paragraphs = wf.fetchWikipedia(url);
      }
      index.indexPage(url, paragraphs);
      queueInternalLinks(paragraphs);     
      return url;
   }

   /**
    * Parsuje akapity i dodaje wewnętrzne łącza do kolejki.
    * 
    * @param paragraphs
    */
   // UWAGA: brak modyfikatora poziomu dostępu oznacza poziom pakietu
   void queueInternalLinks(Elements paragraphs) {
      for (Element paragraph: paragraphs) {
         queueInternalLinks(paragraph);
      }
   }

   /**
    * Parsuje akapit i dodaje wewnętrzne łącza do kolejki.
    * 
    * @param paragraph
    */
   private void queueInternalLinks(Element paragraph) {
      Elements elts = paragraph.select("a[href]");
      for (Element elt: elts) {
         String relURL = elt.attr("href");
         
         if (relURL.startsWith("/wiki/")) {
            String absURL = "https://en.wikipedia.org" + relURL;
            //System.out.println(absURL);
            queue.offer(absURL);
         }
      }
   }

   public static void main(String[] args) throws IOException {
      // utwórz WikiCrawler
      Jedis jedis = JedisMaker.make();
      JedisIndex index = new JedisIndex(jedis);
      String source = "https://en.wikipedia.org/wiki/Java_(programming_language)";
      WikiCrawler wc = new WikiCrawler(source, index);
      
      // w celach testowych załaduj kolejkę
      Elements paragraphs = wf.fetchWikipedia(source);
      wc.queueInternalLinks(paragraphs);

      // powtarzaj aż do zindeksowania strony
      String res;
      do {
         res = wc.crawl(false);
      } while (res == null);
      
      Map<String, Integer> map = index.getCounts("the");
      for (Entry<String, Integer> entry: map.entrySet()) {
         System.out.println(entry);
      }
   }
}