# Ten przykład odwołuje się do kodu Miguel Grinberga na Githubie:
# https://github.com/miguelgrinberg/oreilly-flask-apis-video
#

from flask import Flask, url_for, jsonify, request,\
    make_response, copy_current_request_context, g
from flask_sqlalchemy import SQLAlchemy
from chapter9_pexpect_1 import show_version
import uuid
import functools
from threading import Thread
from werkzeug.security import generate_password_hash, check_password_hash
from flask_httpauth import HTTPBasicAuth

app = Flask(__name__)
app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///network.db'
db = SQLAlchemy(app)
auth = HTTPBasicAuth()

background_tasks = {}
app.config['AUTO_DELETE_BG_TASKS'] = True


class ValidationError(ValueError):
    pass

# Dwie funkcje haseł pochodzą z Flask Werkzeug
class User(db.Model):
    __tablename__ = 'users'
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(64), index=True)
    password_hash = db.Column(db.String(128))

    def set_password(self, password):
        self.password_hash = generate_password_hash(password)

    def verify_password(self, password):
        return check_password_hash(self.password_hash, password)


class Device(db.Model):
    __tablename__ = 'devices'
    id = db.Column(db.Integer, primary_key=True)
    hostname = db.Column(db.String(64), unique=True)
    loopback = db.Column(db.String(120), unique=True)
    mgmt_ip = db.Column(db.String(120), unique=True)
    role = db.Column(db.String(64))
    vendor = db.Column(db.String(64))
    os = db.Column(db.String(64))

    def get_url(self):
        return url_for('get_device', id=self.id, _external=True)

    def export_data(self):
        return {
            'self_url': self.get_url(),
            'hostname': self.hostname,
            'loopback': self.loopback,
            'mgmt_ip': self.mgmt_ip,
            'role': self.role,
            'vendor': self.vendor,
            'os': self.os
        }

    def import_data(self, data):
        try:
            self.hostname = data['hostname']
            self.loopback = data['loopback']
            self.mgmt_ip = data['mgmt_ip']
            self.role = data['role']
            self.vendor = data['vendor']
            self.os = data['os']
        except KeyError as e:
            raise ValidationError('Nieprawidłowe urządzenie: brakuje ' + e.args[0])
        return self


def background(f):
    """Dekorator, który uruchamia opakowaną funkjcę jako zadanie wykonywane
    w tle. Zakłada się, że ta funkcja tworzy nowy zasób, co jest operacją
    długotrwałą. Odpowiedź ma kod statusu 202 Accepted i zawiera nagłówek
    Location z adresem URL zasobu zadania. Wysyłanie żądania GET na ten adres
    będzie powodowało zwracanie odpowiedzi 202 tak długo, jak zadanie będzie 
    realizowane. Kiedy zadanie zostanie zakończone, zostanie zwrócona odpowiedź
    z kodem 303 See Other zawierająca nagłówek Location z adresem URL nowo
    utworzonego zasobu. Następnie klient musi wykonać żądanie DELETE do zasobu
    zadania, aby usunąć go z systemu."""
    @functools.wraps(f)
    def wrapped(*args, **kwargs):
        # Zadanie wykonywane w tle należy poprzedzić dekoratorem Flaska
        # copy_current_request_context, aby mieć dosęp do danych kontekstu.
        @copy_current_request_context
        def task():
            global background_tasks
            try:
                # Wywołujemy opakowaną funkcję i rejestrujemy zwróconą
                # odpowiedź w słowniku background_tasks 
                background_tasks[id] = make_response(f(*args, **kwargs))
            except:
                # Opakowana funkcja zgłosiła wyjątek, zwracamy odpowiedź 500
                background_tasks[id] = make_response(internal_server_error())

        # Zapisujemy zadanie wykonywane w tle pod losowym identyfikatorem 
        # i je uruchamiamy
        global background_tasks
        id = uuid.uuid4().hex
        background_tasks[id] = Thread(target=task)
        background_tasks[id].start()

        # zwracamy odpowiedź 202 Accepted z adresem URL zasobu statusu zadania
        return jsonify({}), 202, {'Location': url_for('get_task_status', id=id)}
    return wrapped

# g jest obiektem kontekstu żądania tworzonym przez Flaska
@auth.verify_password
def verify_password(username, password):
    g.user = User.query.filter_by(username=username).first()
    if g.user is None:
        return False
    return g.user.verify_password(password)

@app.before_request
@auth.login_required
def before_request():
    pass

# Z rozszerzenia HTTPAuath 
@auth.error_handler
def unathorized():
    response = jsonify({'status': 401, 'error': 'unahtorized', 
                        'message': 'Wykonaj uwierzytelnianie'})
    response.status_code = 401
    return response


@app.route('/devices/', methods=['GET'])
def get_devices():
    return jsonify({'device': [device.get_url() 
                               for device in Device.query.all()]})

@app.route('/devices/<int:id>', methods=['GET'])
def get_device(id):
    return jsonify(Device.query.get_or_404(id).export_data())


@app.route('/devices/<int:id>/version', methods=['GET'])
@background
def get_device_version(id):
    device = Device.query.get_or_404(id)
    hostname = device.hostname
    ip = device.mgmt_ip
    prompt = hostname+"#"
    result = show_version(hostname, prompt, ip, 'cisco', 'cisco')
    return jsonify({"version": str(result)})

@app.route('/devices/<device_role>/version', methods=['GET'])
@background
def get_role_version(device_role):
    device_id_list = [device.id for device in Device.query.all() if device.role == device_role]
    result = {}
    for id in device_id_list:
        device = Device.query.get_or_404(id)
        hostname = device.hostname
        ip = device.mgmt_ip
        prompt = hostname + "#"
        device_result = show_version(hostname, prompt, ip, 'cisco', 'cisco')
        result[hostname] = str(device_result)
    return jsonify(result)

@app.route('/devices/', methods=['POST'])
def new_device():
    device = Device()
    device.import_data(request.json)
    db.session.add(device)
    db.session.commit()
    return jsonify({}), 201, {'Location': device.get_url()}

@app.route('/devices/<int:id>', methods=['PUT'])
def edit_device(id):
    device = Device.query.get_or_404(id)
    device.import_data(request.json)
    db.session.add(device)
    db.session.commit()
    return jsonify({})


@app.route('/status/<id>', methods=['GET'])
def get_task_status(id):
    """Query the status of an asynchronous task."""
    # Pobieramy zadanie i je weryfikujemy 
    global background_tasks
    rv = background_tasks.get(id)
    if rv is None:
        return not_found(None)

    # Jeśli obiekt zadania jest obiektem Thread to oznacza to, że zadanie wciąż 
    # działa. W takim przypadku ponownie zwracamy odpowiedź z kodem statusu 202
    if isinstance(rv, Thread):
        return jsonify({}), 202, {'Location': url_for('get_task_status', id=id)}

    # Jeśli obiekt zadania nie jest obiektem Thread to zakładamy, że stanowi on
    # odpowiedź zakończonego zadania, zatem odpowiedź ta zostaje zwrócona.
    # Jeśli aplikacja została skonfigurowana do automatycznego usuwania zasobów
    # odpowiedzi zadań po zakończeniu tych zadań, to operacja usunięcia jest 
    # wykonywana teraz; w przeciwnym razie oczekuje się, że to klient prześle
    # żądanie usunięcia zasobu.
    if app.config['AUTO_DELETE_BG_TASKS']:
        del background_tasks[id]
    return rv


if __name__ == '__main__':
    db.create_all()
    app.run(host='0.0.0.0', debug=True)
