import traceback
import numpy
import numpy as np
from sklearn.base import BaseEstimator
import sklearn.metrics
import scipy.stats

class WQDA(BaseEstimator):
	"""
	Weighted Quadratic Discriminant Analysis (QDA)

	A classifier with a quadratic decision boundary, which allows 
	weighted samples, and which is generated by fitting class 
	conditional densities to the data and using Bayes' rule.

	The model fits a Gaussian density to each class.

	Attributes
	----------
	covariances_ : list of array-like, shape = [n_features, n_features]
		Covariance matrices of each class.

	means_ : array-like, shape = [n_classes, n_features]
		Class means.

	priors_ : array-like, shape = [n_classes]
		Class priors (sum to 1).
	"""
	
	def __init__(self): # LDA
		self.use_shrinkage = False

	#TODO regularization
	def fit(self, X, y, sample_weight=[]):
		"""
		Fit the QDA model according to the given training data and parameters.

		Parameters
		----------
		X : array-like, shape = [n_samples, n_features]
			Training vector, where n_samples in the number of samples and
			n_features is the number of features.

		y : array, shape = [n_samples]
			Target values (integers)
			
		sample_weight : array-like, shape (n_samples,), optional
			Weights applied to individual samples.
			If not provided, uniform weights are assumed.
			
		Returns
		-------
		self : returns an instance of self.
		"""
		
		K = len(set(y))
		if X.shape[0] < X.shape[1]: # less instances than dimensions -> use shrinkage
			self.use_shrinkage = True
		
		kindices = [numpy.where(y==k)[0] for k in range(K)]
		if len(sample_weight) == 0:
			qsum = numpy.bincount(y.astype(int))
			sample_weight = numpy.ones(len(y))
		else:
			qsum = numpy.array([numpy.sum(sample_weight[kindices[k]]) for k in range(K)])
		sample_weight = numpy.reshape(sample_weight, (len(sample_weight),1))
		self.priors_ = qsum / float(len(y))
		self.means_ = []
		self.covariances_ = []
		self.covariance_ = []
		for k in range(K):
			self.means_.append(numpy.average(X[kindices[k], :], axis=0, weights=sample_weight[kindices[k]][:,0]))
			##QDA
			try:
				xm = X[kindices[k], :] - self.means_[k]
				if X.shape[0] > X.shape[1]: # more instances than features
					##normalizing by number of data points (Loog 2015)
					self.covariances_.append(1./(len(kindices[k])) * numpy.multiply(xm, sample_weight[kindices[k]]).T.dot(xm))
					##weighted unbiased sample covariance (from http://stats.stackexchange.com/questions/61225/correct-equation-for-weighted-unbiased-sample-covariance )
					#self.covariances_.append(1./(sample_weight[kindices[k], :].sum()) * numpy.multiply(xm, sample_weight[kindices[k]]).T.dot(xm))
				else: # less instances than features - use shrinkage
					self.covariances_.append(weighted_oas(xm, sample_weight[kindices[k]]))

			except:
				traceback.print_exc()
			
			if len(self.covariance_) == 0:
				self.covariance_ = numpy.copy(self.covariances_[k])
			else:
				self.covariance_ += self.covariances_[k]
		self.covariance_ /= float(K)
		
		return self
				
	def _log_posterior(self, X, normalize = True):
		# https://github.com/probml/pmtk3/blob/5fefd068a2e84ae508684d3e4750bd72a4164ba0/toolbox/SupervisedModels/discrimAnalysis/discrimAnalysisPredict.m
		# Apply Bayes rule with Gaussian class-conditional densities.
		# post[i,c] = P(y=c|x(i,:), params)
		# yhat[i] = arg max_c post[i,c]
		N = X.shape[0]
		Nclasses = len(self.priors_)
		loglik = numpy.zeros((N, Nclasses))
		for c in range(Nclasses):
			try: 
				mvnorm = scipy.stats.multivariate_normal(self.means_[c], self.covariances_[c], allow_singular=True)
			except: 
				mvnorm = scipy.stats.multivariate_normal(self.means_[c], self.covariances_[c])
			loglik[:, c] = mvnorm.logpdf(X)
		logjoint = numpy.log(self.priors_) + loglik

		if normalize:
			normalization = scipy.misc.logsumexp(logjoint, axis=1)
			return logjoint - numpy.reshape(normalization, (len(normalization), 1))
		else:
			return logjoint
				
	def _posterior(self, X):
		return numpy.exp(self._log_posterior(X))
	
	def predict(self, X):
		"""Perform classification on samples in X.

		Parameters
		----------
		X : array-like, shape = [n_samples, n_features]

		Returns
		-------
		y_pred : array, shape = [n_samples]
			Class labels for samples in X.
		"""
		return numpy.argmax(self._posterior(X), axis=1)
	
	def predict_proba(self, X):
		"""Return posterior probabilities of classification.

		Parameters
		----------
		X : array-like, shape = [n_samples, n_features]
			Array of samples/test vectors.

		Returns
		-------
		C : array, shape = [n_samples, n_classes]
			Posterior probabilities of classification per class.
		"""
		return self._posterior(X)
	
	def score(self, X, y, sample_weight=None):
		return sklearn.metrics.accuracy_score(y, self.predict(X), sample_weight=sample_weight)
	
def weighted_oas(X, weights):
	"""Estimate covariance with the Oracle Approximating Shrinkage algorithm.
	
	Parameters
	----------
	X : array-like, shape (n_samples, n_features)
	    Centered data from which to compute the covariance estimate.
	    
	weights: sample weights
	
	Returns
	-------
	shrunk_cov : array-like, shape (n_features, n_features)
	    Shrunk covariance.
	
	Notes
	-----
	The regularised (shrunk) covariance is:
	
	(1 - shrinkage)*cov
	  + shrinkage * mu * np.identity(n_features)
	
	where mu = trace(cov) / n_features
	
	The formula we used to implement the OAS
	does not correspond to the one given in the article. It has been taken
	from the MATLAB program available from the author's webpage
	(https://tbayes.eecs.umich.edu/yilun/covestimation).
	
	"""
	X = np.asarray(X)
	n_samples, n_features = X.shape
	
	#emp_cov = empirical_covariance(X, assume_centered=assume_centered)
	emp_cov = 1./(len(weights)) * numpy.multiply(X, weights).T.dot(X)
	mu = np.trace(emp_cov) / n_features
	
	# formula from Chen et al.'s **implementation**
	alpha = np.mean(emp_cov ** 2)
	num = alpha + mu ** 2
	den = (n_samples + 1.) * (alpha - (mu ** 2) / n_features)
	
	shrinkage = 1. if den == 0 else min(num / den, 1.)
	shrunk_cov = (1. - shrinkage) * emp_cov
	shrunk_cov.flat[::n_features + 1] += shrinkage * mu
	
	return shrunk_cov#, shrinkage
