//Rozdział 21.
//Obserwator

public class FallsIllEventArgs : EventArgs
{
   public string Address;
}

public class Person
{
   public void CatchACold()
    {
        FallsIll?.Invoke(this,
            new FallsIllEventArgs { Address = "Mazowiecka 123" });
    }
   public event EventHandler<FallsIllEventArgs> FallsIll;
}

static void Main()
{
    var person = new Person();
    person.FallsIll += CallDoctor;
    person.CatchACold();
}
private static void CallDoctor(object sender, FallsIllEventArgs
eventArgs)
{
    Console.WriteLine($"Wezwano doktora pod adres {eventArgs.Address}");
}

//Słabe zdarzenie
public class Button
{
   public event EventHandler Clicked;
   public void Fire()
    {
        Clicked?.Invoke(this, EventArgs.Empty);
    }
}

public class Window
{
   public Window(Button button)
    {
        button.Clicked += ButtonOnClicked;
    }
    private void ButtonOnClicked(object sender,
    EventArgs eventArgs)
    {
        WriteLine("Klinięto przycisk (handler w obiekcie Window)");
    }
    ~Window()
    {
        WriteLine("Finalizacja obiektu Window");
    }
}
var btn = new Button();
var window = new Window(btn);
var windowRef = new WeakReference(window);
btn.Fire();

window = null;

FireGC();
WriteLine($"Czy okno jest nadal żywe po wywołaniu GC? {windowRef.IsAlive}");
// True
public class Window2
{
   public Window2(Button button)
    {
        WeakEventManager<Button, EventArgs>
            .AddHandler(button, "Kliknięto", ButtonOnClicked);
    }
    //pozostała część klasy taka sama jak poprzednio
}

//Strumienie zdarzeń

private class Subscription : IDisposable
{
    private Person person;
    public IObserver<Event> Observer;
    public Subscription(Person person, IObserver<Event> observer)
    {
        this.person = person;
        Observer = observer;
    }
    public void Dispose()
    {
        person.subscriptions.Remove(this);
    }
}

public class Event
{
    // tutaj może być cokolwiek
}
public class FallsIllEvent : Event
{
    public string Address;
}

public class Person : IObservable<Event>
{
    private readonly HashSet<Subscription> subscriptions
    = new HashSet<Subscription>();
    public IDisposable Subscribe(IObserver<Event> observer)
    {
        var subscription = new Subscription(this, observer);
        subscriptions.Add(subscription);
        return subscription;
    }
    public void CatchACold()
    {
        foreach (var sub in subscriptions)
            sub.Observer.OnNext(new FallsIllEvent { Address = "Ul. Warszawska 123" });
    }
    private class Subscription : IDisposable { ... }
}

public class Demo : IObserver<Event>
{
    static void Main(string[] args)
    {
        new Demo();
    }
    public Demo()
    {
        var person = new Person();
        var sub = person.Subscribe(this);
    }
    public void OnNext(Event value)
    {
        if (value is FallsIllEvent args)
            WriteLine($"Wezwano lekarza pod adres {args.Address}");
    }
    public void OnError(Exception error) { }
    public void OnCompleted() { }
}

person
    .OfType<FallsIllEvent>()
    .Subscribe(args =>
       WriteLine($"Wezwano lekarza pod adres {args.Address}"));



//Obserwatory właściwości
class Person
{
   public int Age { get; set; }
}

class Person
{
    private int age;
   public int Age
    {
        get => age;
        set
        {
            //todo: tutaj powinno się coś znaleźć
            age = value;
        }
    }
}
public interface INotifyPropertyChanged
{
    /// <summary>Występuje, gdy zmieni się wartość właściwości. </summary>
    event PropertyChangedEventHandler PropertyChanged;
}

public class Person : INotifyPropertyChanged
{
    private int age;
    public int Age
    {
        get => age;
        set
        {
            if (value == age) return;
            age = value;
            OnPropertyChanged();
        }
    }
    public event PropertyChangedEventHandler PropertyChanged;
    [NotifyPropertyChangedInvocator]
    protected virtual void OnPropertyChanged(
    [CallerMemberName] string propertyName = null)
    {
        PropertyChanged?.Invoke(this,
        new PropertyChangedEventArgs(propertyName));
    }
}

//Dwukierunkowe powiązania
var product = new Product { Name = "Book" };
var window = new Window { ProductName = "Book" };
product.PropertyChanged += (sender, eventArgs) =>
{
    if (eventArgs.PropertyName == "Name")
    {
        Console.WriteLine("Zmieniła się właściwość Name w klasie Product");
        window.ProductName = product.Name;
    }
};
window.PropertyChanged += (sender, eventArgs) =>
{
    if (eventArgs.PropertyName == "ProductName")
    {
        Console.WriteLine("Zmieniła się właściwość Name w klasie Window");
        product.Name = window.ProductName;
    }
};

public sealed class BidirectionalBinding : IDisposable
{
    private bool disposed;
    public BidirectionalBinding(
    INotifyPropertyChanged first, Expression<Func<object>> firstProperty,
    INotifyPropertyChanged second, Expression<Func<object>> secondProperty)
    {
        if (firstProperty.Body is MemberExpression firstExpr
        && secondProperty.Body is MemberExpression secondExpr)
        {
            if (firstExpr.Member is PropertyInfo firstProp
            && secondExpr.Member is PropertyInfo secondProp)
            {
                first.PropertyChanged += (sender, args) =>
                {
                    if (!disposed)
                    {
                        secondProp.SetValue(second, firstProp.GetValue(first));
                    }
                };
                second.PropertyChanged += (sender, args) =>
                {
                    if (!disposed)
                    {
                        firstProp.SetValue(first, secondProp.GetValue(second));
                    }
                };
            }
        }
    }
    public void Dispose()
    {
        disposed = true;
    }
}



public event PropertyChangedEventHandler PropertyChanged;
[NotifyPropertyChangedInvocator]
protected virtual void OnPropertyChanged
    ([CallerMemberName] string propertyName = null)
{
    PropertyChanged?.Invoke(this,
        new PropertyChangedEventArgs(propertyName));
}
public int Age
{
    get => age;
    set
    {
        if (value == age) return;
        age = value;
        OnPropertyChanged();
    }
}

//Problemy z zależnościami

public class Person : PropertyNotificationSupport
{
    private int age;
    public int Age
    {
        get => age;
        set
        {
            if (value == age) return;
            age = value;
            OnPropertyChanged();
        }
    }
    public bool CanVote => Age <= 16;
}


public int Age
{
    get => age;
    set
    {
        if (value == age) return;
        age = value;
        OnPropertyChanged();
        if (oldCanVote != CanVote)
            OnPropertyChanged(nameof(CanVote));
    }
}


set
{
    if (value == age) return;
    var oldCanVote = CanVote;
    age = value;
    OnPropertyChanged();
    if (oldCanVote != CanVote)
        OnPropertyChanged(nameof(CanVote));
}



class PropertyNotificationSupport : INotifyPropertyChanged
{
    private readonly Dictionary<string, HashSet<String>> affectedBy
        = new Dictionary<string, HashSet<string>>();
    public event PropertyChangedEventHandler PropertyChanged;
}
[NotifyPropertyChangedInvocator]
protected virtual void OnPropertyChanged
   ([CallerMemberName] string propertyName = null)
{
    PropertyChanged?.Invoke(this,
        new PropertyChangedEventArgs(propertyName));
    foreach (var affector in affectedBy.Keys)
        if (affectedBy[affector].Contains(propertyName))
            OnPropertyChanged(affected);
}
protected Func<T> property<T>(string name,
    Expression<Func<T>> expr)
{ ... }
private class MemberAccessVisitor : ExpressionVisitor { ... }
}

public class Person : PropertyNotificationSupport
{
    private readonly Func<bool> canVote;
    public bool CanVote => canVote();
    public Person()
    {
        canVote = property(nameof(CanVote),
        () => Citizen && Age >= 16);
    }
    // tutaj inne składowe
}

protected Func<T> property<T>(string name, Expression<Func<T>> expr)
{
    Console.WriteLine($"Tworzenie wyliczonej właściwości dla wyrażenia {expr}");
    var visitor = new MemberAccessVisitor(GetType());
    visitor.Visit(expr);
    if (visitor.PropertyNames.Any())
    {
        if (!affectedBy.ContainsKey(name))
            affectedBy.Add(name, new HashSet<string>());
        foreach (var propName in visitor.PropertyNames)
            if (propName != name) affectedBy[name].Add(propName);
    }
    return expr.Compile();
}

private class MemberAccessVisitor : ExpressionVisitor
{
    private readonly Type declaringType;
    public readonly IList<string> PropertyNames = new List<string>();
    public MemberAccessVisitor(Type declaringType)
    {
        this.declaringType = declaringType;
    }
    public override Expression Visit(Expression expr)
    {
        if (expr != null && expr.NodeType == ExpressionType.MemberAccess)
        {
            var memberExpr = (MemberExpression)expr;
            if (memberExpr.Member.DeclaringType == declaringType)
            {
                PropertyNames.Add(memberExpr.Member.Name);
            }
        }
        return base.Visit(expr);
    }
}

var p = new Person();
p.PropertyChanged += (sender, eventArgs) =>
{
    Console.WriteLine($"{eventArgs.PropertyName} zmieniła się");
};
p.Age = 16;
// Właściwość Age zmieniła się
// Właściwość CanVote zmieniła się
p.Citizen = true;
// Właściwość Citizen zmieniła się
// Właściwość CanVote zmieniła się

protected void setValue<T>(T value, ref T field,
 [CallerMemberName] string propertyName = null)
{
    if (value.Equals(field)) return;
    OnPropertyChanging(propertyName);
    field = value;
    OnPropertyChanged(propertyName);
}

public int Age
{
    get => age;
    set => setValue(value, ref age);
}

//Widoki

public class Person
{
    public string Name;
}

public class PersonView : View
{
    protected Person person;
    public PersonView(Person person)
    {
        this.person = person;
    }
    public string Name
    {
        get => person.Name;
        set
        {
            setValue(value, () => person.Name);
        }
    }
}

//Kolekcje obserwowalne


//Obserwowalne zapytania LINQ


//Subskrypcje deklaratywne w Autofac

public interface IEvent { }
public interface ISend<TEvent> where TEvent : IEvent
{
    event EventHandler<TEvent> Sender;
}
public interface IHandle<TEvent> where TEvent : IEvent
{
    void Handle(object sender, TEvent args);
}

public class ButtonPressedEvent : IEvent
{
    public int NumberOfClicks;
}

public class Button : ISend<ButtonPressedEvent>
{
    public event EventHandler<ButtonPressedEvent> Sender;
    public void Fire(int clicks)
    {
        Sender?.Invoke(this, new ButtonPressedEvent
        {
            NumberOfClicks = clicks
        });
    }
}

public class Logging : IHandle<ButtonPressedEvent>
{
    public void Handle(object sender, ButtonPressedEvent args)
    {
        Console.WriteLine(
            $"Naciśnięto przycisk {args.NumberOfClicks} razy");
    }
}

var cb = new ContainerBuilder();
var ass = Assembly.GetExecutingAssembly();
// rejestracja interfejsów publikujących
cb.RegisterAssemblyTypes(ass)
.AsClosedTypesOf(typeof(ISend<>))
.SingleInstance();
// rejestracja subskrybentów
cb.RegisterAssemblyTypes(ass)
.Where(t =>
t.GetInterfaces()
.Any(i =>
i.IsGenericType &&
i.GetGenericTypeDefinition() == typeof(IHandle<>)))
.OnActivated(act =>
{
    var instanceType = act.Instance.GetType();
    var interfaces = instanceType.GetInterfaces();
    foreach (var i in interfaces)
    {
        if (i.IsGenericType
        && i.GetGenericTypeDefinition() == typeof(IHandle<>))
        {
            var arg0 = i.GetGenericArguments()[0];
            var senderType = typeof(ISend<>).MakeGenericType(arg0);
            var allSenderTypes =
            typeof(IEnumerable<>).MakeGenericType(senderType);
            var allServices = act.Context.Resolve(allSenderTypes);
            foreach (var service in (IEnumerable)allServices)
            {
                var eventInfo = service.GetType().GetEvent("Sender");
                var handleMethod = instanceType.GetMethod("Handle");
                var handler = Delegate.CreateDelegate(
                eventInfo.EventHandlerType, null, handleMethod);
                eventInfo.AddEventHandler(service, handler);
            }
        }
    }
})
.SingleInstance()
.AsSelf();

var container = cb.Build();
var button = container.Resolve<Button>();
var logging = container.Resolve<Logging>();
button.Fire(1); // przycisk został kliknięty 1 raz
button.Fire(2); // przycisk został kliknięty 2 razy

