#if _MSC_VER
#pragma warning(disable : 4996)
#endif

#include <algorithm>
#include <cctype>
#include <gtest/gtest.h>
#include <initializer_list>
#include <locale>
#include <vector>

namespace {

//
// Reguła pięciu
class Buffer {
public:
  // Konstruktor
  Buffer(const std::initializer_list<float>& values) : size_{values.size()} {
    ptr_ = new float[values.size()];
    std::copy(values.begin(), values.end(), ptr_);
  }

  // 1. Konstruktor kopiujący.
  Buffer(const Buffer& other) : size_{other.size_} {
    ptr_ = new float[size_];
    std::copy(other.ptr_, other.ptr_ + size_, ptr_);
  }

  // 2. Kopiujący operator przypisania.
  auto& operator=(const Buffer& other) {
    delete[] ptr_;
    ptr_ = new float[other.size_];
    size_ = other.size_;
    std::copy(other.ptr_, other.ptr_ + size_, ptr_);
    return *this;
  }

  // 3. Destruktor.
  ~Buffer() {
    delete[] ptr_; // Prawidłowe: usuwanie wskaźnika nullptr jest poprawne.
    ptr_ = nullptr;
  }

  // 4. Konstruktor przenoszący.
  Buffer(Buffer&& other) noexcept : size_{other.size_}, ptr_{other.ptr_} {
    other.ptr_ = nullptr;
    other.size_ = 0;
  }
  // 5. Przypisanie przez przenoszenie.
  auto& operator=(Buffer&& other) noexcept {
    ptr_ = other.ptr_;
    size_ = other.size_;
    other.ptr_ = nullptr;
    other.size_ = 0;
    return *this;
  }

  // Iteratory służące do dostępu do danych
  auto begin() const { return ptr_; }

  auto end() const { return ptr_ + size_; }

private:
  size_t size_{0};
  float* ptr_{nullptr};
};

TEST(MoveSemantics, RuleOfFive) {
  // Konstruktor
  auto b0 = Buffer({0.0f, 0.5f, 1.0f, 1.5f});
  // 1. Konstruktor kopiujący
  auto b1 = b0;
  // 2. Przypisanie z kopiowaniem, ponieważ b0 jest już zainicjowana
  b0 = b1;
  // 3. Po zakończeniu funkcji automatycznie wywoływane są destruktory
}

//
// Zmienne nazwane i r-wartości
// Reguła zera
class Button {
public:
  Button() {}

  // // Konstruktor kopiujący i kopiujący operator przypisania.
  // Button(const Button&) = default;
  // auto operator=(const Button&) -> Button& = default;

  // // Konstruktor przenoszący i przenoszący operator przypisania.
  // Button(Button&&) noexcept = default;
  // auto operator=(Button&&) noexcept -> Button& = default;

  // // Destruktor
  // ~Button() = default;

  auto set_title(const std::string& s) { title_ = s; }
  auto set_title(std::string&& s) { title_ = std::move(s); }
  std::string title_{};
};

auto get_ok() { return std::string("OK"); }

TEST(MoveSemantics, NamedVariablesAndRvalues) {
  auto button = Button{};
  // Przypadek 1:
  {
    auto str = std::string{"OK"};
    button.set_title(str); // Przypisanie z kopiowaniem
  }
  // Przypadek 2:
  {
    auto str = std::string{"OK"};
    button.set_title(std::move(str)); // Przypisanie z przenoszeniem
  }
  // Przypadek 3:
  {
    button.set_title(get_ok()); // Przypisanie z przenoszeniem
  }
  // Przypadek 4:
  {
    auto str = get_ok();
    button.set_title(str); // Przypisanie z kopiowaniem
  }
  // Przypadek 5:
  {
    const auto str = get_ok();
    button.set_title(std::move(str)); // Przypisanie z kopiowaniem
  }
}

//
// Często popełniany błąd — przenoszenie obiektów bez zasobów
class Menu {
public:
  Menu(const std::initializer_list<std::string>& items) : items_{items} {}
  void select(int i) { index_ = i; }
  auto selected_item() const { return index_ != -1 ? items_[index_] : ""; }
  // ...
private:
  int index_{-1}; // Wybrany indeks
  std::vector<std::string> items_;
};

TEST(MoveSemantics, MovingNonResources) {
  auto a = Menu{"New", "Open", "Close", "Save"};
  a.select(2);
  ASSERT_TRUE(a.selected_item() == "Close");
  auto b = std::move(a);
  std::cout << "after move\n";
  // Obiekt a został przeniesiony, ale a.index_ to nadal 2.
  // Wywołanie a.selected_item() zapewne doprowadzi do błędu,
  // ponieważ a.index_ to 2, a wartość a.items_ została przeniesiona
  // auto selected = a.selected_item(); // Błąd
}

//
// Poprawne przyjmowanie argumentów
auto str_to_lower(std::string str) {
  for (auto& c : str)
    c = std::tolower(c);
  return str;
}

class Widget {
public:
  Widget(std::string s)     // Przez wartość
      : s_{std::move(s)} {} // Konstruktor przenoszący
private:
  std::string s_;
  /* … */
};

TEST(MoveSemantics, StrToLower) {
  std::cout << "tolower\n";
  auto upper = std::string{"ABC"};
  auto lower = str_to_lower(std::move(upper));
  ASSERT_TRUE(upper.empty());
  ASSERT_FALSE(lower.empty());
}

TEST(MoveSemantics, InitializingClassMembers) {
  const auto w1 = Widget{"ABC"};
  const auto s = std::string{"ABC"};
  const auto w2 = Widget{s};
}

} // namespace