/*
 * Copyright (c) 2014, Oracle America, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of Oracle nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

package net.sdo;

import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicLong;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.infra.Blackhole;

@State(Scope.Benchmark)
@BenchmarkMode(Mode.AverageTime)
public class UncontendedTest {
    @Param({"1"})
    private int nLoops;

    private static interface Getter {
        public void getAndIncrement(Blackhole bh);
    }

    private static class AtomicGetter implements Getter {
        private AtomicLong al = new AtomicLong();
	public void getAndIncrement(Blackhole bh) {
	    bh.consume(al.getAndIncrement());
	}
    }

    private static class SyncGetter implements Getter {
        private long l;
	public synchronized void getAndIncrement(Blackhole bh) {
	    bh.consume(l++);
	}
    }

    private static class UnsyncGetter implements Getter {
        private long l;
	public void getAndIncrement(Blackhole bh) {
	    bh.consume(l++);
	}
    }

    AtomicGetter ag;
    SyncGetter sg;
    UnsyncGetter ug;

    @Setup
    public void setup() {
        ag = new AtomicGetter();
        sg = new SyncGetter();
        ug = new UnsyncGetter();
    }

    @Benchmark
    public void testAtomic(Blackhole bh) {
        test(ag, bh);
    }

    @Benchmark
    public void testSync(Blackhole bh) {
        test(sg, bh);
    }

    @Benchmark
    public void testUnsync(Blackhole bh) {
        test(ug, bh);
    }

    public void test(Getter g, Blackhole bh) {
        for (int i = 0; i < nLoops; i++) {
	    g.getAndIncrement(bh);
	}
    }
}
