const express = require('express');
const { ApolloServer, gql } = require('apollo-server-express');
require('dotenv').config();

const db = require('./db');
const models = require('./models');

// Uruchomienie serwera nasłuchującego na porcie wskazanym w pliku .env, czyli na porcie 4000.
const port = process.env.PORT || 4000;
const DB_HOST = process.env.DB_HOST;

// Utworzenie schematu za pomocą języka schematu GraphQL.
const typeDefs = gql`
  type Note {
    id: ID
    content: String
    author: String
  }

  type Query {
    hello: String
    notes: [Note]
    note(id: ID): Note
  }

  type Mutation {
    newNote(content: String!): Note
  }
`;

// Dostarczenie funkcji resolvera dla właściwości schematu.
const resolvers = {
  Query: {
    hello: () => 'Witaj, świecie!',
    notes: async () => {
      return await models.Note.find();
    },
    note: async (parent, args) => {
      return await models.Note.findById(args.id);
    }
  },
  Mutation: {
    newNote: async (parent, args) => {
      return await models.Note.create({
        content: args.content,
        author: 'Adam Scott'
      });
    }
  }
};

const app = express();

db.connect(DB_HOST);

// Konfiguracja serwera Apollo.
const server = new ApolloServer({ typeDefs, resolvers });

// Zastosowanie oprogramowania pośredniczącego Apollo GraphQL i zdefiniowanie ścieżki dostępu do /api.
server.applyMiddleware({ app, path: '/api' });

app.listen({ port }, () =>
  console.log(
    `Serwer GraphQL działa pod adresem http://localhost:${port}${server.graphqlPath}`
  )
);
