const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');
const {
  AuthenticationError,
  ForbiddenError
} = require('apollo-server-express');
require('dotenv').config();

const gravatar = require('../util/gravatar');

module.exports = {
  newNote: async (parent, args, { models }) => {
    return await models.Note.create({
      content: args.content,
      author: 'Adam Scott'
    });
  },
  deleteNote: async (parent, { id }, { models }) => {
    try {
      await models.Note.findOneAndRemove({ _id: id });
      return true;
    } catch (err) {
      return false;
    }
  },
  updateNote: async (parent, { content, id }, { models }) => {
    try {
      return await models.Note.findOneAndUpdate(
        {
          _id: id
        },
        {
          $set: {
            content
          }
        },
        {
          new: true
        }
      );
    } catch (err) {
      throw new Error('Błąd podczas uaktualniania notatki.');
    }
  },
  signUp: async (parent, { username, email, password }, { models }) => {
    // Normalizacja adresu e-mail.
    email = email.trim().toLowerCase();
    // Utworzenie wartości hash na podstawie hasła.
    const hashed = await bcrypt.hash(password, 10);
    // Utworzenie adresu URL gravatar.
    const avatar = gravatar(email);
    try {
      const user = await models.User.create({
        username,
        email,
        avatar,
        password: hashed
      });

      // Utworzenie i zwrot tokena JSON Web.
      return jwt.sign({ id: user._id }, process.env.JWT_SECRET);
    } catch (err) {
      // W przypadku wystąpienia problemu podczas tworzenia konta, należy zgłosić błąd.
      throw new Error('Błąd podczas tworzenia konta.');
    }
  },
  signIn: async (parent, { username, email, password }, { models }) => {
    if (email) {
      // Normalizacja adresu e-mail.
      email = email.trim().toLowerCase();
    }

    const user = await models.User.findOne({
      $or: [{ email }, { username }]
    });

    // Jeżeli użytkownik nie zostanie znaleziony, należy zgłosić błąd uwierzytelniania.
    if (!user) {
      throw new AuthenticationError('Błąd podczas uwierzytelniania.');
    }

    // Jeżeli hasła są różne, należy zgłosić błąd uwierzytelniania.
    const valid = await bcrypt.compare(password, user.password);
    if (!valid) {
      throw new AuthenticationError('Błąd podczas uwierzytelniania.');
    }

    // Utworzenie i zwrot tokena JSON Web.
    return jwt.sign({ id: user._id }, process.env.JWT_SECRET);
  }
};
