﻿//***************************************************************************************
// Effects.h Frank Luna (C) 2011 Wszelkie prawa zastrzeżone.
//
// Definiuje lekkie opakowania efektów pozwalające zgrupować efekt i jego zmienne.
// Oprócz tego definiuje statyczną klasę Effect, z poziomu której możemy uzyskać dostęp do wszystkich naszych efektów.
//***************************************************************************************

#ifndef EFFECTS_H
#define EFFECTS_H

#include "d3dUtil.h"

#pragma region Effect
class Effect
{
public:
	Effect(ID3D11Device* device, const std::wstring& filename);
	virtual ~Effect();

private:
	Effect(const Effect& rhs);
	Effect& operator=(const Effect& rhs);

protected:
	ID3DX11Effect* mFX;
};
#pragma endregion

#pragma region BasicEffect
class BasicEffect : public Effect
{
public:
	BasicEffect(ID3D11Device* device, const std::wstring& filename);
	~BasicEffect();

	void SetWorldViewProj(CXMMATRIX M)                  { WorldViewProj->SetMatrix(reinterpret_cast<const float*>(&M)); }
	void SetWorld(CXMMATRIX M)                          { World->SetMatrix(reinterpret_cast<const float*>(&M)); }
	void SetWorldInvTranspose(CXMMATRIX M)              { WorldInvTranspose->SetMatrix(reinterpret_cast<const float*>(&M)); }
	void SetTexTransform(CXMMATRIX M)                   { TexTransform->SetMatrix(reinterpret_cast<const float*>(&M)); }
	void SetEyePosW(const XMFLOAT3& v)                  { EyePosW->SetRawValue(&v, 0, sizeof(XMFLOAT3)); }
	void SetFogColor(const FXMVECTOR v)                 { FogColor->SetFloatVector(reinterpret_cast<const float*>(&v)); }
	void SetFogStart(float f)                           { FogStart->SetFloat(f); }
	void SetFogRange(float f)                           { FogRange->SetFloat(f); }
	void SetDirLights(const DirectionalLight* lights)   { DirLights->SetRawValue(lights, 0, 3*sizeof(DirectionalLight)); }
	void SetMaterial(const Material& mat)               { Mat->SetRawValue(&mat, 0, sizeof(Material)); }
	void SetDiffuseMap(ID3D11ShaderResourceView* tex)   { DiffuseMap->SetResource(tex); }

	ID3DX11EffectTechnique* Light1Tech;
	ID3DX11EffectTechnique* Light2Tech;
	ID3DX11EffectTechnique* Light3Tech;

	ID3DX11EffectTechnique* Light0TexTech;
	ID3DX11EffectTechnique* Light1TexTech;
	ID3DX11EffectTechnique* Light2TexTech;
	ID3DX11EffectTechnique* Light3TexTech;

	ID3DX11EffectTechnique* Light0TexAlphaClipTech;
	ID3DX11EffectTechnique* Light1TexAlphaClipTech;
	ID3DX11EffectTechnique* Light2TexAlphaClipTech;
	ID3DX11EffectTechnique* Light3TexAlphaClipTech;

	ID3DX11EffectTechnique* Light1FogTech;
	ID3DX11EffectTechnique* Light2FogTech;
	ID3DX11EffectTechnique* Light3FogTech;

	ID3DX11EffectTechnique* Light0TexFogTech;
	ID3DX11EffectTechnique* Light1TexFogTech;
	ID3DX11EffectTechnique* Light2TexFogTech;
	ID3DX11EffectTechnique* Light3TexFogTech;

	ID3DX11EffectTechnique* Light0TexAlphaClipFogTech;
	ID3DX11EffectTechnique* Light1TexAlphaClipFogTech;
	ID3DX11EffectTechnique* Light2TexAlphaClipFogTech;
	ID3DX11EffectTechnique* Light3TexAlphaClipFogTech;

	ID3DX11EffectMatrixVariable* WorldViewProj;
	ID3DX11EffectMatrixVariable* World;
	ID3DX11EffectMatrixVariable* WorldInvTranspose;
	ID3DX11EffectMatrixVariable* TexTransform;
	ID3DX11EffectVectorVariable* EyePosW;
	ID3DX11EffectVectorVariable* FogColor;
	ID3DX11EffectScalarVariable* FogStart;
	ID3DX11EffectScalarVariable* FogRange;
	ID3DX11EffectVariable* DirLights;
	ID3DX11EffectVariable* Mat;

	ID3DX11EffectShaderResourceVariable* DiffuseMap;
};
#pragma endregion

#pragma region WaveRenderEffect
class WaveRenderEffect : public Effect
{
public:
	WaveRenderEffect(ID3D11Device* device, const std::wstring& filename);
	~WaveRenderEffect();

	void SetWorldViewProj(CXMMATRIX M)                     { WorldViewProj->SetMatrix(reinterpret_cast<const float*>(&M)); }
	void SetWorld(CXMMATRIX M)                             { World->SetMatrix(reinterpret_cast<const float*>(&M)); }
	void SetWorldInvTranspose(CXMMATRIX M)                 { WorldInvTranspose->SetMatrix(reinterpret_cast<const float*>(&M)); }
	void SetTexTransform(CXMMATRIX M)                      { TexTransform->SetMatrix(reinterpret_cast<const float*>(&M)); }
	void SetEyePosW(const XMFLOAT3& v)                     { EyePosW->SetRawValue(&v, 0, sizeof(XMFLOAT3)); }
	void SetFogColor(const FXMVECTOR v)                    { FogColor->SetFloatVector(reinterpret_cast<const float*>(&v)); }
	void SetFogStart(float f)                              { FogStart->SetFloat(f); }
	void SetFogRange(float f)                              { FogRange->SetFloat(f); }
	void SetGridSpatialStep(float f)                       { GridSpatialStep->SetFloat(f); }
	void SetDisplacementTexelSize(const XMFLOAT2& v)       { DisplacementMapTexelSize->SetRawValue(&v, 0, sizeof(XMFLOAT2)); }
	void SetDirLights(const DirectionalLight* lights)      { DirLights->SetRawValue(lights, 0, 3*sizeof(DirectionalLight)); }
	void SetMaterial(const Material& mat)                  { Mat->SetRawValue(&mat, 0, sizeof(Material)); }
	void SetDiffuseMap(ID3D11ShaderResourceView* tex)      { DiffuseMap->SetResource(tex); }
	void SetDisplacementMap(ID3D11ShaderResourceView* tex) { DisplacementMap->SetResource(tex); }

	ID3DX11EffectTechnique* Light3Tech;
	ID3DX11EffectTechnique* Light3TexTech;
	ID3DX11EffectTechnique* Light3TexFogTech;

	ID3DX11EffectMatrixVariable* WorldViewProj;
	ID3DX11EffectMatrixVariable* World;
	ID3DX11EffectMatrixVariable* WorldInvTranspose;
	ID3DX11EffectMatrixVariable* TexTransform;
	ID3DX11EffectVectorVariable* EyePosW;
	ID3DX11EffectVectorVariable* FogColor;
	ID3DX11EffectScalarVariable* FogStart;
	ID3DX11EffectScalarVariable* FogRange;
	ID3DX11EffectScalarVariable* GridSpatialStep;
	ID3DX11EffectVectorVariable* DisplacementMapTexelSize; 

	ID3DX11EffectVariable* DirLights;
	ID3DX11EffectVariable* Mat;

	ID3DX11EffectShaderResourceVariable* DiffuseMap;
	ID3DX11EffectShaderResourceVariable* DisplacementMap;
};
#pragma endregion

#pragma region BlurEffect
class BlurEffect : public Effect
{
public:
	BlurEffect(ID3D11Device* device, const std::wstring& filename);
	~BlurEffect();

	void SetWeights(const float weights[9])           { Weights->SetFloatArray(weights, 0, 9); }
	void SetInputMap(ID3D11ShaderResourceView* tex)   { InputMap->SetResource(tex); }
	void SetOutputMap(ID3D11UnorderedAccessView* tex) { OutputMap->SetUnorderedAccessView(tex); }

	ID3DX11EffectTechnique* HorzBlurTech;
	ID3DX11EffectTechnique* VertBlurTech;

	ID3DX11EffectScalarVariable* Weights;
	ID3DX11EffectShaderResourceVariable* InputMap;
	ID3DX11EffectUnorderedAccessViewVariable* OutputMap;
};
#pragma endregion

#pragma region WaveSimEffect
class WaveSimEffect : public Effect
{
public:
	WaveSimEffect(ID3D11Device* device, const std::wstring& filename);
	~WaveSimEffect();

	void SetWaveConstants(float s[3])               { WaveConstants->SetFloatArray(s, 0, 3); }
	void SetDisturbMag(float s)						{ DisturbMag->SetFloat(s); }
	void SetDisturbIndex(int row, int col)          
	{ 
		int v[4] = {col, row, -1, -1};
		DisturbIndex->SetIntVector(v); 
	}

	void SetPrevSolInput(ID3D11ShaderResourceView* srv)   { PrevSolInput->SetResource(srv); }
	void SetCurrSolInput(ID3D11ShaderResourceView* srv)   { CurrSolInput->SetResource(srv); }
	void SetCurrSolOutput(ID3D11UnorderedAccessView* uav) { CurrSolOutput->SetUnorderedAccessView(uav); }
	void SetNextSolOutput(ID3D11UnorderedAccessView* uav) { NextSolOutput->SetUnorderedAccessView(uav); }

	ID3DX11EffectTechnique* UpdateWavesTech;
	ID3DX11EffectTechnique* DisturbWavesTech;

	ID3DX11EffectScalarVariable* WaveConstants;
	ID3DX11EffectScalarVariable* DisturbMag;
	ID3DX11EffectVectorVariable* DisturbIndex;

	ID3DX11EffectShaderResourceVariable* PrevSolInput;
	ID3DX11EffectShaderResourceVariable* CurrSolInput;
	ID3DX11EffectUnorderedAccessViewVariable* CurrSolOutput;
	ID3DX11EffectUnorderedAccessViewVariable* NextSolOutput;
};
#pragma endregion

#pragma region Effects
class Effects
{
public:
	static void InitAll(ID3D11Device* device);
	static void DestroyAll();

	static BasicEffect* BasicFX;
	static BlurEffect* BlurFX;
	static WaveSimEffect* WaveSimFX;
	static WaveRenderEffect* WaveRenderFX;
};
#pragma endregion

#endif // EFFECTS_H