﻿using System;
using System.Collections.Generic;

namespace GenericEnumeration
{
	public class Pilgrim : IComparable<Pilgrim>
	{
		private string name;
		public Pilgrim(string name)
		{
			this.name = name;
		}
		public override string ToString()
		{
			return this.name;
		}

		// implementacja interfejsu IComparable
		public int CompareTo(Pilgrim rhs)
		{
			return this.name.CompareTo(rhs.name);
		}
		public bool Equals(Pilgrim rhs)
		{
			return this.name == rhs.name;
		}
	}


	// węzeł musi implementować interfejs IComparable dla węzłów Node typu T
	// teraz dodatkowo implementuje interfejs IEnumerable do użytku w pętlach foreach
	public class Node<T> : IComparable<Node<T>>, IEnumerable<Node<T>> where T:IComparable<T>
	{
		// pola składowe
		private T data;
		private Node<T> next = null;
		private Node<T> prev = null;

		// konstrukto
		public Node(T data)
		{
			this.data = data;
		}

		// właściwości
		public T Data { get { return this.data; } }

		public Node<T> Next
		{
			get { return this.next; }
		}

		public int CompareTo(Node<T> rhs)
		{
			return data.CompareTo(rhs.data);

		}
		public bool Equals(Node<T> rhs)
		{
			return this.data.Equals(rhs.data);
		}




		// metody
		public Node<T> Add(Node<T> newNode)
		{
			if (this.CompareTo(newNode) > 0) // wstawienie przed węzeł bieżący
			{
				newNode.next = this;  // wskaźnik next ustawiany na węzeł bieżąc

				// jeśli istnieje węzeł poprzedni, powinien od tego momentu
				// wskazywać polem next nowy węze
				if (this.prev != null)
				{
					this.prev.next = newNode;
					newNode.prev = this.prev;
				}

				// wskaźnik poprzednika węzła bieżącego ma wskazywać nowy węzeł
				this.prev = newNode;

				// zwrócenie referencji nowego węzła, jeśli stał się nowym czołem listy
				return newNode;
			}
			else			// wstawienie za węzeł bieżąc
			{
				// jeśli bieżący nie jest ostatnim, całą operację przejmuje następny
				if (this.next != null)
				{
					this.next.Add(newNode);
				}

				// brak następnego węzła — nowy węzeł trzeba skojarzyć z polem next bieżącego;
				// a w polu prev nowego wstawić referencję do bieżącego
				else
				{
					this.next = newNode;
					newNode.prev = this;
				}

				return this;
			}
		}

		public override string ToString()
		{
			string output = data.ToString();

			if (next != null)
			{
				output += ", " + next.ToString();
			}

			return output;
		}


		// Metody wymagane przez IEnumerable
		IEnumerator<Node<T>> IEnumerable<Node<T>>.GetEnumerator()
		{
			
			Node<T> nextNode = this;

			// przeglądanie wszystkich węzłów listy,  
			// zwracanie (yield) kolejnych węzłów
			do
			{
				Node<T> returnNode = nextNode;
				nextNode = nextNode.next;
				yield return returnNode;
			} while (nextNode != null);
		}

      System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
      {
         throw new NotImplementedException();
      }


	}		// koniec klasy


	// implementacja IEnumerable pozwalająca na stosowanie 
	// klasy LinkedList w pętlach foreach
	public class LinkedList<T> : IEnumerable<T> where T : IComparable<T>
	{
		// pola składowych
		private Node<T> headNode = null;


		// właściwości

		// indekser
		public T this[int index]
		{
			get
			{
				int ctr = 0;
				Node<T> node = headNode;

				while (node != null && ctr <= index)
				{
					if (ctr == index)
					{
						return node.Data;
					}
					else
					{
						node = node.Next;
					}

					++ctr;
				} // koniec while
				throw new ArgumentOutOfRangeException();
			}      // koniec get
		}          // koniec indeksera


		// konstruktor
		public LinkedList()
		{
		}

		// metody
		public void Add(T data)
		{
			if (headNode == null)
			{
				headNode = new Node<T>(data);
			}
			else
			{
				headNode = headNode.Add(new Node<T>(data));
			}
		}
		public override string ToString()
		{
			if (this.headNode != null)
			{
				return this.headNode.ToString();
			}
			else
			{
				return string.Empty;
			}
		}


		// Implementacja wymaganej metody IEnumerable
		// przeglądająca węzły (również implementujące ten interfejs)
		// i zwracająca (yield) dane zwrócone z węzła
		IEnumerator<T> IEnumerable<T>.GetEnumerator()
		{
			foreach (Node<T> node in this.headNode)
			{
				yield return node.Data;
			}
		}

      System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
      {
         throw new NotImplementedException();
      }

	}


	class Program
	{

		private static void DisplayList<T>(string intro, LinkedList<T> theList) where T : IComparable<T>
		{
			Console.WriteLine(intro + ": " + theList);
		}

		// punkt wejścia
		static void Main(string[] args)
		{
			LinkedList<Pilgrim> pilgrims = new LinkedList<Pilgrim>();
			pilgrims.Add(new Pilgrim("Rycerz"));
			pilgrims.Add(new Pilgrim("Młynarz"));
			pilgrims.Add(new Pilgrim("Szeryf"));
			pilgrims.Add(new Pilgrim("Kucharz"));
			pilgrims.Add(new Pilgrim("Adwokat"));

			DisplayList<Pilgrim>("Pilgrims", pilgrims);

			Console.WriteLine("Przeglądanie listy pielgrzymów...");

			// teraz lista daje się przeglądać, więc można ją
			// zastosować w pętli foreach
			foreach (Pilgrim p in pilgrims)
			{
				Console.WriteLine("Zawód pielgrzyma to " + p.ToString());
			}

		}
	}
}
