﻿Imports Microsoft.DirectX
Imports Microsoft.DirectX.Direct3D

Public Class TetrahedronForm
    ' The Direct3D device.
    Private m_Device As Device

    ' Data variables.
    Private Const NUM_TRIANGLES As Integer = 8
    Private Const NUM_POINTS As Integer = 3 * NUM_TRIANGLES

    ' The vertex buffer that holds drawing data.
    Private m_VertexBuffer As VertexBuffer = Nothing

    ' initialize the graphics device. Return True if successful.
    Public Function InitializeGraphics() As Boolean
        Try
            Dim params As New PresentParameters
            params.Windowed = True
            params.SwapEffect = SwapEffect.Discard
            params.EnableAutoDepthStencil = True            ' Depth stencil on.
            params.AutoDepthStencilFormat = DepthFormat.D16

            ' Use hardware vertex processing if supported.
            m_Device = New Device(0, DeviceType.Hardware, Me, _
                CreateFlags.SoftwareVertexProcessing, params)

            ' Turn on the Z-buffer and enable lighting.
            m_Device.RenderState.ZBufferEnable = True
            m_Device.RenderState.Lighting = False '@True

            ' Don't rely on culling to remove hidden surfaces.
            m_Device.RenderState.CullMode = Cull.None

            ' Create the vertex data.
            CreateVertexBuffer()

            Return True
        Catch ex As Exception
            MessageBox.Show("Error initializing Direct3D" & _
                vbCrLf & vbCrLf & ex.Message, _
                "Direct3D Error", MessageBoxButtons.OK)
            Return False
        End Try
    End Function

    Public Sub Render()
        If m_Device Is Nothing Then Exit Sub

        ' Clear the back buffer.
        m_Device.Clear(ClearFlags.Target Or ClearFlags.ZBuffer, Me.BackColor, 1, 0)

        ' Make a scene.
        m_Device.BeginScene()

        ' Setup the world, view, and projection matrices.
        SetupMatrices()

        ' Set the device's data stream source (the vertex buffer).
        m_Device.SetStreamSource(0, m_VertexBuffer, 0)

        ' Tell the device the format of the vertices.
        m_Device.VertexFormat = CustomVertex.PositionColored.Format

        ' Draw the primitives in the data stream.
        m_Device.DrawPrimitives(PrimitiveType.TriangleList, 0, NUM_TRIANGLES)

        ' End the scene and display.
        m_Device.EndScene()
        m_Device.Present()
    End Sub

    ' Create a vertex buffer for the device.
    Public Sub CreateVertexBuffer()
        ' Create a vertex buffer.
        m_VertexBuffer = New VertexBuffer( _
            GetType(CustomVertex.PositionColored), _
            NUM_POINTS, m_Device, 0, _
            CustomVertex.PositionColored.Format, _
            Pool.Default)

        ' Lock the vertex buffer. 
        ' Lock returns an array of PositionColored objects.
        Dim vertices As CustomVertex.PositionColored() = _
            CType(m_VertexBuffer.Lock(0, 0),  _
                CustomVertex.PositionColored())
        Dim start_index As Integer = 0

        Dim sqrt3 As Single = Math.Sqrt(3)
        Dim y As Single = 2 * Math.Sqrt(2 / 3)
        Dim offset As Single = y / 3
        MakeTriangle(vertices, start_index, Color.FromArgb(255, 0, 128, 0), _
          0, 0 - offset, sqrt3 - 1 / sqrt3, _
          0, y - offset, 0, _
          -1, 0 - offset, -1 / sqrt3)
        MakeTriangle(vertices, start_index, Color.FromArgb(255, 0, 108, 0), _
          0, 0 - offset, sqrt3 - 1 / sqrt3, _
          1, 0 - offset, -1 / sqrt3, _
          0, y - offset, 0)
        MakeTriangle(vertices, start_index, Color.FromArgb(255, 0, 88, 0), _
          1, 0 - offset, -1 / sqrt3, _
          -1, 0 - offset, -1 / sqrt3, _
          0, y - offset, 0)
        MakeTriangle(vertices, start_index, Color.DarkGreen, _
          0, 0 - offset, sqrt3 - 1 / sqrt3, _
          -1, 0 - offset, -1 / sqrt3, _
          1, 0 - offset, -1 / sqrt3)

        MakeTriangle(vertices, start_index, Color.FromArgb(255, 0, 0, 128), _
            0, -(0 - offset), sqrt3 - 1 / sqrt3, _
            0, -(y - offset), 0, _
            -1, -(0 - offset), -1 / sqrt3)
        MakeTriangle(vertices, start_index, Color.FromArgb(255, 0, 0, 108), _
          0, -(0 - offset), sqrt3 - 1 / sqrt3, _
          1, -(0 - offset), -1 / sqrt3, _
          0, -(y - offset), 0)
        MakeTriangle(vertices, start_index, Color.FromArgb(255, 0, 0, 88), _
          1, -(0 - offset), -1 / sqrt3, _
          -1, -(0 - offset), -1 / sqrt3, _
          0, -(y - offset), 0)
        MakeTriangle(vertices, start_index, Color.DarkBlue, _
          0, -(0 - offset), sqrt3 - 1 / sqrt3, _
          -1, -(0 - offset), -1 / sqrt3, _
          1, -(0 - offset), -1 / sqrt3)

        m_VertexBuffer.Unlock()
    End Sub

    Private Sub MakeTriangle( _
        ByVal vertices As CustomVertex.PositionColored(), _
        ByRef start_index As Integer, ByVal clr As Color, _
        ByVal x1 As Single, ByVal y1 As Single, ByVal z1 As Single, _
        ByVal x2 As Single, ByVal y2 As Single, ByVal z2 As Single, _
        ByVal x3 As Single, ByVal y3 As Single, ByVal z3 As Single _
    )
        Const PT_SCALE As Single = 2
        x1 *= PT_SCALE
        y1 *= PT_SCALE
        z1 *= PT_SCALE
        x2 *= PT_SCALE
        y2 *= PT_SCALE
        z2 *= PT_SCALE
        x3 *= PT_SCALE
        y3 *= PT_SCALE
        z3 *= PT_SCALE

        With vertices(start_index)
            .X = x1
            .Y = y1
            .Z = z1
            .Color = clr.ToArgb
        End With
        start_index += 1

        With vertices(start_index)
            .X = x2
            .Y = y2
            .Z = z2
            .Color = clr.ToArgb
        End With
        start_index += 1

        With vertices(start_index)
            .X = x3
            .Y = y3
            .Z = z3
            .Color = clr.ToArgb
        End With
        start_index += 1
    End Sub

    ' Return the cross product of two vectors.
    Private Function CrossProduct( _
        ByVal x1 As Single, ByVal y1 As Single, ByVal z1 As Single, _
        ByVal x2 As Single, ByVal y2 As Single, ByVal z2 As Single _
    ) As Vector3
        Dim v1 As New Vector3(x1, y1, z1)
        Dim v2 As New Vector3(x2, y2, z2)
        Return Vector3.Cross(v1, v2)
    End Function

    ' Setup the world, view, and projection matrices.
    Private Sub SetupMatrices()
        ' World Matrix:
        ' Rotate the object around a crooked axis by
        ' 2 * Pi radians per 3000 ticks (3 seconds).
        Const TICKS_PER_REV As Integer = 3000
        Dim ms_rotated As Integer = Environment.TickCount Mod TICKS_PER_REV
        Dim angle As Double = ms_rotated * (2 * Math.PI) / TICKS_PER_REV
        m_Device.Transform.World = Matrix.RotationYawPitchRoll(Math.PI / 2, Math.PI / 4, CSng(angle))

        ' View Matrix:
        ' This is defined by giving:
        '       An eye point            (0, 3, 10)
        '       A point to look at      (0, 0, 0)
        '       An "up" direction       <0, 1, 0>
        m_Device.Transform.View = Matrix.LookAtLH( _
            New Vector3(0, 3, 10), _
            New Vector3(0, 0, 0), _
            New Vector3(0, 1, 0))

        ' Projection Matrix:
        ' Perspective transformation defined by:
        '       Field of view           Pi / 4
        '       Aspect ratio            1
        '       Near clipping plane     Z = 1
        '       Far clipping plane      Z = 100
        m_Device.Transform.Projection = _
            Matrix.PerspectiveFovLH(Math.PI / 4, 1, 1, 100)
    End Sub
End Class
