from typing import Tuple
import numpy as np
from scipy.special import logsumexp


def to_2d(a: np.ndarray,
          type: str="col") -> np.ndarray:
    '''
    Przekształca tablicę jednowymiarową w dwuwymiarową
    '''

    assert a.ndim == 1, \
        "Tablice wejściowe muszą być jednowymiarowe"

    if type == "col":
        return a.reshape(-1, 1)
    elif type == "row":
        return a.reshape(1, -1)


def normalize(a: np.ndarray):
    other = 1 - a
    return np.concatenate([a, other], axis=1)


def unnormalize(a: np.ndarray):
    return a[np.newaxis, 0]


def permute_data(X: np.ndarray, y: np.ndarray):
    perm = np.random.permutation(X.shape[0])
    return X[perm], y[perm]


Batch = Tuple[np.ndarray, np.ndarray]


def generate_batch(X: np.ndarray,
                   y: np.ndarray,
                   start: int = 0,
                   batch_size: int = 10) -> Batch:

    assert (X.dim() == 2) and (y.dim() == 2), \
        "X i Y muszą być dwuwymiarowe"

    if start+batch_size > X.shape[0]:
        batch_size = X.shape[0] - start

    X_batch, y_batch = X[start:start+batch_size], y[start:start+batch_size]

    return X_batch, y_batch


def assert_same_shape(output: np.ndarray,
                      output_grad: np.ndarray):
    assert output.shape == output_grad.shape, \
        '''
        Obie tablice powinny mieć ten sam kształt,
        ale pierwsza tablica ma kształt {0},
        a druga {1}.
        '''.format(tuple(output_grad.shape), tuple(output.shape))
    return None


def assert_dim(t: np.ndarray,
               dim: int):
    assert t.ndim == dim, \
        '''
        Tablica powinna mieć wymiar {0}, ale ma wymiar {1}
        '''.format(dim, len(t.shape))
    return None


def softmax(x, axis=None):
    return np.exp(x - logsumexp(x, axis=axis, keepdims=True))
