import { injectable, multiInject } from 'inversify';
import { Operator } from '../interfaces/operator.interface';
import { TYPES } from '../types';
import { tryParseNumberString, tryParseOperatorSymbol } from "../utils/math";

@injectable()
export class Calculator {
    constructor(@multiInject(TYPES.Operator) private operators: Operator[]) {}

    evaluate(expression: string) {
        const expressionParts = expression.match(/[\d\.]+|\D+/g);
        if (expressionParts === null) return null;

        const parsedExpressionParts = expressionParts.map(part => {
            const numberParseResult = tryParseNumberString(part);
            if (numberParseResult.isNumberString) return numberParseResult.number;
            
            const operatorParseResult = tryParseOperatorSymbol(part, this.operators);
            if (operatorParseResult.isOperatorSymbol) return operatorParseResult.operator;
    
            throw new Error(`Nieoczekiwana część: ${part}`);
        });
     
     const { result } = parsedExpressionParts.reduce<{
          result: number; 
          queuedOperator: Operator | null
        }>((acc, part) => {
            if (typeof part === 'number') {
            // To jest pierwsza napotkana liczba i ustawiasz ją jako wynik
              if (acc.queuedOperator === null) {
                  return { ...acc, result: part };
            }

            // Istnieje operator kolejkowany — za jego pomocą przeprowadzasz ewaluację
            // poprzedniego wyniku i czyścisz kolejkowany operator
            return {
                queuedOperator: null,
                result: acc.queuedOperator.evaluate(acc.result, part),
             };
        }

        // To jest operator — kolejkujesz go do późniejszego wykonania
        return {
            ...acc,
            queuedOperator: part,
        };
    }, 
    { result: 0, queuedOperator: null });

    return result; 
  }
}
