var game = {
    // Zaczynamy inicjalizowanie obiektów, wczytujemy zasoby i wyświetlamy ekran powitalny
    init: function() {
        // Pobieramy uchwyt elementu canvas oraz kontekstu gry
        game.canvas = document.getElementById("gamecanvas");
        game.context = game.canvas.getContext("2d");

        // Inicjalizujemy obiekty
        levels.init();
        loader.init();
        mouse.init();

        // Ukrywamy wszystkie poziomy gry oraz wyświetlamy ekran startowy
        game.hideScreens();
        game.showScreen("gamestartscreen");
    },

    hideScreens: function() {
        var screens = document.getElementsByClassName("gamelayer");

        // Iterujemy wszystkie warstwy gry i wyłączamy ich wyświetlanie
        for (let i = screens.length - 1; i >= 0; i--) {
            var screen = screens[i];

            screen.style.display = "none";
        }
    },

    hideScreen: function(id) {
        var screen = document.getElementById(id);

        screen.style.display = "none";
    },

    showScreen: function(id) {
        var screen = document.getElementById(id);

        screen.style.display = "block";
    },

    showLevelScreen: function() {
        game.hideScreens();
        game.showScreen("levelselectscreen");
    },

    // Zapisujemy bieżący stan gry: intro, wait-for-firing, firing, fired, load-next-hero, success, failure
    mode: "intro",

    // Współrzędne X i Y procy
    slingshotX: 140,
    slingshotY: 280,

    // Współrzędne X i Y punktu, w którym taśma jest przyczepiona do procy
    slingshotBandX: 140 + 55,
    slingshotBandY: 280 + 23,

    // Flaga służąca do sprawdzenia, czy gra się zakończyła
    ended: false,

    // Wynik gry
    score: 0,

    // Przesunięcie wzdłuż osi X, na podstawie którego przesuwamy ekran od lewej do prawej
    offsetLeft: 0,

    start: function() {
        game.hideScreens();

        // Wyświetlamy element canvas gry oraz wynik
        game.showScreen("gamecanvas");
        game.showScreen("scorescreen");

        game.mode = "intro";
        game.currentHero = undefined;

        game.offsetLeft = 0;
        game.ended = false;

        game.animationFrame = window.requestAnimationFrame(game.animate, game.canvas);

    },

    // Maksymalna prędkość przesuwania klatki w pikselach
    maxSpeed: 3,

    // Przesuwamy ekran, aby jego środek znajdował się w punkcie newCenter
    // (a przynajmniej możliwie jak najbliżej)
    panTo: function(newCenter) {

        // Minimalne i maksymalne przesunięcie
        var minOffset = 0;
        var maxOffset = game.currentLevel.backgroundImage.width - game.canvas.width;

        // Bieżący środek ekranu znajduje się w odległości połowy szerokości ekranu od lewego przesunięcia
        var currentCenter = game.offsetLeft + game.canvas.width / 2;

        // Kontynuujemy przesuwanie, jeśli odległość między nowym a bieżącym środkiem jest większa od 0 i nie przekroczyliśmy minimalnego i maksymalnego przesunięcia
        if (Math.abs(newCenter - currentCenter) > 0 && game.offsetLeft <= maxOffset && game.offsetLeft >= minOffset) {
            // W każdym tyknięciu przesuwamy o połowę odległości od punktu newCenter do punktu currentCenter
            // To ułatwia spowalnianie ruchu
            var deltaX = (newCenter - currentCenter) / 2;

            // Jednakże jeśli wartość deltaX jest bardzo wysoka, ekran będzie się przesuwać zbyt szybko, dlatego jeśli jest ona większa niż wartość maxSpeed
            if (Math.abs(deltaX) > game.maxSpeed) {
                // ograniczamy wartość deltaX do wartości game.maxSpeed (nie zmieniając znaku zmiennej deltaX)
                deltaX = game.maxSpeed * Math.sign(deltaX);
            }

            // Jeśli prawie osiągnęliśmy cel, przejdźmy do zakończenia rundy
            if (Math.abs(deltaX) <= 1) {
                deltaX = (newCenter - currentCenter);
            }

            // Na końcu dodajemy poprawioną wartość deltaX do wartości offsetX, tym samym przesuwając ekran o wartość deltaX
            game.offsetLeft += deltaX;

            // Upewniamy się, że nie przekroczyliśmy minimalnego i maksymalnego limitu
            if (game.offsetLeft <= minOffset) {
                game.offsetLeft = minOffset;

                // Informujemy funkcję wywołującą, że przesunęliśmy się możliwie najbliżej do punktu newCenter
                return true;
            } else if (game.offsetLeft >= maxOffset) {
                game.offsetLeft = maxOffset;

                // Informujemy funkcję wywołującą, że przesunęliśmy się możliwie najbliżej do punktu newCenter
                return true;
            }

        } else {
            // Informujemy funkcję wywołującą, że przesunęliśmy się możliwie najbliżej do punktu newCenter
            return true;
        }
    },

    handleGameLogic: function() {
        if (game.mode === "intro") {
            if (game.panTo(700)) {
                game.mode = "load-next-hero";
            }
        }

        if (game.mode === "wait-for-firing") {
            if (mouse.dragging) {
                game.panTo(mouse.x + game.offsetLeft);
            } else {
                game.panTo(game.slingshotX);
            }
        }

        if (game.mode === "load-next-hero") {
            // Najpierw liczymy bohaterów i czarne charaktery oraz zapełniamy nimi odpowiednie tablice
            // Sprawdzamy, czy jakiś czarny charakter jest żywy, a jeśli nie, kończymy poziom (sukces)
            // Sprawdzamy, czy mamy do wczytania jakichś bohaterów, a jeśli nie, kończymy poziom (porażka)
            // Wczytujemy bohatera i ustawiamy tryb na wait-for-firing
            game.mode = "wait-for-firing";
        }

        if (game.mode === "firing") {
            // Jeśli przycisk myszy jest naciśnięty, umożliwiamy przeciągnięcie bohatera i wycelowanie
            // Jeśli nie, wystrzeliwujemy bohatera w powietrze
        }

        if (game.mode === "fired") {
            // Przesuwamy do położenia bieżącego bohatera w trakcie jego lotu
            // Czekamy, aż bohater przestanie się poruszać lub znajdzie się poza granicami ekranu
        }


        if (game.mode === "level-success" || game.mode === "level-failure") {
            // Najpierw przesuwamy grę z powrotem w lewo
            // Następnie pokazujemy koniec gry i ekran końcowy
        }

    },

    animate: function() {

        // Obsługa przesuwania, stanów gry oraz sterowanie przepływem
        game.handleGameLogic();

        // Rysujemy tło i definiujemy przewijanie z efektem paralaksy
        // Najpierw rysujemy obraz tła przesunięty o ułamek odległości offsetLeft (1/4)
        // Im większa wartość ułamka, tym bliższe wydaje się tło
        game.context.drawImage(game.currentLevel.backgroundImage, game.offsetLeft / 4, 0, game.canvas.width, game.canvas.height, 0, 0, game.canvas.width, game.canvas.height);
        // Następnie rysujemy obraz pierwszego planu przesunięty o całą odległość offsetLeft
        game.context.drawImage(game.currentLevel.foregroundImage, game.offsetLeft, 0, game.canvas.width, game.canvas.height, 0, 0, game.canvas.width, game.canvas.height);


        // Rysujemy podstawę procy przesuniętą o całą odległość offsetLeft
        game.context.drawImage(game.slingshotImage, game.slingshotX - game.offsetLeft, game.slingshotY);

        // Rysujemy przód procy przesunięty o całą odległość offsetLeft
        game.context.drawImage(game.slingshotFrontImage, game.slingshotX - game.offsetLeft, game.slingshotY);

        if (!game.ended) {
            game.animationFrame = window.requestAnimationFrame(game.animate, game.canvas);
        }
    },

};

var levels = {
    // Dane poziomu
    data: [{   // Pierwszy poziom
        foreground: "desert-foreground",
        background: "clouds-background",
        entities: []
    }, {   // Drugi poziom
        foreground: "desert-foreground",
        background: "clouds-background",
        entities: []
    }],

    // Inicjalizowanie ekranu wyboru poziomów
    init: function() {
        var levelSelectScreen = document.getElementById("levelselectscreen");

        // Funkcja obsługi zdarzeń, którą należy wywołać
        var buttonClickHandler = function() {
            game.hideScreen("levelselectscreen");

            // Etykiety poziomów mają wartości 1 i 2. Poziomy mają wartości 0 i 1.
            levels.load(this.value - 1);
        };


        for (let i = 0; i < levels.data.length; i++) {
            var button = document.createElement("input");

            button.type = "button";
            button.value = (i + 1); // Etykiety poziomów mają wartości 1 i 2
            button.addEventListener("click", buttonClickHandler);

            levelSelectScreen.appendChild(button);
        }

    },

    // Wczytujemy wszystkie dane i obrazy dla określonego poziomu
    load: function(number) {

        // Deklarujemy nowy obiekt currentLevel
        game.currentLevel = { number: number };
        game.score = 0;

        document.getElementById("score").innerHTML = "Wynik: " + game.score;
        var level = levels.data[number];

        // Wczytujemy obrazy tła, pierwszego planu i procy
        game.currentLevel.backgroundImage = loader.loadImage("images/backgrounds/" + level.background + ".png");
        game.currentLevel.foregroundImage = loader.loadImage("images/backgrounds/" + level.foreground + ".png");
        game.slingshotImage = loader.loadImage("images/slingshot.png");
        game.slingshotFrontImage = loader.loadImage("images/slingshot-front.png");

        // Po wczytaniu zasobów wywołujemy metodę game.start()
        loader.onload = game.start;

    }
};

var loader = {
    loaded: true,
    loadedCount: 0, // Zasoby, które zostały dotychczas wczytane
    totalCount: 0, // Całkowita liczba zasobów, które oczekują na wczytanie

    init: function() {
        // Sprawdzamy wsparcie dla zasobów dźwiękowych
        var mp3Support, oggSupport;
        var audio = document.createElement("audio");

        if (audio.canPlayType) {
               // Obecnie metoda canPlayType() zwraca: "", "maybe" lub "probably"
            mp3Support = "" !== audio.canPlayType("audio/mpeg");
            oggSupport = "" !== audio.canPlayType("audio/ogg; codecs=\"vorbis\"");
        } else {
            // Brak wsparcia dla znacznika audio
            mp3Support = false;
            oggSupport = false;
        }

        // Sprawdzamy wsparcie formatu ogg, następnie mp3, a w razie niepowodzenia ustawiamy wartość zmiennej soundFileExtn na undefined
        loader.soundFileExtn = oggSupport ? ".ogg" : mp3Support ? ".mp3" : undefined;
    },

    loadImage: function(url) {
        this.loaded = false;
        this.totalCount++;

        game.showScreen("loadingscreen");

        var image = new Image();

        image.addEventListener("load", loader.itemLoaded, false);
        image.src = url;

        return image;
    },

    soundFileExtn: ".ogg",

    loadSound: function(url) {
        this.loaded = false;
        this.totalCount++;

        game.showScreen("loadingscreen");

        var audio = new Audio();

        audio.addEventListener("canplaythrough", loader.itemLoaded, false);
        audio.src = url + loader.soundFileExtn;

        return audio;
    },

    itemLoaded: function(ev) {
        // Przestajemy nasłuchiwać zdarzenia (load lub canplaythrough) dla elementu, który został już wczytany
        ev.target.removeEventListener(ev.type, loader.itemLoaded, false);

        loader.loadedCount++;

        document.getElementById("loadingmessage").innerHTML = "Loaded " + loader.loadedCount + " of " + loader.totalCount;

        if (loader.loadedCount === loader.totalCount) {
            // Wczytywanie zostało zakończone
            // Resetujemy i czyścimy obiekt wczytujący
            loader.loaded = true;
            loader.loadedCount = 0;
            loader.totalCount = 0;

            // Ukrywamy ekran wczytywania
            game.hideScreen("loadingscreen");

            // i wywołujemy metodę loader.onload, o ile istnieje
            if (loader.onload) {
                loader.onload();
                loader.onload = undefined;
            }
        }
    }
};

var mouse = {
    x: 0,
    y: 0,
    down: false,
    dragging: false,

    init: function() {
        var canvas = document.getElementById("gamecanvas");

        canvas.addEventListener("mousemove", mouse.mousemovehandler, false);
        canvas.addEventListener("mousedown", mouse.mousedownhandler, false);
        canvas.addEventListener("mouseup", mouse.mouseuphandler, false);
        canvas.addEventListener("mouseout", mouse.mouseuphandler, false);
    },

    mousemovehandler: function(ev) {
        var offset = game.canvas.getBoundingClientRect();

        mouse.x = ev.clientX - offset.left;
        mouse.y = ev.clientY - offset.top;

        if (mouse.down) {
            mouse.dragging = true;
        }

        ev.preventDefault();
    },

    mousedownhandler: function(ev) {
        mouse.down = true;

        ev.preventDefault();
    },

    mouseuphandler: function(ev) {
        mouse.down = false;
        mouse.dragging = false;

        ev.preventDefault();
    }
};

// Inicjalizujemy grę dopiero po całkowitym wczytaniu strony
window.addEventListener("load", function() {
    game.init();
});
