// Dla wygody deklarujemy zmienne dla wszystkich często wykorzystywanych obiektów 
var b2Vec2 = Box2D.Common.Math.b2Vec2;
var b2BodyDef = Box2D.Dynamics.b2BodyDef;
var b2Body = Box2D.Dynamics.b2Body;
var b2FixtureDef = Box2D.Dynamics.b2FixtureDef;
var b2World = Box2D.Dynamics.b2World;
var b2PolygonShape = Box2D.Collision.Shapes.b2PolygonShape;
var b2CircleShape = Box2D.Collision.Shapes.b2CircleShape;
var b2DebugDraw = Box2D.Dynamics.b2DebugDraw;
var b2RevoluteJointDef = Box2D.Dynamics.Joints.b2RevoluteJointDef;

var world;

// 30 pikseli naszego obiektu canvas reprezentuje 1 metr w świecie box2d
var scale = 30;

function init() {
    // Definiujemy świat box2d, który będzie wykonywać większość obliczeń fizycznych 
    var gravity = new b2Vec2(0, 9.8); // Deklarujemy grawitację o wartości 9.8 m/s^2, skierowaną w dół

    // Umożliwiamy zasypianie obiektów w spoczynku oraz ich wykluczanie z obliczeń 
    var allowSleep = true;

    world = new b2World(gravity, allowSleep);

    createFloor();

    // Tworzymy ciała z prostymi kształtami
    createRectangularBody();
    createCircularBody();
    createSimplePolygonBody();

    // Tworzymy ciało składające się z dwóch kształtów 
    createComplexBody();

    // Łączymy dwa ciała za pomocą złączenia przegubowego 
    createRevoluteJoint();

    // Tworzymy ciało ze specjalnymi danymi użytkownika
    createSpecialBody();

    // Tworzymy funkcje obsługi kontaktu i śledzimy zdarzenia
    listenForContact();    

    setupDebugDraw();

    // Rozpoczynamy pętlę animacji Box2D
    animate();
}

function createFloor() {
    // Definicja ciała zawiera wszystkie dane potrzebne do skonstruowania ciała sztywnego.
    var bodyDef = new b2BodyDef;

    bodyDef.type = b2Body.b2_staticBody;
    bodyDef.position.x = 640 / 2 / scale;
    bodyDef.position.y = 450 / scale;

    // Mocowanie służy do przyłączenia kształtu do ciała, w celu zapewnienia wykrywania kolizji
    // Za pomocą następującej definicji tworzymy obiekt typu fixture 
    var fixtureDef = new b2FixtureDef;

    fixtureDef.density = 1.0;
    fixtureDef.friction = 0.5;
    fixtureDef.restitution = 0.2;

    fixtureDef.shape = new b2PolygonShape;
    fixtureDef.shape.SetAsBox(320 / scale, 10 / scale); // 640 pikseli szerokości i 20 pikseli wysokości

    var body = world.CreateBody(bodyDef);
    var fixture = body.CreateFixture(fixtureDef);
}


var context;

function setupDebugDraw() {
    context = document.getElementById("canvas").getContext("2d");

    var debugDraw = new b2DebugDraw();

    // Używamy kontekstu obiektu canvas, aby narysować ekran debugowania
    debugDraw.SetSprite(context);
    // Ustawiamy skalę
    debugDraw.SetDrawScale(scale);
    // Wypełniamy pola przezroczystością o wartości 0.3
    debugDraw.SetFillAlpha(0.3);
    // Rysujemy linie o grubości 1
    debugDraw.SetLineThickness(1.0);
    // Wyświetlamy wszystkie kształty i złączenia
    debugDraw.SetFlags(b2DebugDraw.e_shapeBit | b2DebugDraw.e_jointBit);

    // Zaczynamy rysowanie w naszym świecie w trybie debugowania 
    world.SetDebugDraw(debugDraw);
}

var timeStep = 1 / 60;

// Zgodnie z podręcznikiem Box2d, sugerowana liczba iteracji dla Box2D wynosi 8 w przypadku prędkości oraz 3 w przypadku położenia.
var velocityIterations = 8;
var positionIterations = 3;

function animate() {
    world.Step(timeStep, velocityIterations, positionIterations);
    world.ClearForces();

    world.DrawDebugData();

    // Niestandardowe rysowanie 
    if (specialBody) {
        drawSpecialBody();
    }

    // Niszczymy ciało specjalne po jego śmierci
    if (specialBody && specialBody.GetUserData().life <= 0) {
        world.DestroyBody(specialBody);
        specialBody = undefined;
        console.log("Ciało specjalne uległo zniszczeniu");
    }

    setTimeout(animate, timeStep);
}

function createRectangularBody() {
    var bodyDef = new b2BodyDef;

    bodyDef.type = b2Body.b2_dynamicBody;
    bodyDef.position.x = 40 / scale;
    bodyDef.position.y = 100 / scale;

    var fixtureDef = new b2FixtureDef;

    fixtureDef.density = 1.0;
    fixtureDef.friction = 0.5;
    fixtureDef.restitution = 0.3;

    fixtureDef.shape = new b2PolygonShape;
    fixtureDef.shape.SetAsBox(30 / scale, 50 / scale);

    var body = world.CreateBody(bodyDef);
    var fixture = body.CreateFixture(fixtureDef);
}

function createCircularBody() {
    var bodyDef = new b2BodyDef;

    bodyDef.type = b2Body.b2_dynamicBody;
    bodyDef.position.x = 130 / scale;
    bodyDef.position.y = 100 / scale;

    var fixtureDef = new b2FixtureDef;

    fixtureDef.density = 1.0;
    fixtureDef.friction = 0.5;
    fixtureDef.restitution = 0.7;

    fixtureDef.shape = new b2CircleShape(30 / scale);

    var body = world.CreateBody(bodyDef);
    var fixture = body.CreateFixture(fixtureDef);
}

function createSimplePolygonBody() {
    var bodyDef = new b2BodyDef;

    bodyDef.type = b2Body.b2_dynamicBody;
    bodyDef.position.x = 230 / scale;
    bodyDef.position.y = 50 / scale;

    var fixtureDef = new b2FixtureDef;

    fixtureDef.density = 1.0;
    fixtureDef.friction = 0.5;
    fixtureDef.restitution = 0.6;

    fixtureDef.shape = new b2PolygonShape;
    // Tworzymy tablicę punktów b2Vec2 w kierunku zgodnym z ruchem wskazówek zegara
    var points = [
        new b2Vec2(0, 0),
        new b2Vec2(40 / scale, 50 / scale),
        new b2Vec2(50 / scale, 100 / scale),
        new b2Vec2(-50 / scale, 100 / scale),
        new b2Vec2(-40 / scale, 50 / scale),

    ];

    // Za pomocą funkcji SetAsArray() definiujemy kształt korzystając z tablicy punktów
    fixtureDef.shape.SetAsArray(points, points.length);

    var body = world.CreateBody(bodyDef);

    var fixture = body.CreateFixture(fixtureDef);

}

function createComplexBody() {
    var bodyDef = new b2BodyDef;

    bodyDef.type = b2Body.b2_dynamicBody;
    bodyDef.position.x = 350 / scale;
    bodyDef.position.y = 50 / scale;
    var body = world.CreateBody(bodyDef);

    // Tworzymy pierwsze mocowanie i dołączamy do ciała okrągły kształt 
    var fixtureDef = new b2FixtureDef;

    fixtureDef.density = 1.0;
    fixtureDef.friction = 0.5;
    fixtureDef.restitution = 0.7;
    fixtureDef.shape = new b2CircleShape(40 / scale);
    body.CreateFixture(fixtureDef);

    // Tworzymy drugie mocowanie i dołączamy do ciała prostokątny kształt.
    fixtureDef.shape = new b2PolygonShape;
    var points = [
        new b2Vec2(0, 0),
        new b2Vec2(40 / scale, 50 / scale),
        new b2Vec2(50 / scale, 100 / scale),
        new b2Vec2(-50 / scale, 100 / scale),
        new b2Vec2(-40 / scale, 50 / scale),
    ];

    fixtureDef.shape.SetAsArray(points, points.length);
    body.CreateFixture(fixtureDef);
}

function createRevoluteJoint() {
    //Definiujemy pierwsze ciało
    var bodyDef1 = new b2BodyDef;

    bodyDef1.type = b2Body.b2_dynamicBody;
    bodyDef1.position.x = 480 / scale;
    bodyDef1.position.y = 50 / scale;
    var body1 = world.CreateBody(bodyDef1);

    // Tworzymy pierwsze mocowanie i dołączamy do ciała prostokątny kształt
    var fixtureDef1 = new b2FixtureDef;

    fixtureDef1.density = 1.0;
    fixtureDef1.friction = 0.5;
    fixtureDef1.restitution = 0.5;
    fixtureDef1.shape = new b2PolygonShape;
    fixtureDef1.shape.SetAsBox(50 / scale, 10 / scale);

    body1.CreateFixture(fixtureDef1);

    // Definiujemy drugie ciało
    var bodyDef2 = new b2BodyDef;

    bodyDef2.type = b2Body.b2_dynamicBody;
    bodyDef2.position.x = 470 / scale;
    bodyDef2.position.y = 50 / scale;
    var body2 = world.CreateBody(bodyDef2);

    // Tworzymy drugie mocowanie i dołączamy do ciała wielokątny kształt
    var fixtureDef2 = new b2FixtureDef;

    fixtureDef2.density = 1.0;
    fixtureDef2.friction = 0.5;
    fixtureDef2.restitution = 0.5;
    fixtureDef2.shape = new b2PolygonShape;
    var points = [
        new b2Vec2(0, 0),
        new b2Vec2(40 / scale, 50 / scale),
        new b2Vec2(50 / scale, 100 / scale),
        new b2Vec2(-50 / scale, 100 / scale),
        new b2Vec2(-40 / scale, 50 / scale),
    ];

    fixtureDef2.shape.SetAsArray(points, points.length);
    body2.CreateFixture(fixtureDef2);


    // Tworzymy złączenie między ciałami body1 i body2
    var jointDef = new b2RevoluteJointDef;
    var jointCenter = new b2Vec2(470 / scale, 50 / scale);

    jointDef.Initialize(body1, body2, jointCenter);
    world.CreateJoint(jointDef);
}

var specialBody;

function createSpecialBody() {
    var bodyDef = new b2BodyDef;

    bodyDef.type = b2Body.b2_dynamicBody;
    bodyDef.position.x = 450 / scale;
    bodyDef.position.y = 0 / scale;

    specialBody = world.CreateBody(bodyDef);
    specialBody.SetUserData({ name: "special", life: 250 });

    // Tworzymy mocowanie, aby dołączyć do ciała okrągły kształt
    var fixtureDef = new b2FixtureDef;

    fixtureDef.density = 1.0;
    fixtureDef.friction = 0.5;
    fixtureDef.restitution = 0.5;

    fixtureDef.shape = new b2CircleShape(30 / scale);

    var fixture = specialBody.CreateFixture(fixtureDef);
}

function listenForContact() {
    var listener = new Box2D.Dynamics.b2ContactListener;

    listener.PostSolve = function(contact, impulse) {
        var body1 = contact.GetFixtureA().GetBody();
        var body2 = contact.GetFixtureB().GetBody();

        // Jeśli jedno z tych ciał jest ciałem specjalnym, zmniejszamy jego żywotność
        if (body1 == specialBody || body2 == specialBody) {
            var impulseAlongNormal = impulse.normalImpulses[0];

            specialBody.GetUserData().life -= impulseAlongNormal;
            console.log("To ciało specjalne uległo kolizji o sile impulsu  ", impulseAlongNormal, ", a jego żywotność wynosi obecnie ", specialBody.GetUserData().life);
        }
    };
    world.SetContactListener(listener);
}

function drawSpecialBody() {
    // Pobieramy położenie i kąt ciała
    var position = specialBody.GetPosition();
    var angle = specialBody.GetAngle();

    // Dokonujemy translacji i obracamy oś, aby uzgodnić ją z położeniem i kątem obrotu ciała
    context.translate(position.x * scale, position.y * scale);
    context.rotate(angle);

    // Rysujemy wypełnioną okrągłą twarz
    context.fillStyle = "rgb(200, 150, 250)";
    context.beginPath();
    context.arc(0, 0, 30, 0, 2 * Math.PI, false);
    context.fill();

    // Rysujemy dwoje prostokątnych oczu
    context.fillStyle = "rgb(255, 255, 255)";
    context.fillRect(-15, -15, 10, 5);
    context.fillRect(5, -15, 10, 5);

    // W zależności od okoliczności, rysujemy łuk wygięty w górę lub w dół, reprezentujący uśmiech lub smutny wyraz twarzy
    context.strokeStyle = "rgb(255, 255, 255)";
    context.beginPath();
    if (specialBody.GetUserData().life > 100) {
        context.arc(0, 0, 10, Math.PI, 2 * Math.PI, true);
    } else {
        context.arc(0, 10, 10, Math.PI, 2 * Math.PI, false);
    }
    context.stroke();

    // Dokonujemy translacji i obrotu osi, aby przywrócić jej oryginalne położenie i kąt 
    context.rotate(-angle);
    context.translate(-position.x * scale, -position.y * scale);
}