/*
 * $Revision$
 * $Date$
 *
* Prawa autorskie (c) 2007 O'Reilly Media.  Wszystkie prawa zastrzeone.
 *
 * Objte licencj Apache License w wersji 2.0 ("Licencja"); Pliku
 * mona uy tylko przy zachowaniu zgodnoci z warunkami Licencji. Kopi
 * Licencji mona uzyska pod adresem:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Jeli nie obowizuje odpowiednie prawo lub nie uzyskano pisemnej zgody,
 * oprogramowanie objte Licencj jest rozpowszechniane "W STANIE, W JAKIM JEST" BEZ 
 * GWARANCJI LUB JAKICHKOLWIEK WARUNKW jawnych lub domniemanych. W Licencji mona
 * znale szczegy dotyczce zarzdzania zezwoleniami i ograniczeniami.
 */

package com.oreilly.tomcat.filter;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Odfiltrowuje z da HTTP dane wprowadzone przez uytkownika ze zymi zamiarami
 * w celu uniknicia midzy innymi atakw XSS (Cross Site Scripting), a take bazujcych
 * na wprowadzaniu instrukcji SQL i kodu HTML.
 *
 * @autor Jason Brittain
 */
public class BadInputFilter implements Filter {

    // --------------------------------------------- Zmienne statyczne

    /**
     * Opis implementacji.
     */
    protected static String info =
        "com.oreilly.tomcat.filter.BadInputFilter/2.0";

    /**
     * Pusta tablica acuchowa wielokrotnego uycia jako wskanik typu dla
     * toArray( ).
     */
    private static final String[] STRING_ARRAY = new String[0];

    // ------------------------------------------- Zmienne instancji

    /**
     * Znacznik okrelajcy, czy przed cudzysowami stanowicymi cz
     * dania zostanie wstawiony znak \ czy nie.
     */
    protected boolean escapeQuotes = false;

    /**
     * Znacznik okrelajcy, czy przed nawiasami <>  stanowicymi cz
     * dania zostanie wstawiony znak \ czy nie.
     */
    protected boolean escapeAngleBrackets = false;

    /**
     * Znacznik okrelajcy, czy przed nazwami obiektw i funkcji JavaScript stanowicymi cz
     * dania zostanie wstawiony znak \ czy nie.
     */
    protected boolean escapeJavaScript = false;

    /**
     * Mapowanie zastpowania (pasujce wyraenie regularne, podstawienie),
     * ktre suy do zastpowania pojedynczych (') i podwjnych (") znakw
     * cudzysowu odpowiednikami ze znakiem \ na pocztku (nie mona wykorzysta
     * ich do niecnych celw).
     */
    protected HashMap<String, String> quotesHashMap =
        new HashMap<String, String>( );

    /**
     * Mapowanie zastpowania (pasujce wyraenie regularne, podstawienie),
     * ktre suy do zastpowania nawiasw <> odpowiednikami ze znakiem \ 
     * na pocztku (nie mona wykorzysta ich do niecnych celw).
     */
    protected HashMap<String, String> angleBracketsHashMap =
        new HashMap<String, String>( );

    /**
     * Mapowanie zastpowania (pasujce wyraenie regularne, podstawienie),
     * ktre suy do zastpowania potencjalnie niebezpiecznych wywoa funkcji JavaScript 
     * odpowiednikami ze znakiem \ na pocztku (nie mona wykorzysta ich do niecnych celw).
     */
    protected HashMap<String, String> javaScriptHashMap =
        new HashMap<String, String>( );

    /**
     * Zestaw wyrae <code>allow</code> oddzielonych przecinkami.
     */
    protected String allow = null;

    /**
     * Zestaw wyrae regularnych <code>allow</code>, ktre bd przetwarzane.
     */
    protected Pattern allows[] = new Pattern[0];

    /**
     * Zestaw wyrae regularnych <code>allow</code>, ktre bd przetwarzane..
     */
    protected Pattern denies[] = new Pattern[0];

    /**
     * Zestaw wyrae <code>deny</code> oddzielonych przecinkami.
     */
    protected String deny = null;

    /**
     * Mapa wyrae regularnych uywanych do filtrowania parametrw. Kluczem
     * jest szukane wyraenie regularne String, a wartoci wyraenie regularne
     * String suce do modyfikowania parametru, jeli zostanie znaleziony
     * acuch String.
     */
    protected HashMap<String, String> parameterEscapes =
        new HashMap<String, String>( );

    /**
     * Element ServletContext, w obrbie ktrego dziaa filtr. Stosowane na potrzeby 
     * rejestrowania.
     */
    protected ServletContext servletContext;

    /**
     * W przypadku Tomcata mapa parametrw musi be odblokowana, zmodyfikowana, a nastpnie
     * zablokowana.  Jednak klasa dysponujca metod umoliwiajc to wchodzi w skad Tomcata,
     * a nie serwletowego interfejsu API. W zwizku z tym klasa ta nie powinna by widoczna dla aplikacji WWW
     * (w przypadku serwera Tomcat 6.0 domylnie tak nie jest).
     * Do wywoania tej metody filtr uywa refleksji.
     */
    protected Method setLockedMethod;

    // ------------------------------------------------- Konstruktory

    /**
     * Konstruowanie nowego egzemplarza klasy z domylnymi wartociami waciwoci.
     */
    public BadInputFilter( ) {

        // Uycie map wyrae regularnych ze znakami \.
        quotesHashMap.put("\"", "&quot;");
        quotesHashMap.put("\'", "&#39;");
        quotesHashMap.put("`", "&#96;");
        angleBracketsHashMap.put("<", "&lt;");
        angleBracketsHashMap.put(">", "&gt;");
        javaScriptHashMap.put(
            "document(.*)\\.(.*)cookie", "document&#46;&#99;ookie");
        javaScriptHashMap.put("eval(\\s*)\\(", "eval&#40;");
        javaScriptHashMap.put("setTimeout(\\s*)\\(", "setTimeout$1&#40;");
        javaScriptHashMap.put("setInterval(\\s*)\\(", "setInterval$1&#40;");
        javaScriptHashMap.put("execScript(\\s*)\\(", "exexScript$1&#40;");
        javaScriptHashMap.put("(?i)javascript(?-i):", "javascript&#58;");

    }

    // --------------------------------------------------- Waciwoci

    /**
     * Pobiera znacznik, ktry okrela, czy filtr umieci znak \
     * przed wszelkimi znakami cudzysowu (zarwno podwjnego, jak i pojedynczego)
     * stanowicymi cz dania, ktre ma by przetworzone.
     */
    public boolean getEscapeQuotes( ) {

        return escapeQuotes;

    }

    /**
     * Ustawia znacznik, ktry okrela, czy filtr umieci znak \
     * przed wszelkimi znakami cudzysowu (zarwno podwjnego, jak i pojedynczego)
     * stanowicymi cz dania, ktre ma by przetworzone.
     *
     * @param escapeQuotes
     */
    public void setEscapeQuotes(boolean escapeQuotes) {

        this.escapeQuotes = escapeQuotes;
        if (escapeQuotes) {
            // Wstawienie znaku \ dla wszystkich cudzysoww.
            parameterEscapes.putAll(quotesHashMap);
        }

    }

    /**
     * Pobiera znacznik, ktry okrela, czy filtr umieci znak \
     * przed wszelkimi nawiasami <> stanowicymi cz dania, ktre ma by przetworzone.
     */
    public boolean getEscapeAngleBrackets( ) {

        return escapeAngleBrackets;

    }

    /**
     * Ustawia znacznik, ktry okrela, czy filtr umieci znak \
     * przed wszelkimi nawiasami <> stanowicymi cz dania, ktre ma by przetworzone.
     *
     * @param escapeAngleBrackets
     */
    public void setEscapeAngleBrackets(boolean escapeAngleBrackets) {

        this.escapeAngleBrackets = escapeAngleBrackets;
        if (escapeAngleBrackets) {
            // Wstawienie znaku \ przed wszystkimi nawiasami <>.
            parameterEscapes.putAll(angleBracketsHashMap);
        }

    }

    /**
     * Pobiera znacznik, ktry okrela, czy filtr umieci znak \
     * przed wszelkimi potencjalnie niebezpiecznymi odwoaniami do funkcji
     * i obiektw JavaScript stanowicych cz dania, ktre ma by przetworzone.
     */
    public boolean getEscapeJavaScript( ) {
        return escapeJavaScript;

    }

    /**
     * Ustawia znacznik, ktry okrela, czy filtr umieci znak \
     * przed wszelkimi potencjalnie niebezpiecznymi odwoaniami do funkcji
     * i obiektw JavaScript stanowicych cz dania, ktre ma by przetworzone.
     *
     * @param escapeJavaScript
     */
    public void setEscapeJavaScript(boolean escapeJavaScript) {

        this.escapeJavaScript = escapeJavaScript;
        if (escapeJavaScript) {
            // Wstawienie znaku \ przed potencjalnie niebezpiecznymi wywoaniami metod JavaScript.
            parameterEscapes.putAll(javaScriptHashMap);
        }

    }

    /**
     * Zwraca zestaw wyrae <code>allow</code> oddzielonych przecinkami i
     * skonfigurowanych dla filtra. W przeciwnym razie zwraca <code>null</code>.
     */
    public String getAllow( ) {

        return (this.allow);

    }

    /**
     * Ustawia zestaw wyrae <code>allow</code> oddzielonych przecinkami i
     * skonfigurowanych dla filtra.
     *
     * @Parametr allow; Nowy zestaw wyrae zezwalajcych
     */
    public void setAllow(String allow) {

        this.allow = allow;
        allows = precalculate(allow);
        servletContext.log("BadInputFilter: allow = " + deny);

    }

    /**
     * Zwraca zestaw wyrae <code>deny</code> oddzielonych przecinkami i
     * skonfigurowanych dla filtra. W przeciwnym razie zwraca <code>null</code>.
     */
    public String getDeny( ) {
        return (this.deny);

    }

    /**
     * Ustawia zestaw wyrae <code>deny</code> oddzielonych przecinkami i
     * skonfigurowanych dla filtra.
     *
     * @Parametr deny; Nowy zestaw wyrae odrzucajcych
     */
    public void setDeny(String deny) {

        this.deny = deny;
        denies = precalculate(deny);
        servletContext.log("BadInputFilter: deny = " + deny);

    }

    // ----------------------------------------------- Publiczne metody

    /**
     * {@inheritDoc}
     */
    public void init(FilterConfig filterConfig) throws ServletException {

        servletContext = filterConfig.getServletContext( );

        // Analizowanie parametrw inicjalizacyjnych filtra
        setAllow(filterConfig.getInitParameter("allow"));
        setDeny(filterConfig.getInitParameter("deny"));
        String initParam = filterConfig.getInitParameter("escapeQuotes");
        if (initParam != null) {
            boolean flag = Boolean.parseBoolean(initParam);
            setEscapeQuotes(flag);
        }
        initParam = filterConfig.getInitParameter("escapeAngleBrackets");
        if (initParam != null) {
            boolean flag = Boolean.parseBoolean(initParam);
            setEscapeAngleBrackets(flag);
        }
        initParam = filterConfig.getInitParameter("escapeJavaScript");
        if (initParam != null) {
            boolean flag = Boolean.parseBoolean(initParam);
            setEscapeJavaScript(flag);
        }

        servletContext.log(toString( ) + " initialized.");

    }

    /**
     * Porzdkuje parametry dania zanim dane wejciowe uytkownika ze zymi
     * zamiarami trafi do aplikacji WWW.
     *
     * @Parametr request; danie serwletowe do przetworzenia
     * @Parametr response; Odpowied serwletowa do utworzenia
     *
     * @Wyjtek IOException, jeli wystpi bd wejcia/wyjcia
     * @Wyjtek ServletException, jeli wystpi bd serwleta
     */
    public void doFilter(ServletRequest request, ServletResponse response,
                             FilterChain filterChain)
        throws IOException, ServletException {

        // Pominicie filtrowania dla da i odpowiedzi innych ni HTTP.
        if (!(request instanceof HttpServletRequest) ||
            !(response instanceof HttpServletResponse)) {
            filterChain.doFilter(request, response);
            return;
        }

        // Przepuszczanie da tylko na podstawie regu zezwalajcych i odrzucajcych.
        if (processAllowsAndDenies(request, response)) {
            // Filtrowanie danych wejciowych pod ktem potencjalnie niebezpiecznego
            // kodu JavaScript, dziki czemu przed przetworzeniem przez Tomcata
            // dania zostan z niego usunite dane wprowadzone przez uytkownika
            // ze zymi zamiarami.
            filterParameters(request);

            // Przetworzenie dania.
            filterChain.doFilter(request, response);
        }

    }

    /**
     * Zatrzymuje dania zawierajce w nazwach i wartociach parametrw 
     * niedozwolone wzorce acuchw.
     *
     * @Parametr request; danie serwletowe do przetworzenia
     * @Parametr response; Odpowied serwletowa do utworzenia
     *
     * @Wyjtek IOException, jeli wystpi bd wejcia/wyjcia
     * @Wyjtek ServletException, jeli wystpi bd serwleta
     *
     * @Zwraca false, jeli danie jest niedozwolone; w przeciwnym razie true.
     */
    public boolean processAllowsAndDenies(ServletRequest request,
                                          ServletResponse response)
        throws IOException, ServletException {

        Map paramMap = request.getParameterMap( );
        // Wykonanie ptli dla listy parametrw.
        Iterator y = paramMap.keySet().iterator( );
        while (y.hasNext( )) {
            String name = (String) y.next( );
            String[] values = request.getParameterValues(name);

            // Sprawdzenie, czy nazwa zawiera niedozwolony wzorzec.
            if (!checkAllowsAndDenies(name, response)) {
                return false;
            }

            // Sprawdzenie dla wzorca wartoci parametru.
            if (values != null) {
                for (int i = 0; i < values.length; i++) {
                    String value = values[i];
                    if (!checkAllowsAndDenies(value, response)) {
                        return false;
                    }
                }
            }
        }

        // aden parametr nie spowodowa odrzucenia.  danie powinno by dalej 
        // przetwarzane.
        return true;

    }

    /**
     * Przeprowadzenie filtrowania skonfigurowanego dla tego filtru,
     * polegajcego na dopasowywaniu okrelonej waciwoci dania. Jeli danie
     * moe by kontynuowane, metoda zwrci warto true.  W przeciwnym razie
     * metoda wyle stron odpowiedzi z bdem Forbidden i zwrci warto false.
     *
     * <br><br>
     *
     * Metoda intensywnie korzysta z RequestFilterValve.process( ).
     *
     * @Parametr property; Waciwo dania uywana podczas filtrowania
     * @Parametr response; Odpowied serwletowa do przetworzenia
     *
     * @Wyjtek IOException, jeli wystpi bd wejcia/wyjcia
     * @Wyjtek ServletException, jeli wystpi bd serwleta
     *
     * @Zwraca true, jeli w dalszym cigu danie moe by przetwarzane
     */
    public boolean checkAllowsAndDenies(String property,
                                        ServletResponse response)
        throws IOException, ServletException {

        // Jeli nie byo adnych regu odrzucajcych i zezwalajcych, danie bdzie przetwarzane.
        if (denies.length == 0 && allows.length == 0) {
            return true;
        }

        // Sprawdzenie wzorcw odrzucajcych (jeli istniej)
        for (int i = 0; i < denies.length; i++) {
            Matcher m = denies[i].matcher(property);
            if (m.find( )) {
                if (response instanceof HttpServletResponse) {
                    HttpServletResponse hres =
                        (HttpServletResponse) response;
                    hres.sendError(HttpServletResponse.SC_FORBIDDEN);
                    return false;
                }
            }
        }

        // Sprawdzenie wzorcw zezwalajcych (jeli istniej)
        for (int i = 0; i < allows.length; i++) {
            Matcher m = allows[i].matcher(property);
            if (m.find( )) {
                return true;
            }
        }

        // Zezwolenie, jeli okrelono reguy odrzucajce, lecz nie zezwalajce
        if (denies.length > 0 && allows.length == 0) {
            return true;
        }

        // W przeciwnym razie nastpi odrzucenie dania.
        if (response instanceof HttpServletResponse) {
            HttpServletResponse hres = (HttpServletResponse) response;
            hres.sendError(HttpServletResponse.SC_FORBIDDEN);
        }
        return false;

    }

    /**
     * Filtrowanie wszystkich istniejcych parametrw pod ktem potencjalnie
     * niebezpiecznej zawartoci i umieszczanie na jej pocztku znaku \.
     *
     * @Parametr request; danie zawierajce parametry.
     */
    @SuppressWarnings("unchecked")
    public void filterParameters(ServletRequest request) {

        Map paramMap = ((HttpServletRequest) request).getParameterMap( );
        // Odblokowanie mapy parametrw w celu umoliwienia modyfikowania ich.
        try {
            if (setLockedMethod == null) {
                setLockedMethod = paramMap.getClass( ).getMethod(
                    "setLocked", new Class[] { Boolean.TYPE });
            }
            setLockedMethod.invoke(paramMap, new Object[] { Boolean.FALSE });
        } catch (Exception e) {
            // Brak moliwoci odblokowania parametrw. Jeli dojdzie do tego w czasie pracy Tomcata,
            // nie bdzie mona filtrowa parametrw.
            servletContext.log("BadInputFilter: Nie mona filtrowa parametrw!");
        }


        // Wykonanie ptli dla kadego wzorca zastpujcego.
        Iterator escapesIterator = parameterEscapes.keySet().iterator( );
        while (escapesIterator.hasNext( )) {
            String patternString = (String) escapesIterator.next( );
            Pattern pattern = Pattern.compile(patternString);

            // Wykonanie ptli dla listy parametrw.
            @SuppressWarnings("unchecked")
            String[] paramNames =
                (String[]) paramMap.keySet( ).toArray(STRING_ARRAY);
            for (int i = 0; i < paramNames.length; i++) {
                String name = paramNames[i];
                String[] values = ((HttpServletRequest)
                    request).getParameterValues(name);
                // Sprawdzenie, czy nazwa zawiera wzorzec.
                boolean nameMatch;
                Matcher matcher = pattern.matcher(name);
                nameMatch = matcher.matches( );
                if (nameMatch) {
                    // Poniewa nazwa parametru jest zgodna z wzorcem,
                    // zmienilimy nazw, dodajc parametr o nowej nazwie
                    // i usuwajc  star nazw.
                    String newName = matcher.replaceAll(
                        (String) parameterEscapes.get(patternString));
                    paramMap.remove(name);
                    paramMap.put(newName, values);
                    servletContext.log("Nazwa parametru " + name +
                        " pasuje do wzorca \"" + patternString +
                        "\".  Zdalny adres: " +
                        ((HttpServletRequest) request).getRemoteAddr( ));
                }
                // Sprawdzenie dla wzorca wartoci parametru.
                if (values != null) {
                    for (int j = 0; j < values.length; j++) {
                        String value = values[j];
                        boolean valueMatch;
                        matcher = pattern.matcher(value);
                        valueMatch = matcher.find( );
                        if (valueMatch) {
                            // Poniewa warto okazaa si zgodna, zmodyfikowalimy j,
                            // a nastpnie umiecilimy w tablicy.
                            String newValue;
                            newValue = matcher.replaceAll((String)
                                parameterEscapes.get(patternString));
                            values[j] = newValue;
                            servletContext.log("Warto parametru \"" + name +
                                "\"postaci \"" + value +
                                "\" pasuje do wzorca \"" +
                                patternString + "\".  Zdalny adres: " +
                                ((HttpServletRequest)
                                request).getRemoteAddr( ));
                        }
                    }
                }
            }
        }
        // Po zakoczeniu prba ponownego zablokowania mapy parametrw.
        try {
            if (setLockedMethod == null) {
                setLockedMethod = paramMap.getClass( ).getMethod(
                    "setLocked", new Class[] { Boolean.TYPE });
            }
            setLockedMethod.invoke(paramMap, new Object[] { Boolean.TRUE });
        } catch (Exception e) {
            // Poniewa w tym przypadku zarejestrowano ju informacje, nie musi by nic robione.
        }

    }

    /**
     * Zwrcenie tekstowej reprezentacji obiektu.
     */
    @Override
    public String toString( ) {

        return "BadInputFilter";
    }
    /**
     * {@inheritDoc}
     */
    public void destroy( ) {

    }

    // -------------------------------------------- Chronione metody
 
   /**
     * Zwraca tablic obiektw wyraenia regularnego inicjalizowanych z poziomu
     * okrelonego argumentu, ktry musi mie posta <code>null</code> lub
     * listy wzorcw wyrae regularnych oddzielonych przecinkami.
     *
     * @Parametr list; Lista wzorcw oddzielonych przecinkami
     *
     * @Wyjtek IllegalArgumentException, jeli jeden z wzorcw ma
     *  niepoprawn skadni
     */
    protected Pattern[] precalculate(String list) {

        if (list == null)
            return (new Pattern[0]);
        list = list.trim( );
        if (list.length( ) < 1)
            return (new Pattern[0]);
        list += ",";

        ArrayList<Pattern> reList = new ArrayList<Pattern>( );
        while (list.length( ) > 0) {
            int comma = list.indexOf(',');
            if (comma < 0)
                break;
            String pattern = list.substring(0, comma).trim( );
            try {
                reList.add(Pattern.compile(pattern));
            } catch (PatternSyntaxException e) {
                IllegalArgumentException iae = new IllegalArgumentException(
                    "Bd skadni w wzorcu filtra da" + pattern);
                iae.initCause(e);
                throw iae;
            }
            list = list.substring(comma + 1);
        }

        Pattern reArray[] = new Pattern[reList.size( )];
        return ((Pattern[]) reList.toArray(reArray));
 
   }
}