/*
 * $Revision$
 * $Date$
 *
 * Prawa autorskie (c) 2007 O'Reilly Media.  Wszystkie prawa zastrzeone.
 *
 * Objte licencj Apache License w wersji 2.0 ("Licencja"); Pliku
 * mona uy tylko przy zachowaniu zgodnoci z warunkami Licencji. Kopi
 * Licencji mona uzyska pod adresem:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Jeli nie obowizuje odpowiednie prawo lub nie uzyskano pisemnej zgody,
 * oprogramowanie objte Licencj jest rozpowszechniane "W STANIE, W JAKIM JEST" BEZ 
 * GWARANCJI LUB JAKICHKOLWIEK WARUNKW jawnych lub domniemanych. W Licencji mona
 * znale szczegy dotyczce zarzdzania zezwoleniami i ograniczeniami.
 */

package com.oreilly.tomcat.valve;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.servlet.ServletException;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.catalina.connector.Request;
import org.apache.catalina.connector.Response;
import org.apache.catalina.util.ParameterMap;
import org.apache.catalina.valves.RequestFilterValve;
import org.apache.juli.logging.Log;
import org.apache.juli.logging.LogFactory;

/**
 * Odfiltrowuje z da HTTP dane wprowadzone przez uytkownika ze zymi zamiarami
 * w celu uniknicia midzy innymi atakw XSS (Cross Site Scripting), a take bazujcych
 * na wprowadzaniu instrukcji SQL i kodu HTML.
 *
 * @autor Jason Brittain
 */
public class BadInputValve extends RequestFilterValve {

    // --------------------------------------------- Zmienne statyczne

    /**
     * Egzemplarz Log sucy do rejestrowania.
     */
    private static Log log = LogFactory.getLog(BadInputValve.class);

    /**
     * Opis implementacji.
     */
    protected static String info =
        "com.oreilly.tomcat.valve.BadInputValve/2.0";

    /**
     * Pusta tablica acuchowa wielokrotnego uycia jako wskanik typu dla
     * toArray( ).
     */
    private static final String[] STRING_ARRAY = new String[0];

    // ------------------------------------------- Zmienne instancji

    /**
     * Znacznik okrelajcy, czy przed cudzysowami stanowicymi cz
     * dania zostanie wstawiony znak \ czy nie.
     */
    protected boolean escapeQuotes = false;

    /**
     * Znacznik okrelajcy, czy przed nawiasami <>  stanowicymi cz
     * dania zostanie wstawiony znak \ czy nie.
     */
    protected boolean escapeAngleBrackets = false;

    /**
     * Znacznik okrelajcy, czy przed nazwami obiektw i funkcji JavaScript stanowicymi cz
     * dania zostanie wstawiony znak \ czy nie.
     */
    protected boolean escapeJavaScript = false;

    /**
     * Mapowanie zastpowania (pasujce wyraenie regularne, podstawienie),
     * ktre suy do zastpowania pojedynczych (') i podwjnych (") znakw
     * cudzysowu odpowiednikami ze znakiem \ na pocztku (nie mona wykorzysta
     * ich do niecnych celw).
     */
    protected HashMap<String, String> quotesHashMap =
        new HashMap<String, String>( );

    /**
     * Mapowanie zastpowania (pasujce wyraenie regularne, podstawienie),
     * ktre suy do zastpowania nawiasw <> odpowiednikami ze znakiem \ 
     * na pocztku (nie mona wykorzysta ich do niecnych celw).
     */
    protected HashMap<String, String> angleBracketsHashMap =
        new HashMap<String, String>( );

    /**
     * Mapowanie zastpowania (pasujce wyraenie regularne, podstawienie),
     * ktre suy do zastpowania potencjalnie niebezpiecznych wywoa funkcji JavaScript 
     * odpowiednikami ze znakiem \ na pocztku (nie mona wykorzysta ich do niecnych celw).
     */
    protected HashMap<String, String> javaScriptHashMap =
        new HashMap<String, String>( );

    /**
     * Mapa wyrae regularnych uywanych do filtrowania parametrw. Kluczem
     * jest szukane wyraenie regularne String, a wartoci wyraenie regularne
     * String suce do modyfikowania parametru, jeli zostanie znaleziony
     * acuch String.
     */
    protected HashMap<String, String> parameterEscapes =
        new HashMap<String, String>( );

    // ------------------------------------------------- Konstruktory

    /**
     * Konstruowanie nowego egzemplarza klasy z domylnymi wartociami waciwoci.
     */
    public BadInputValve( ) {

        super( );

        // Uycie map wyrae regularnych ze znakami \.
        quotesHashMap.put("\"", "&quot;");
        quotesHashMap.put("\'", "&#39;");
        quotesHashMap.put("`", "&#96;");
        angleBracketsHashMap.put("<", "&lt;");
        angleBracketsHashMap.put(">", "&gt;");
        javaScriptHashMap.put(
            "document(.*)\\.(.*)cookie", "document&#46;&#99;ookie");
        javaScriptHashMap.put("eval(\\s*)\\(", "eval&#40;");
        javaScriptHashMap.put("setTimeout(\\s*)\\(", "setTimeout$1&#40;");
        javaScriptHashMap.put("setInterval(\\s*)\\(", "setInterval$1&#40;");
        javaScriptHashMap.put("execScript(\\s*)\\(", "exexScript$1&#40;");
        javaScriptHashMap.put("(?i)javascript(?-i):", "javascript&#58;");
        log.info("BadInputValve instantiated.");

    }

    // --------------------------------------------------- Waciwoci

    /**
     * Pobiera znacznik, ktry okrela, czy element Valve umieci znak \
     * przed wszelkimi znakami cudzysowu (zarwno podwjnego, jak i pojedynczego)
     * stanowicymi cz dania, ktre ma by przetworzone.
     */
    public boolean getEscapeQuotes( ) {

        return escapeQuotes;

    }

    /**
     * Ustawia znacznik, ktry okrela, czy element Valve umieci znak \
     * przed wszelkimi znakami cudzysowu (zarwno podwjnego, jak i pojedynczego)
     * stanowicymi cz dania, ktre ma by przetworzone.
     *
     * @param escapeQuotes
     */
    public void setEscapeQuotes(boolean escapeQuotes) {

        this.escapeQuotes = escapeQuotes;
        if (escapeQuotes) {
            // Wstawienie znaku \ dla wszystkich cudzysoww.
            parameterEscapes.putAll(quotesHashMap);
        }

    }

    /**
     * Pobiera znacznik, ktry okrela, czy element Valve umieci znak \
     * przed wszelkimi nawiasami <> stanowicymi cz dania, ktre ma by przetworzone.
     */
    public boolean getEscapeAngleBrackets( ) {
        return escapeAngleBrackets;

    }

    /**
     * Ustawia znacznik, ktry okrela, czy element Valve umieci znak \
     * przed wszelkimi nawiasami <> stanowicymi cz dania, ktre ma by przetworzone.
     *
     * @param escapeAngleBrackets
     */
    public void setEscapeAngleBrackets(boolean escapeAngleBrackets) {

        this.escapeAngleBrackets = escapeAngleBrackets;
        if (escapeAngleBrackets) {
            // Wstawienie znaku \ przed wszystkimi nawiasami <>.
            parameterEscapes.putAll(angleBracketsHashMap);
        }

    }

    /**
     * Pobiera znacznik, ktry okrela, czy element Valve umieci znak \
     * przed wszelkimi potencjalnie niebezpiecznymi odwoaniami do funkcji
     * i obiektw JavaScript stanowicych cz dania, ktre ma by przetworzone.
     */
    public boolean getEscapeJavaScript( ) {

        return escapeJavaScript;

    }

    /**
     * Ustawia znacznik, ktry okrela, czy element Valve umieci znak \
     * przed wszelkimi potencjalnie niebezpiecznymi odwoaniami do funkcji
     * i obiektw JavaScript stanowicych cz dania, ktre ma by przetworzone.
     *
     * @param escapeJavaScript
     */
    public void setEscapeJavaScript(boolean escapeJavaScript) {

        this.escapeJavaScript = escapeJavaScript;
        if (escapeJavaScript) {
            // Wstawienie znaku \ przed potencjalnie niebezpiecznymi wywoaniami metod JavaScript.
            parameterEscapes.putAll(javaScriptHashMap);
        }

    }
    
    /**
     * Wywietlany opis tej implementacji elementu Valve.
     */
    public String getInfo( ) {

        return info;

    }

    // ----------------------------------------------- Publiczne metody

    /**
     * Porzdkuje parametry dania zanim dane wejciowe uytkownika ze zymi
     * zamiarami trafi do aplikacji WWW.
     *
     * @Parametr request; danie serwletowe do przetworzenia
     * @Parametr response; Odpowied serwletowa do utworzenia
     *
     * @Wyjtek IOException, jeli wystpi bd wejcia/wyjcia
     * @Wyjtek ServletException, jeli wystpi bd serwleta
     */
    @Override
    public void invoke(Request request, Response response)
        throws IOException, ServletException {

        // Pominicie filtrowania dla da i odpowiedzi innych ni HTTP.
        if (!(request instanceof HttpServletRequest) ||
            !(response instanceof HttpServletResponse)) {
            getNext( ).invoke(request, response);
            return;
        }

        // Przepuszczanie da tylko na podstawie regu zezwalajcych i odrzucajcych.
        if (processAllowsAndDenies(request, response)) {

            // Filtrowanie danych wejciowych pod ktem potencjalnie niebezpiecznego
            // kodu JavaScript, dziki czemu przed przetworzeniem przez Tomcata
            // dania zostan z niego usunite dane wprowadzone przez uytkownika
            // ze zymi zamiarami.
            filterParameters(request);

            // Przetworzenie dania.
            getNext( ).invoke(request, response);
        }

    }

    /**
     * Uywa funkcji elementu RequestFilterValve w celu zablokowania da,
     * ktre w nazwach i wartociach parametrw zawieraj niedozwolone
     * wzorce acuchw.
     *
     * @Parametr request; danie serwletowe do przetworzenia
     * @Parametr response; Odpowied serwletowa do utworzenia
     *
     * @Wyjtek IOException, jeli wystpi bd wejcia/wyjcia
     * @Wyjtek ServletException, jeli wystpi bd serwleta
     *
     * @Zwraca false, jeli danie jest niedozwolone; w przeciwnym razie true.
     */
    public boolean processAllowsAndDenies(Request request, Response response)
        throws IOException, ServletException {

        ParameterMap paramMap =
            (ParameterMap) ((HttpServletRequest) request).getParameterMap( );
        // Wykonanie ptli dla listy parametrw.
        Iterator y = paramMap.keySet().iterator( );
        while (y.hasNext( )) {
            String name = (String) y.next( );
            String[] values = ((HttpServletRequest)
                request).getParameterValues(name);

            // Sprawdzenie, czy nazwa zawiera niedozwolony wzorzec.
            if (!checkAllowsAndDenies(name, response)) {
                return false;
            }

            // Sprawdzenie dla wzorca wartoci parametru.
            if (values != null) {
                for (int i = 0; i < values.length; i++) {
                    String value = values[i];
                    if (!checkAllowsAndDenies(value, response)) {
                        return false;
                    }
                }
            }
        }

        // aden parametr nie spowodowa odrzucenia.  danie powinno by dalej 
        // przetwarzane.
        return true;
    }

    /**
     * Przeprowadzenie skonfigurowanego dla tego elementu Valve filtrowania,
     * polegajcego na dopasowywaniu okrelonej waciwoci dania. Jeli danie
     * moe by kontynuowane, metoda zwrci warto true.  W przeciwnym razie
     * metoda wyle stron odpowiedzi z bdem Forbidden i zwrci warto false.
     *
     * <br><br>
     *
     * Metoda intensywnie korzysta z RequestFilterValve.process( ).
     * Metoda rni si tym, e zwraca dane typu logicznego i nie wywouje
     * getNext( ).invoke(request, response).
     *
     * @Parametr property; Waciwo dania uywana podczas filtrowania
     * @Parametr response; Odpowied serwletowa do przetworzenia
     *
     * @Wyjtek IOException, jeli wystpi bd wejcia/wyjcia
     * @Wyjtek ServletException, jeli wystpi bd serwleta
     *
     * @Zwraca true, jeli w dalszym cigu danie moe by przetwarzane
     */
    public boolean checkAllowsAndDenies(String property, Response response)
        throws IOException, ServletException {

        // Jeli nie byo adnych regu odrzucajcych i zezwalajcych, danie bdzie przetwarzane.
        if (denies.length == 0 && allows.length == 0) {
            return true;
        }

        // Sprawdzenie wzorcw odrzucajcych (jeli istniej)
        for (int i = 0; i < denies.length; i++) {
            Matcher m = denies[i].matcher(property);
            if (m.find( )) {
                ServletResponse sres = response.getResponse( );
                if (sres instanceof HttpServletResponse) {
                    HttpServletResponse hres = (HttpServletResponse) sres;
                    hres.sendError(HttpServletResponse.SC_FORBIDDEN);
                    return false;
                }
            }
        }

        // Sprawdzenie wzorcw zezwalajcych (jeli istniej)
        for (int i = 0; i < allows.length; i++) {
            Matcher m = allows[i].matcher(property);
            if (m.find( )) {
                return true;
            }
        }

        // Zezwolenie, jeli okrelono reguy odrzucajce, lecz nie zezwalajce
        if (denies.length > 0 && allows.length == 0) {
            return true;
        }

        // W przeciwnym razie nastpi odrzucenie dania.
        ServletResponse sres = response.getResponse( );
        if (sres instanceof HttpServletResponse) {
            HttpServletResponse hres = (HttpServletResponse) sres;
            hres.sendError(HttpServletResponse.SC_FORBIDDEN);
        }
        return false;
    }

    /**
     * Filtrowanie wszystkich istniejcych parametrw pod ktem potencjalnie
     * niebezpiecznej zawartoci i umieszczanie na jej pocztku znaku \.
     *
     * @Parametr request; danie zawierajce parametry.
     */
    public void filterParameters(Request request) {

        ParameterMap paramMap =
            (ParameterMap) ((HttpServletRequest) request).getParameterMap( );
        // Odblokowanie mapy parametrw w celu umoliwienia modyfikowania ich.
        paramMap.setLocked(false);

        // Wykonanie ptli dla kadego wzorca zastpujcego.
        Iterator escapesIterator = parameterEscapes.keySet().iterator( );
        while (escapesIterator.hasNext( )) {
            String patternString = (String) escapesIterator.next( );
            Pattern pattern = Pattern.compile(patternString);

            // Wykonanie ptli dla listy parametrw.
            @SuppressWarnings("unchecked")
            String[] paramNames =
                (String[]) paramMap.keySet( ).toArray(STRING_ARRAY);
            for (int i = 0; i < paramNames.length; i++) {
                String name = paramNames[i];
                String[] values = ((HttpServletRequest)
                    request).getParameterValues(name);
                // Sprawdzenie, czy nazwa zawiera wzorzec.
                boolean nameMatch;
                Matcher matcher = pattern.matcher(name);
                nameMatch = matcher.find( );
                if (nameMatch) {
                    // Poniewa nazwa parametru jest zgodna z wzorcem,
                    // zmienilimy nazw, dodajc parametr o nowej nazwie
                    // i usuwajc  star nazw.
                    String newName = matcher.replaceAll(
                        (String) parameterEscapes.get(patternString));
                    request.addParameter(newName, values);
                    paramMap.remove(name);
                    log.warn("Nazwa parametru " + name +
                        " pasuje do wzorca \"" + patternString +
                        "\".  Zdalny adres: " +
                        ((HttpServletRequest) request).getRemoteAddr( ));
                }
                // Sprawdzenie dla wzorca wartoci parametru.
                if (values != null) {
                for (int j = 0; j < values.length; j++) {
                        String value = values[j];
                        boolean valueMatch;
                        matcher = pattern.matcher(value);
                        valueMatch = matcher.find( );
                        if (valueMatch) {
                            // Poniewa warto okazaa si zgodna, zmodyfikowalimy j,
                            // a nastpnie umiecilimy w tablicy.
                            String newValue;
                            newValue = matcher.replaceAll((String)
                                parameterEscapes.get(patternString));
                            values[j] = newValue;
                            log.warn("Warto parametru \"" + name +
                                "\"postaci \"" + value +
                                "\" pasuje do wzorca \"" +
                                patternString + "\".  Zdalny adres: " +
                                ((HttpServletRequest)
                                    request).getRemoteAddr( ));
                        }
                    }
                }
            }
        }
        // Sprawdzenie, czy po zakoczeniu mapa parametrw zostaa ponownie zablokowana.
        paramMap.setLocked(true);

    }

    /**
     * Zwrcenie tekstowej reprezentacji obiektu.
     */
    @Override
    public String toString( ) {
 
       return "BadInputValve";
    }
}