//: concurrency/CarBuilder.java
// Skomplikowany przykad wspdziaania zada.
import java.util.concurrent.*;
import java.util.*;
import static net.mindview.util.Print.*;

class Car {
  private final int id;
  private boolean
    engine = false, driveTrain = false, wheels = false;
  public Car(int idn)  { id = idn; }
  // Puste obiekty Car:
  public Car()  { id = -1; }
  public synchronized int getId() { return id; }
  public synchronized void addEngine() { engine = true; }
  public synchronized void addDriveTrain() {
    driveTrain = true;
  }
  public synchronized void addWheels() { wheels = true; }
  public synchronized String toString() {
    return "Auto " + id + " [" + " silnik: " + engine
      + " napd: " + driveTrain
      + " koa: " + wheels + " ]";
  }
}

class CarQueue extends LinkedBlockingQueue<Car> {}

class ChassisBuilder implements Runnable {
  private CarQueue carQueue;
  private int counter = 0;
  public ChassisBuilder(CarQueue cq) { carQueue = cq; }
  public void run() {
    try {
      while(!Thread.interrupted()) {
        TimeUnit.MILLISECONDS.sleep(500);
        // Skadanie nadwozia:
        Car c = new Car(counter++);
        print("Zadanie ChassisBuilder zmontowao karoseri " + c);
        // Wstawianie do kolejki
        carQueue.put(c);
      }
    } catch(InterruptedException e) {
      print("Przerwanie: ChassisBuilder");
    }
    print("Zadanie ChassisBuilder wyczone");
  }
}

class Assembler implements Runnable {
  private CarQueue chassisQueue, finishingQueue;
  private Car car;
  private CyclicBarrier barrier = new CyclicBarrier(4);
  private RobotPool robotPool;
  public Assembler(CarQueue cq, CarQueue fq, RobotPool rp){
    chassisQueue = cq;
    finishingQueue = fq;
    robotPool = rp;
  }
  public Car car() { return car; }
  public CyclicBarrier barrier() { return barrier; }
  public void run() {
    try {
      while(!Thread.interrupted()) {
        // Blokowanie od momentu zmontowania karoserii:
        car = chassisQueue.take();
        // Uruchomienie robotw:
        robotPool.hire(EngineRobot.class, this);
        robotPool.hire(DriveTrainRobot.class, this);
        robotPool.hire(WheelRobot.class, this);
        barrier.await(); // Oczekiwanie na zakoczenie montau
        // Umieszczenie gotowego auta w kolejce finishingQueue
        finishingQueue.put(car);
      }
    } catch(InterruptedException e) {
      print("Zadanie Assembler zakoczone przerwaniem");
    } catch(BrokenBarrierException e) {
      // O tym wyjtku lepiej wiedzie
      throw new RuntimeException(e);
    }
    print("Zadanie Assembler wyczone");
  }
}

class Reporter implements Runnable {
  private CarQueue carQueue;
  public Reporter(CarQueue cq) { carQueue = cq; }
  public void run() {
    try {
      while(!Thread.interrupted()) {
        print(carQueue.take());
      }
    } catch(InterruptedException e) {
      print("Zadanie Reporter zakoczone przerwaniem");
    }
    print("Zadanie Reporter wyczone");
  }
}

abstract class Robot implements Runnable {
  private RobotPool pool;
  public Robot(RobotPool p) { pool = p; }
  protected Assembler assembler;
  public Robot assignAssembler(Assembler assembler) {
    this.assembler = assembler;
    return this;
  }
  private boolean engage = false;
  public synchronized void engage() {
    engage = true;
    notifyAll();
  }
  // Cz metody run() rnicujca dziaanie robotw:
  abstract protected void performService();
  public void run() {
    try {
      powerDown(); // Oczekiwanie do nastpnego uruchomienia
      while(!Thread.interrupted()) {
        performService();
        assembler.barrier().await(); // Synchronizacja
        // Gotowe...
        powerDown();
      }
    } catch(InterruptedException e) {
      print("Zadanie " + this + " zakoczone przerwaniem");
    } catch(BrokenBarrierException e) {
      // O tym wyjtku warto wiedzie
      throw new RuntimeException(e);
    }
    print("Zadanie " + this + " wyczone");
  }
  private synchronized void
  powerDown() throws InterruptedException {
    engage = false;
    assembler = null; // Odczenie od linii montaowej
    // Umieszczamy si w kolejce robotw gotowych do pracy:
    pool.release(this);
    while(engage == false)  // Zasilanie wyczone
      wait();
  }
  public String toString() { return getClass().getName(); }
}

class EngineRobot extends Robot {
  public EngineRobot(RobotPool pool) { super(pool); }
  protected void performService() {
    print(this + ": instalacja silnika");
    assembler.car().addEngine();
  }
}

class DriveTrainRobot extends Robot {
  public DriveTrainRobot(RobotPool pool) { super(pool); }
  protected void performService() {
    print(this + ": instalacja przeniesienia napdu");
    assembler.car().addDriveTrain();
  }
}

class WheelRobot extends Robot {
  public WheelRobot(RobotPool pool) { super(pool); }
  protected void performService() {
    print(this + ": instalacja k");
    assembler.car().addWheels();
  }
}

class RobotPool {
  // Dyskretne zapobieganie dublowaniu wpisw:
  private Set<Robot> pool = new HashSet<Robot>();
  public synchronized void add(Robot r) {
    pool.add(r);
    notifyAll();
  }
  public synchronized void
  hire(Class<? extends Robot> robotType, Assembler d)
  throws InterruptedException {
    for(Robot r : pool)
      if(r.getClass().equals(robotType)) {
        pool.remove(r);
        r.assignAssembler(d);
        r.engage(); // Przekazanie do linii
        return;
      }
    wait(); // Brak dostpnych robotw
    hire(robotType, d); // Jeszcze jedna prba (rekurencyjna)
  }
  public synchronized void release(Robot r) { add(r); }
}

public class CarBuilder {
  public static void main(String[] args) throws Exception {
    CarQueue chassisQueue = new CarQueue(),
             finishingQueue = new CarQueue();
    ExecutorService exec = Executors.newCachedThreadPool();
    RobotPool robotPool = new RobotPool();
    exec.execute(new EngineRobot(robotPool));
    exec.execute(new DriveTrainRobot(robotPool));
    exec.execute(new WheelRobot(robotPool));
    exec.execute(new Assembler(
      chassisQueue, finishingQueue, robotPool));
    exec.execute(new Reporter(finishingQueue));
    // Uruchomienie caoci przez monta karoserii:
    exec.execute(new ChassisBuilder(chassisQueue));
    TimeUnit.SECONDS.sleep(7);
    exec.shutdownNow();
  }
} /* (Execute to see output) *///:~
