//: concurrency/restaurant2/RestaurantWithQueues.java
// {Args: 5}
package concurrency.restaurant2;
import enumerated.menu.*;
import java.util.concurrent.*;
import java.util.*;
import static net.mindview.util.Print.*;

// Obiekt wrczany kelnerowi, ktry przekazuje go kucharzowi:
class Order { // (obiekt transferu danych)
  private static int counter = 0;
  private final int id = counter++;
  private final Customer customer;
  private final WaitPerson waitPerson;
  private final Food food;
  public Order(Customer cust, WaitPerson wp, Food f) {
    customer = cust;
    waitPerson = wp;
    food = f;
  }
  public Food item() { return food; }
  public Customer getCustomer() { return customer; }
  public WaitPerson getWaitPerson() { return waitPerson; }
  public String toString() {
    return "Zamwienie: " + id + " na: " + food +
      " dla: " + customer +
      " obsugiwane przez: " + waitPerson;
  }
}

// A to otrzymujemy od kucharza:
class Plate {
  private final Order order;
  private final Food food;
  public Plate(Order ord, Food f) {
    order = ord;
    food = f;
  }
  public Order getOrder() { return order; }
  public Food getFood() { return food; }
  public String toString() { return food.toString(); }
}

class Customer implements Runnable {
  private static int counter = 0;
  private final int id = counter++;
  private final WaitPerson waitPerson;
  // Jedno danie na raz:
  private SynchronousQueue<Plate> placeSetting =
    new SynchronousQueue<Plate>();
  public Customer(WaitPerson w) { waitPerson = w; }
  public void
  deliver(Plate p) throws InterruptedException {
    // Wywoanie blokowane, jeli klient wci
    // spoywa poprzednie danie:
    placeSetting.put(p);
  }
  public void run() {
    for(Course course : Course.values()) {
      Food food = course.randomSelection();
      try {
        waitPerson.placeOrder(this, food);
        // Blokowanie do momentu otrzymania dania:
        print(this + "spoywa " + placeSetting.take());
      } catch(InterruptedException e) {
        print("przerwano oczekiwanie " + this + " na " +
          course);
        break;
      }
    }
    print(this + "zakoczy obiad, wychodzi");
  }
  public String toString() {
    return "Klient " + id + " ";
  }
}

class WaitPerson implements Runnable {
  private static int counter = 0;
  private final int id = counter++;
  private final Restaurant restaurant;
  BlockingQueue<Plate> filledOrders =
    new LinkedBlockingQueue<Plate>();
  public WaitPerson(Restaurant rest) { restaurant = rest; }
  public void placeOrder(Customer cust, Food food) {
    try {
      // Nie powinno blokowa, bo to kolejka
      // LinkedBlockingQueue, bez ograniczenia pojemnoci:
      restaurant.orders.put(new Order(cust, this, food));
    } catch(InterruptedException e) {
      print(this + " przerwanie w metodzie placeOrder");
    }
  }
  public void run() {
    try {
      while(!Thread.interrupted()) {
        // Blokowanie do momentu gotowoci dania
        Plate plate = filledOrders.take();
        print(this + "odebra " + plate +
          " dla " +
          plate.getOrder().getCustomer());
        plate.getOrder().getCustomer().deliver(plate);
      }
    } catch(InterruptedException e) {
      print(this + " przerwany");
    }
    print(this + " po subie");
  }
  public String toString() {
    return "Kelner " + id + " ";
  }
}

class Chef implements Runnable {
  private static int counter = 0;
  private final int id = counter++;
  private final Restaurant restaurant;
  private static Random rand = new Random(47);
  public Chef(Restaurant rest) { restaurant = rest; }
  public void run() {
    try {
      while(!Thread.interrupted()) {
        // Blokowanie do momentu pojawienia si zamwienia:
        Order order = restaurant.orders.take();
        Food requestedItem = order.item();
        // Czas na przygotowanie zamwienia:
        TimeUnit.MILLISECONDS.sleep(rand.nextInt(500));
        Plate plate = new Plate(order, requestedItem);
        order.getWaitPerson().filledOrders.put(plate);
      }
    } catch(InterruptedException e) {
      print(this + " przerwany");
    }
    print(this + " po subie");
  }
  public String toString() { return "Kucharz " + id + " "; }
}

class Restaurant implements Runnable {
  private List<WaitPerson> waitPersons =
    new ArrayList<WaitPerson>();
  private List<Chef> chefs = new ArrayList<Chef>();
  private ExecutorService exec;
  private static Random rand = new Random(47);
  BlockingQueue<Order>
    orders = new LinkedBlockingQueue<Order>();
  public Restaurant(ExecutorService e, int nWaitPersons,
    int nChefs) {
    exec = e;
    for(int i = 0; i < nWaitPersons; i++) {
      WaitPerson waitPerson = new WaitPerson(this);
      waitPersons.add(waitPerson);
      exec.execute(waitPerson);
    }
    for(int i = 0; i < nChefs; i++) {
      Chef chef = new Chef(this);
      chefs.add(chef);
      exec.execute(chef);
    }
  }
  public void run() {
    try {
      while(!Thread.interrupted()) {
        // Przybywa nowy klient; przypisa mu kelnera (WaitPerson):
        WaitPerson wp = waitPersons.get(
          rand.nextInt(waitPersons.size()));
        Customer c = new Customer(wp);
        exec.execute(c);
        TimeUnit.MILLISECONDS.sleep(100);
      }
    } catch(InterruptedException e) {
      print("Przerwano zadanie Restaurant");
    }
    print("Zamykanie restauracji");
  }
}

public class RestaurantWithQueues {
  public static void main(String[] args) throws Exception {
    ExecutorService exec = Executors.newCachedThreadPool();
    Restaurant restaurant = new Restaurant(exec, 5, 2);
    exec.execute(restaurant);
    if(args.length > 0) // Argument opcjonalny
      TimeUnit.SECONDS.sleep(new Integer(args[0]));
    else {
      print("Aby zakoczy, nacinij 'Enter'");
      System.in.read();
    }
    exec.shutdownNow();
  }
} /* Output: (Sample)
Aby zakoczy, nacinij 'Enter'
Kelner 0 odebra SPRING_ROLLS dla Klient 1 
Klient 1 spoywa SPRING_ROLLS
Kelner 3 odebra SPRING_ROLLS dla Klient 0 
Klient 0 spoywa SPRING_ROLLS
Kelner 0 odebra BURRITO dla Klient 1 
Klient 1 spoywa BURRITO
Kelner 3 odebra SPRING_ROLLS dla Klient 2 
Klient 2 spoywa SPRING_ROLLS
Kelner 3 odebra BURRITO dla Klient 0 
Klient 0 spoywa BURRITO
Kelner 1 odebra SPRING_ROLLS dla Klient 3 
Klient 3 spoywa SPRING_ROLLS
...
*///:~
