//: concurrency/BankTellerSimulation.java
// Kolejki i wielowtkowo.
// {Args: 5}
import java.util.concurrent.*;
import java.util.*;

// Obiekty niemodyfikowalne, a wic nie wymagajce synchronizacji:
class Customer {
  private final int serviceTime;
  public Customer(int tm) { serviceTime = tm; }
  public int getServiceTime() { return serviceTime; }
  public String toString() {
    return "[" + serviceTime + "]";
  }
}

// Klasa kolejki klientw; niech wypisuje si sama na wyjciu:
class CustomerLine extends ArrayBlockingQueue<Customer> {
  public CustomerLine(int maxLineSize) {
    super(maxLineSize);
  }
  public String toString() {
    if(this.size() == 0)
      return "[Pusto]";
    StringBuilder result = new StringBuilder();
    for(Customer customer : this)
      result.append(customer);
    return result.toString();
  }
}

// Losowe dodawanie klientw do kolejki:
class CustomerGenerator implements Runnable {
  private CustomerLine customers;
  private static Random rand = new Random(47);
  public CustomerGenerator(CustomerLine cq) {
    customers = cq;
  }
  public void run() {
    try {
      while(!Thread.interrupted()) {
        TimeUnit.MILLISECONDS.sleep(rand.nextInt(300));
        customers.put(new Customer(rand.nextInt(1000)));
      }
    } catch(InterruptedException e) {
      System.out.println("Przerwano zadanie CustomerGenerator");
    }
    System.out.println("Zakoczono zadanie CustomerGenerator");
  }
}

class Teller implements Runnable, Comparable<Teller> {
  private static int counter = 0;
  private final int id = counter++;
  // Liczba klientw obsuonych w czasie zmiany:
  private int customersServed = 0;
  private CustomerLine customers;
  private boolean servingCustomerLine = true;
  public Teller(CustomerLine cq) { customers = cq; }
  public void run() {
    try {
      while(!Thread.interrupted()) {
        Customer customer = customers.take();
        TimeUnit.MILLISECONDS.sleep(
          customer.getServiceTime());
        synchronized(this) {
          customersServed++;
          while(!servingCustomerLine)
            wait();
        }
      }
    } catch(InterruptedException e) {
      System.out.println("Przerwano zadanie " + this);
    }
    System.out.println("Zakoczono zadanie " + this);
  }
  public synchronized void doSomethingElse() {
    customersServed = 0;
    servingCustomerLine = false;
  }
  public synchronized void serveCustomerLine() {
    assert !servingCustomerLine:"ju obsuguje: " + this;
    servingCustomerLine = true;
    notifyAll();
  }
  public String toString() { return "Kasjer (Teller) " + id + " "; }
  public String shortString() { return "K" + id; }
  // Na uytek kolejki priorytetowej:
  public synchronized int compareTo(Teller other) {
    return customersServed < other.customersServed ? -1 :
      (customersServed == other.customersServed ? 0 : 1);
  }
}

class TellerManager implements Runnable {
  private ExecutorService exec;
  private CustomerLine customers;
  private PriorityQueue<Teller> workingTellers =
    new PriorityQueue<Teller>();
  private Queue<Teller> tellersDoingOtherThings =
    new LinkedList<Teller>();
  private int adjustmentPeriod;
  private static Random rand = new Random(47);
  public TellerManager(ExecutorService e,
    CustomerLine customers, int adjustmentPeriod) {
    exec = e;
    this.customers = customers;
    this.adjustmentPeriod = adjustmentPeriod;
    // Zaczynamy od pojedynczego kasjera:
    Teller teller = new Teller(customers);
    exec.execute(teller);
    workingTellers.add(teller);
  }
  public void adjustTellerNumber() {
    // To waciwy system sterowania. Przez dostosowywanie parametrw
    // mona ujawni niestabilnoci mechanizmu kontrolnego.
    // Jeli kolejka jest zbyt duga, otwieramy nastpne okienko kasowe:
    if(customers.size() / workingTellers.size() > 2) {
        // Kiedy kasjerzy maj przerw, albo s zajci czym
        // innym, przywoujemy jednego z nich z zaplecza sali:
        if(tellersDoingOtherThings.size() > 0) {
          Teller teller = tellersDoingOtherThings.remove();
          teller.serveCustomerLine();
          workingTellers.offer(teller);
          return;
        }
      // Albo zatrudniamy (konstruujemy) nowego
      Teller teller = new Teller(customers);
      exec.execute(teller);
      workingTellers.add(teller);
      return;
    }
    // Kiedy kolejka si skrci, zamykamy okienko:
    if(workingTellers.size() > 1 &&
      customers.size() / workingTellers.size() < 2)
        reassignOneTeller();
    // Kiedy kolejka zniknie, wystarczy jeden kasjer:
    if(customers.size() == 0)
      while(workingTellers.size() > 1)
        reassignOneTeller();
  }
  // Zadanie kasjerowi innej pracy albo zwolnienie na przerw:
  private void reassignOneTeller() {
    Teller teller = workingTellers.poll();
    teller.doSomethingElse();
    tellersDoingOtherThings.offer(teller);
  }
  public void run() {
    try {
      while(!Thread.interrupted()) {
        TimeUnit.MILLISECONDS.sleep(adjustmentPeriod);
        adjustTellerNumber();
        System.out.print(customers + " { ");
        for(Teller teller : workingTellers)
          System.out.print(teller.shortString() + " ");
        System.out.println("}");
      }
    } catch(InterruptedException e) {
      System.out.println("Przerwano zadanie " + this);
    }
    System.out.println("Zakoczono zadanie " + this);
  }
  public String toString() { return "Zadanie TellerManager "; }
}

public class BankTellerSimulation {
  static final int MAX_LINE_SIZE = 50;
  static final int ADJUSTMENT_PERIOD = 1000;
  public static void main(String[] args) throws Exception {
    ExecutorService exec = Executors.newCachedThreadPool();
    // Jeli kolejka jest zbyt duga, klienci si zniechcaj:
    CustomerLine customers =
      new CustomerLine(MAX_LINE_SIZE);
    exec.execute(new CustomerGenerator(customers));
    // Kierownik sali przyjmuje i odwouje kasjerw w miar potrzeb:
    exec.execute(new TellerManager(
      exec, customers, ADJUSTMENT_PERIOD));
    if(args.length > 0) // Argument opcjonalny
      TimeUnit.SECONDS.sleep(new Integer(args[0]));
    else {
      System.out.println("Aby zakoczy, nacinij 'Enter'");
      System.in.read();
    }
    exec.shutdownNow();
  }
} /* Output: (Sample)
[429][200][207] { K0 K1 }
[861][258][140][322] { K0 K1 }
[575][342][804][826][896][984] { K0 K1 K2 }
[984][810][141][12][689][992][976][368][395][354] { K0 K1 K2 K3 }
Przerwano zadanie Kasjer (Teller) 2 
Zakoczono zadanie Kasjer (Teller) 2 
Przerwano zadanie Kasjer (Teller) 1 
Zakoczono zadanie Kasjer (Teller) 1 
Przerwano zadanie Zadanie TellerManager 
Zakoczono zadanie Zadanie TellerManager 
Przerwano zadanie Kasjer (Teller) 3 
Zakoczono zadanie Kasjer (Teller) 3 
Przerwano zadanie Kasjer (Teller) 0 
Zakoczono zadanie Kasjer (Teller) 0 
Przerwano zadanie CustomerGenerator
Zakoczono zadanie CustomerGenerator
*///:~
