using System.Collections.Generic;
using System.Linq;
using SpeakerMeet.DTO;
using SpeakerMeet.Services.Interfaces;
using SpeakerMeet.Services.Tests.Factories;
using SpeakerMeet.Services.Tests.Fakes;
using Xunit;

namespace SpeakerMeet.Services.Tests.SpeakerServiceTests
{
    [Trait("Category", "SpeakerService")]
    public class Search
    {
        private readonly SpeakerService _speakerService;

        public Search()
        {
            var fakeRepository = new FakeRepository();
            SpeakerFactory.Create(fakeRepository);
            SpeakerFactory.Create(fakeRepository, name: "Jan");
            SpeakerFactory.Create(fakeRepository, name: "Janina");
            SpeakerFactory.Create(fakeRepository, name: "Bartosz");


            _speakerService = new SpeakerService(fakeRepository);
        }

        [Fact]
        public void ItHasSearchMethod()
        {
            _speakerService.Search("test");
        }

        [Fact]
        public void ItReturnsCollectionOfSpeakers()
        {
            // Arrange
            // Act
            var speakers = _speakerService.Search("test");

            // Assert
            Assert.NotNull(speakers);
            Assert.IsAssignableFrom<IEnumerable<Speaker>>(speakers);
        }

        [Fact]
        public void ItImplementsISpeakerService()
        {
            var speakerService = _speakerService;

            Assert.IsAssignableFrom<ISpeakerService>(speakerService);
        }

        [Fact]
        public void GivenExactMatchThenOneSpeakerInCollection()
        {
            // Arrange
            // Act
            var result = _speakerService.Search("Janusz");

            // Assert
            var speakers = result.ToList();
            Assert.Equal(1, speakers.Count);
            Assert.Equal("Janusz", speakers[0].Name);
        }

        [Theory]
        [InlineData("Janusz")]
        [InlineData("janusz")]
        [InlineData("JaNuSz")]
        public void GivenCaseInsensitveMatchThenSpeakerInCollection(string searchString)
        {
            // Arrange
            // Act
            var result = _speakerService.Search(searchString);

            // Assert
            var speakers = result.ToList();
            Assert.Equal(1, speakers.Count);
            Assert.Equal("Janusz", speakers[0].Name);
        }

        [Fact]
        public void GivenNoMatchThenEmptyCollection()
        {
            // Arrange
            // Act
            var result = _speakerService.Search("ZZZ");

            // Assert
            var speakers = result.ToList();
            Assert.Equal(0, speakers.Count);
        }

        [Fact]
        public void Given3MatchThenCollectionWith3Speakers()
        {
            // Arrange
            // Act
            var result = _speakerService.Search("jan");

            // Assert
            var speakers = result.ToList();
            Assert.Equal(3, speakers.Count);
            Assert.True(speakers.Any(s => s.Name == "Jan"));
            Assert.True(speakers.Any(s => s.Name == "Janusz"));
            Assert.True(speakers.Any(s => s.Name == "Janina"));
        }
    }
}
