import uvicorn
from fastapi import File, UploadFile, Request, Response, Form, FastAPI
from web import explorer, creature, game
from fastapi.responses import FileResponse
from pathlib import Path
from typing import Generator
from fastapi.responses import StreamingResponse
from fastapi.staticfiles import StaticFiles
from fastapi.templating import Jinja2Templates
import plotly.express as px
from collections import Counter
from service.creature import get_all
from fake import creature as service
import plotly.express as px
import country_converter as coco

app = FastAPI()

# Katalog zawierający main.py
top = Path(__file__).resolve().parent
template_obj = Jinja2Templates(directory=f"{top}/template")

# Zaimportowanie małej listy odkrywców i stworzeń
from fake.creature import _creatures as fake_creatures
from fake.explorer import _explorers as fake_explorers

# (insert these lines in web/creature.py)

@app.get("/map")
def map():
    creatures = service.get_all()    
    iso2_codes = set(creature.country for creature in creatures)
    iso3_codes = coco.convert(names=iso2_codes, to="ISO3")
    fig = px.choropleth(
        locationmode="ISO-3",
        locations=iso3_codes)
        
    fig_bytes = fig.to_image(format="png")
    return Response(content=fig_bytes, media_type="image/png")


@app.get("/plot")
def plot():
    creatures = get_all()
    letters = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    counts = Counter(creature.name[0] for creature in creatures)
    y = { letter: counts.get(letter, 0) for letter in letters }
    fig = px.histogram(x=list(letters), y=y, title="Creature Names",
        labels={"x": "Initial", "y": "Initial"})
    fig_bytes = fig.to_image(format="png")
    return Response(content=fig_bytes, media_type="image/png")


@app.get("/test")
def test():
    df = px.data.iris()
    fig = px.scatter(df, x="sepal_width", y="sepal_length", color="species")
    fig_bytes = fig.to_image(format="png")
    return Response(content=fig_bytes, media_type="image/png")

@app.get("/list")
def explorer_list(request: Request):
    return template_obj.TemplateResponse("list.html",
        {"request": request,
        "explorers": fake_explorers,
        "creatures": fake_creatures})

@app.post("/who2")
def greet2(name: str = Form()):
    return f"Witaj, {name}?"

app.mount("/static",
    StaticFiles(directory=f"{top}/static", html=True),
    name="free")

def gen_file(path: str) -> Generator:
    with open(file=path, mode="rb") as file:
        yield file.read()

@app.get("/download_big/{name}")
async def download_big_file(name:str):
    gen_expr = gen_file(Path(name))
    response = StreamingResponse(
        content=gen_expr,
        status_code=200,
    )
    return response

@app.post("/big")
async def upload_big_file(big_file: UploadFile) -> str:
    return f"file size: {big_file.size}, name: {big_file.filename}"

@app.post("/small")
async def upload_small_file(small_file: bytes = File()) -> str:
    return f"file size: {len(small_file)}"

@app.get("/small/{name}")
async def download_small_file(name):
    return FileResponse(name)

app.include_router(explorer.router)
app.include_router(creature.router)
app.include_router(game.router)

if __name__ == "__main__":
    uvicorn.run("main:app", reload=True, host="0.0.0.0", port=8000)