package org.reactivedesignpatterns.chapter2.future

import scala.concurrent.Future
import scala.concurrent.ExecutionContext
import java.util.concurrent.ForkJoinPool

trait InventoryService {
  def currentInventoryInWarehouse(productSku: Long, postalCode: String): Long
  def currentInventoryOverallByWarehouse(productSku: Long): Map[String, Long]
}

class StagedFuturesForExample(inventoryService: InventoryService) {

  /**
   * Zwraca futurę z lokalnym stanem i ogólnym stanem produktu.
   * Oba zapytania są wysyłane osobno, jednak futura zwracająca
   * parę wartości nie zostanie wykonana dopóki obie wartości
   * nie zostaną zwrócone.
   * Lokalny stan jest po prostu licznikiem, a ogólny stan jest
   * mapą identyfikatora magazynu i licznika.
   */
   
  def getProductInventoryByPostalCode(productSku: Long, postalCode: String): Future[(Long, Map[String, Long])] = {
    // Import DSL wykorzystywanego jako timeout
    import scala.concurrent.duration._

    // Pula wątków i czas oczekiwania na futurę
    implicit val ec = ExecutionContext.fromExecutor(new ForkJoinPool())
    implicit val timeout = 250 milliseconds

    // Definicja futur wykonujących właściwe operacje
    val localInventoryFuture = Future {
      inventoryService.currentInventoryInWarehouse(productSku, postalCode)
    }
    val overallInventoryFutureByWarehouse = Future {
      inventoryService.currentInventoryOverallByWarehouse(productSku)
    }

    // Odczytanie wartości i zwrócenie futury z połączonym wynikiem
    for {
      local <- localInventoryFuture
      overall <- overallInventoryFutureByWarehouse
    } yield (local, overall)
  }
}