package com.reactivedesignpatterns.chapter11

import org.scalatest.WordSpec
import org.scalatest.Matchers
import com.reactivedesignpatterns.Defaults._
import scala.util.Success
import scala.concurrent.ExecutionContext
import scala.concurrent.Await
import scala.concurrent.duration._
import org.scalatest.concurrent.Eventually
import org.scalatest.BeforeAndAfterAll
import akka.actor.ActorSystem
import akka.testkit.TestProbe
import akka.actor.Actor
import akka.actor.ActorRef
import akka.actor.Props

object TranslationServiceSpec {
  import TranslationService._

  case object ExpectNominal
  case object ExpectError
  case class Unexpected(msg: Any)

  class MockV1(reporter: ActorRef) extends Actor {
    def receive = initial

    override def unhandled(msg: Any) = {
      reporter ! Unexpected(msg)
    }

    val initial: Receive = {
      case ExpectNominal => context.become(expectingNominal)
      case ExpectError   => context.become(expectingError)
    }

    val expectingNominal: Receive = {
      case TranslateV1("en:pl:How are you?", replyTo) =>
        replyTo ! "Jak się masz?"
        context.become(initial)
    }
    
    val expectingError: Receive = {
      case TranslateV1(other, replyTo) =>
        replyTo ! s"błąd: niezrozumiałe wyrażenie '$other'"
        context.become(initial)
    }
  }

  def mockV1props(reporter: ActorRef): Props = Props(new MockV1(reporter))

}

class TranslationServiceSpec extends WordSpec with Matchers with Eventually with BeforeAndAfterAll {
  import TranslationServiceSpec._

  "A TranslationService" should {

    "correctly translate from Swedish" when {

      "using SynchronousEventLoop" in {
        val tr = new TranslationService
        val input = "How are you?"
        val output = "Jak się masz?"
        val ec = SynchronousEventLoop
        tr.translate(input, ec).value.get should be(Success(output))
      }

      "using Await.result" in {
        val tr = new TranslationService
        val input = "How are you?"
        val output = "Jak się masz?"
        val future = tr.translate(input)
        Await.result(future, 1.second) should be(output)
      }

      "using eventually()" in {
        val tr = new TranslationService
        val input = "How are you?"
        val output = "Jak się masz?"
        val future = tr.translate(input)
        eventually {
          future.value.get should be(Success(output))
        }
      }

    }

    "not respond immediately when running asynchronously" in {
      val tr = new TranslationService
      val input = "How are you?"
      val output = "Jak się masz?"
      val ec = ExecutionContext.global
      val future = tr.translate(input, ec)
      future.value should be(None)
      Await.result(future, 1.second) should be(output)
    }

  }

  implicit val system = ActorSystem("TranslationServiceSpec")
  override def afterAll(): Unit = system.shutdown()

  "A TranslationService version adapter" should {
    import TranslationService._

    "translate requests" in {
      val v1 = TestProbe()
      val v2 = system.actorOf(propsV2(v1.ref))
      val client = TestProbe()

      // Zainicjowanie zapytania do adaptera.
      v2 ! TranslateV2("How are you?", "en", "pl", client.ref)

      // Sprawdzenie, czy adapter kontaktuje się z usługą za pomocą protokołu w wersji 1.
      val req1 = v1.expectMsgType[TranslateV1]
      req1.query should be("en:pl:How are you?")

      // Zainicjowanie odpowiedzi.
      req1.replyTo ! "Jak się masz?"

      // Sprawdzenie, czy adapter poprawnie przekształca dane.
      client.expectMsg(TranslationV2("How are you?", "Jak się masz?", "en", "pl"))

      // Sprawdzenie błędów tłumaczenia.
      v2 ! TranslateV2("How is it going?", "en", "pl", client.ref)
      val req2 = v1.expectMsgType[TranslateV1]
      // Niejawne sprawdzenie, czy nie nie miała miejsca inna komunikacja.
      req2.query should be("en:pl'How is it going?")
      req2.replyTo ! "błąd: niezrozumiałe wyrażenie 'How is it going?'"
      client.expectMsg(TranslationErrorV2("How is it going?", "en", "pl", "błąd: niezrozumiałe wyrażenie 'How is it going?'"))

      v1.expectNoMsg(1.second)
    }

    "translate requests with async error reporting" in {
      val asyncErrors = TestProbe()
      val v1 = system.actorOf(mockV1props(asyncErrors.ref))
      val v2 = system.actorOf(propsV2(v1))
      val client = TestProbe()

      // Zainicjowanie zapytania do adaptera.
      v1 ! ExpectNominal
      v2 ! TranslateV2("How are you?", "en", "pl", client.ref)

      // Sprawdzenie, czy adapter poprawnie przekształca dane.
      client.expectMsg(TranslationV2("How are you?", "Jak się masz?", "en", "pl"))

      // Nieblokujące sprawdzenie błędów asynchronicznych.
      asyncErrors.expectNoMsg(0.seconds)

      // Sprawdzenie błędów tłumaczenia.
      v1 ! ExpectError
      v2 ! TranslateV2("How is it going?", "en", "pl", client.ref)
      client.expectMsg(TranslationErrorV2("How is it going?", "en", "pl", "błąd: niezrozumiałe wyrażenie 'en:pl:How is it going?'"))

      // Końcowe sprawdzenie błędów asynchronicznych.
      asyncErrors.expectNoMsg(1.second)
    }

  }

}