/**
 * Copyright (C) 2015 Roland Kuhn <http://rolandkuhn.com>
 */
package com.reactivedesignpatterns.chapter17

import com.typesafe.config.ConfigFactory
import akka.actor._
import akka.cluster._
import akka.cluster.sharding._
import java.net.URI
import java.util.UUID

object ShardSupport {
  /*
   * Zastosowanie referencji shoppingCart jako klucza odłamkowania; częściowa funkcja
   * musi zwracać zarówno klucz jak i komunikat do przesłania; w przypadku niezgodności
   * komunikat jest usuwany.
   */
  val extractEntityId: ShardRegion.ExtractEntityId = {
    case mc @ ManagerCommand(cmd, _, _) => cmd.shoppingCart.id.toString -> mc
    case mc @ ManagerQuery(query, _, _) => query.shoppingCart.id.toString -> mc
  }

  /*
   * Alokacja 256 obiektów shoppingCarts na podstawie najmłodszych 8 bitów
   * kodu identyfikatora; jest to funkcja, którą trzeba zdefiniować dla wszystkich
   * przesyłanych komunikatów.
   */
  val extractShardId: ShardRegion.ExtractShardId = {
    case ManagerCommand(cmd, _, _) => toHex(cmd.shoppingCart.id.hashCode & 255)
    case ManagerQuery(query, _, _) => toHex(query.shoppingCart.id.hashCode & 255)
  }
  private def toHex(b: Int) = new java.lang.StringBuilder(2).append(hexDigits(b >> 4)).append(hexDigits(b & 15)).toString
  private val hexDigits = "0123456789ABCDEF"

  val RegionName = "ShoppingCart"
}

object ShardingExample extends App {
  val clusterConfig = ConfigFactory.parseString("""
akka.loglevel = INFO
akka.actor.provider = "akka.cluster.ClusterActorRefProvider"
akka.actor.warn-about-java-serializer-usage = off
akka.cluster.min-nr-of-members = 2
akka.remote.netty.tcp {
  hostname = localhost
  port = 0
}
akka.cluster.sharding.state-store-mode = ddata
""")
  val node1Config = ConfigFactory.parseString("akka.remote.netty.tcp.port = 2552")

  val sys1 = ActorSystem("ShardingExample", node1Config.withFallback(clusterConfig))
  val seed = Cluster(sys1).selfAddress

  def startNode(sys: ActorSystem): Unit = {
    Cluster(sys).join(seed)
    ClusterSharding(sys).start(
      typeName = ShardSupport.RegionName,
      entityProps = Props(new Manager),
      settings = ClusterShardingSettings(sys1),
      extractEntityId = ShardSupport.extractEntityId,
      extractShardId = ShardSupport.extractShardId)
  }

  startNode(sys1)

  val sys2 = ActorSystem("ShardingExample", clusterConfig)
  startNode(sys2)

  /*
   * Od tego momentu komunikujemy się z odłamkowanym koszykiem zakupów
   * za pomocą regionu odłamków, który pełni rolę lokalnego mediatora
   * wysyłającego polecenia do odpowiednich węzłów.
   */
  val manager = ClusterSharding(sys1).shardRegion(ShardSupport.RegionName)

  def mkURI(): URI = URI.create(UUID.randomUUID().toString)

  val customer = CustomerRef(mkURI())
  val item1, item2 = ItemRef(mkURI())
  val shoppingCart1, shoppingCart2 = ShoppingCartRef(mkURI())

  Cluster(sys1).registerOnMemberUp(
    sys1.actorOf(Props(new Actor with ActorLogging {
      manager ! ManagerCommand(SetOwner(shoppingCart1, customer), 0, self)
      manager ! ManagerCommand(AddItem(shoppingCart1, item1, 5), 1, self)
      manager ! ManagerCommand(AddItem(shoppingCart1, item1, -3), 2, self)
      manager ! ManagerCommand(AddItem(shoppingCart1, item2, 6), 3, self)
      manager ! ManagerCommand(RemoveItem(shoppingCart1, item1, 3), 4, self)
      manager ! ManagerQuery(GetItems(shoppingCart1), 5, self)

      def receive = {
        case ManagerEvent(id, event)   => log.info("udało się ({}): {}", id, event)
        case ManagerRejection(id, msg) => log.warning("odrzucono ({}): {}", id, msg)
        case ManagerResult(id, result) =>
          log.info("wynik ({}): {}", id, result)

          manager ! ManagerCommand(SetOwner(shoppingCart2, customer), 10, self)
          manager ! ManagerCommand(AddItem(shoppingCart2, item2, 15), 11, self)
          manager ! ManagerCommand(AddItem(shoppingCart2, item2, -3), 12, self)
          manager ! ManagerCommand(AddItem(shoppingCart2, item1, 60), 13, self)
          manager ! ManagerCommand(RemoveItem(shoppingCart2, item2, 3), 14, self)
          manager ! ManagerQuery(GetItems(shoppingCart2), 15, self)

          context.become(second)
      }
      def second: Receive = {
        case ManagerEvent(id, event)   => log.info("udało się ({}): {}", id, event)
        case ManagerRejection(id, msg) => log.warning("odrzucono ({}): {}", id, msg)
        case ManagerResult(id, result) =>
          log.info("wynik ({}): {}", id, result)
          sys1.terminate()
          sys2.terminate()
      }
    }), "client"))
}
