<?php

/*
 * This file is part of sonata-project.
 *
 * (c) 2010 Thomas Rabaix
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Sonata\BlockBundle\Twig\Extension;

use Sonata\BlockBundle\Block\BlockServiceManagerInterface;
use Sonata\BlockBundle\Model\BlockInterface;
use Sonata\BlockBundle\Block\BlockLoaderInterface;
use Sonata\BlockBundle\Block\BlockRendererInterface;

use Sonata\CacheBundle\Cache\CacheManagerInterface;
use Symfony\Component\HttpFoundation\Response;

class BlockExtension extends \Twig_Extension
{
    private $blockServiceManager;

    private $cacheManager;

    private $environment;

    private $cacheBlocks;

    private $blockLoader;

    private $blockRenderer;

    /**
     * @param \Sonata\BlockBundle\Block\BlockServiceManagerInterface $blockServiceManager
     * @param \Sonata\CacheBundle\Cache\CacheManagerInterface        $cacheManager
     * @param array                                                  $cacheBlocks
     * @param \Sonata\BlockBundle\Block\BlockLoaderInterface         $blockLoader
     * @param \Sonata\BlockBundle\Block\BlockRendererInterface       $blockRenderer
     */
    public function __construct(BlockServiceManagerInterface $blockServiceManager, CacheManagerInterface $cacheManager, array $cacheBlocks, BlockLoaderInterface $blockLoader, BlockRendererInterface $blockRenderer)
    {
        $this->blockServiceManager = $blockServiceManager;
        $this->cacheManager        = $cacheManager;
        $this->cacheBlocks         = $cacheBlocks;
        $this->blockLoader         = $blockLoader;
        $this->blockRenderer       = $blockRenderer;
    }

    /**
     * Returns a list of functions to add to the existing list.
     *
     * @return array An array of functions
     */
    public function getFunctions()
    {
        return array(
            'sonata_block_render'               => new \Twig_Function_Method($this, 'renderBlock', array('is_safe' => array('html'))),
            'sonata_block_include_javascripts'  => new \Twig_Function_Method($this, 'includeJavascripts', array('is_safe' => array('html'))),
            'sonata_block_include_stylesheets'  => new \Twig_Function_Method($this, 'includeStylesheets', array('is_safe' => array('html'))),
        );
    }

    /**
     * {@inheritdoc}
     */
    public function initRuntime(\Twig_Environment $environment)
    {
        $this->environment = $environment;
    }

    /**
     * Returns the name of the extension.
     *
     * @return string The extension name
     */
    public function getName()
    {
        return 'sonata_block';
    }

    /**
     * @param string $media screen|all ....
     *
     * @return array|string
     */
    public function includeJavascripts($media)
    {
        $javascripts = array();

        foreach ($this->blockServiceManager->getLoadedServices() as $service) {
            $javascripts = array_merge($javascripts, $service->getJavacripts($media));
        }

        if (count($javascripts) == 0) {
            return '';
        }

        $html = "";
        foreach ($javascripts as $javascript) {
            $html .= "\n" . sprintf('<script src="%s" type="text/javascript"></script>', $javascript);
        }

        return $html;
    }

    /**
     * @param array $media
     *
     * @return array|string
     */
    public function includeStylesheets($media)
    {
        $stylesheets = array();

        foreach ($this->blockServiceManager->getLoadedServices() as $service) {
            $stylesheets = array_merge($stylesheets, $service->getStylesheets($media));
        }

        if (count($stylesheets) == 0) {
            return '';
        }

        $html = sprintf("<style type='text/css' media='%s'>", $media);

        foreach ($stylesheets as $stylesheet) {
            $html .= "\n" . sprintf('@import url(%s);', $stylesheet, $media);
        }

        $html .= "\n</style>";

        return $html;
    }

    /**
     * @throws \RuntimeException
     *
     * @param mixed $block
     * @param bool  $useCache
     * @param array $extraCacheKeys
     *
     * @return string
     */
    public function renderBlock($block, $useCache = true, array $extraCacheKeys = array())
    {
        if (!$block instanceof BlockInterface) {
            $block = $this->blockLoader->load($block);
        }

        $cacheService = $cacheKeys = false;

        if ($useCache && ($cacheService = $this->getCacheService($block))) {
            $cacheKeys = array_merge(
                $extraCacheKeys,
                $this->blockServiceManager->get($block)->getCacheKeys($block)
            );

            if ($cacheService->has($cacheKeys)) {
                $cacheElement = $cacheService->get($cacheKeys);

                if (!$cacheElement->isExpired() && $cacheElement->getData() instanceof Response) {
                    return $cacheElement->getData()->getContent();
                }
            }
        }

        $recorder = $this->cacheManager->getRecorder();

        if ($recorder) {
            $recorder->push();
        }

        $response = $this->blockRenderer->render($block);

        $contextualKeys = $recorder ? $recorder->pop() : array();

        if ($response->isCacheable() && $useCache && $cacheKeys && $cacheService) {
            $cacheService->set($cacheKeys, $response, $block->getTtl(), $contextualKeys);
        }

        return $response->getContent();
    }

    /**
     * @param \Sonata\BlockBundle\Model\BlockInterface $block
     *
     * @return \Sonata\CacheBundle\Cache\CacheInterface;
     */
    protected function getCacheService(BlockInterface $block)
    {
        $type = isset($this->cacheBlocks[$block->getType()]) ? $this->cacheBlocks[$block->getType()] : false;

        if (!$type) {
            return false;
        }

        return $this->cacheManager->getCacheService($type);
    }
}

