/*---------------------------------------------------------------------------*/
    #include <errno.h>
    #include <stdio.h>
    #include <stdlib.h>
    #include <sys/ptrace.h>
    #include <asm/user.h>     /* dla struktury user */
    
    #define MODE_ATTACH 1
    #define MODE_LAUNCH 2

    /* skrt do dostpu do rejestrw debugowania */
    #define DR( u, num ) u.u_debugreg[num]

    /* pooenie rejestru 'num' wzgldem pocztku struktury user */
    #define DR_OFF( u, num ) (long)(&u.u_debugreg[num]) - (long)&u

    /* pobranie rejestru debugowania o numerze 'num' struktury user 'u' procesu 'pid' */
    #define GET_DR( u, num, pid )                        \
        DR(u, num) = ptrace( PTRACE_PEEKUSER, pid,         \
                  DR_OFF(u, num), NULL );

    /* ustawienie rejestru debugowania o numerze 'num' struktury user 'u' procesu 'pid' */
    /* UWAGA: Strona man funkcji ptrace(2) zawiera bd: ostatni argument
       POKEUSER musi by sowem, a nie adresem sowa w przestrzeni adresowej
       Procesu nadrzdnego. Wicej na ten temat w arch/i386/kernel/ptrace.c  */
    #define SET_DR( u, num, pid )                        \
        ptrace( PTRACE_POKEUSER, pid, DR_OFF(u, num), DR(u, num) );

    /* zwraca liczb bajtw do << w celu ustawienia lub pobrania bitu uaktywnienia lokalnego */
    #define LOCAL_ENABLE( num ) ( 1 << num )

    #define DR_LEN_MASK 0x3
    #define DR_LEN( num ) (16 + (4*num))

    #define DR_RWX_MASK 0x3
    #define DR_RWX( num )  (18 + (4*num))

    /* !=0 jeeli puapka jest spowodowana prac krokow */
    #define DR_STAT_STEP( dr6 ) ( dr6 & 0x2000 )

    /* !=0 jeeli puapka jest spowodowana przeczeniem zadania */
    #define DR_STAT_TASK( dr6 ) ( dr6 & 0x4000 )
    
    /* !=0 jeeli puapka jest spowodowana wykryciem dostpu do rejestru debugowania */
    #define DR_STAT_DRPERM( dr6 ) ( dr6 & 0x8000 )

    /* zwraca rejestr debugowania, ktry spowodowa puapk */
    #define DR_STAT_DR( dr6 ) ( (dr6 & 0x0F)  )

    /* dugo wynosi 1 bajt, 2 bajty, jest nieokrelona lub wynosi 4 bajty */
    enum dr_len { len_byte = 0, len_hword, len_unk, len_word };

    /* warunkiem punktu wstrzymania jest wykonanie, zapis, odczyt lub zapis urzdzenia I/O albo zapis lub czytanie danych */
    enum dr_rwx { bp_x = 0, bp_w, bp_iorw, bp_rw };

    int set_bp(int pid, unsigned long rva, enum dr_len len, enum dr_rwx rwx){
        struct user u = {0};
        int x, err, dreg = -1;

        err = errno;
        GET_DR( u, 7, pid );
        if ( err != errno ) {
            fprintf(stderr, "BP_SET bd czytania dr7: %s\n", 
                    strerror(errno));
            return(0);
        }

        /* szukanie nieuywanych rejestrw */
        for ( x = 0; x < 4; x++ ){
            if ( ! DR(u, 7) & LOCAL_ENABLE( x ) ) {
                dreg = x;
                break;
            }
        }
        if ( dreg != -1 ) {
            /* set bp */
            DR(u, dreg) = rva;
            err = SET_DR( u, dreg, pid );
            if ( err == -1 ) {
                fprintf(stderr, "BP_SET DR%d bd: %s\n", dreg, 
                        strerror(errno));
                return;
            }
            /* uaktywnienie punktu wstrzymania i warunkw w DR7 */
            DR(u, 7) &= ~(DR_LEN_MASK << DR_LEN(dreg)); 
            DR(u, 7) &= ~(DR_RWX_MASK << DR_RWX(dreg)); 
    
            DR(u, 7) |= len << DR_LEN(dreg); 
            DR(u, 7) |= rwx << DR_RWX(dreg); 
            DR(u, 7) |= LOCAL_ENABLE(dreg);
            err = SET_DR( u, 7, pid );
            if ( err == -1 ) {
                fprintf(stderr, "BP_SET DR7 error: %s\n", 
                        strerror(errno));
                return;
            }
        }

        return( dreg );    /* -1 oznacza brak wolnych rejestrw debugowania */
    }

    int unset_bp( int pid, unsigned long rva ) {
        struct user u = {0};
        int x, err, dreg = -1;

        for ( x = 0; x < 4; x++ ){
            err = errno;
            GET_DR(u, x, pid);
            if ( err != errno ) {
                fprintf(stderr, "BP_UNSET bd pobrania DR%d: %s\n", x, 
                            strerror(errno));
                return(0);
            }
            if ( DR(u, x) == rva ) {
                dreg = x;
                break;
            }
        }
        if ( dreg != -1 ) {
            err = errno;
            GET_DR( u, 7, pid );
            if ( err != errno ) {
                fprintf(stderr, "BP_UNSET bd pobrania DR7: %s\n", 
                        strerror(errno));
                return(0);
            }
            DR(u, 7) &= ~(LOCAL_ENABLE(dreg));
            err = SET_DR( u, 7, pid ) ;
            if ( err == -1 ) {
                fprintf(stderr, "BP_UNSET bd DR7: %s\n", 
                        strerror(errno));
                return;
            }
        }
        return(dreg);    /* -1 oznacza brak rejestrw debugowania ustawionych na rva */
    }

    /* powd puapki punktu wstrzymania */
    enum bp_status { bp_trace, bp_task, bp_perm, bp_0, bp_1, bp_2, bp_3, 
                 bp_unk };

    enum bp_status get_bp_status( int pid ) {
        int dreg;
        struct user u = {0};
        enum bp_status rv = bp_unk;
    
        GET_DR( u, 6, pid );
        printf("Proces potomny zatrzymany na ");
        if ( DR_STAT_STEP( DR(u, 6) ) ) {
            rv = bp_trace;
        } else if  ( DR_STAT_TASK(DR(u,6)) ){
            rv = bp_task;
        } else if ( DR_STAT_DRPERM(DR(u,6)) ) {
            rv = bp_perm;
        } else {
            dreg = DR_STAT_DR(DR(u,6));
            if ( dreg == 1 ) {
                rv = bp_0;
            } else if ( dreg == 2 ) {
                rv = bp_1;
            } else if ( dreg == 4 ) {
                rv = bp_2;
            } else if ( dreg == 8 ) {
                rv = bp_3;
            }
        }
        return( rv );
    }
/*---------------------------------------------------------------------------*/
