---------------------------------------------------------------------------
-- SQL w praktyce. Jak dzięki danym uzyskiwać cenne informacje. Wydanie II
-- Anthony DeBarros

-- Rozdział 16. Przykłady z kodem
----------------------------------------------------------------------------

-- Listing 16.1: Dane formatu JSON z informacjami o dwóch filmach

[{
	"title": "The Incredibles",
	"year": 2004,
	"rating": {
		"MPAA": "PG"
	},
	"characters": [{
		"name": "Mr. Incredible",
		"actor": "Craig T. Nelson"
	}, {
		"name": "Elastigirl",
		"actor": "Holly Hunter"
	}, {
		"name": "Frozone",
		"actor": "Samuel L. Jackson"
	}],
	"genre": ["animation", "action", "sci-fi"]
}, {
	"title": "Cinema Paradiso",
	"year": 1988,
	"characters": [{
		"name": "Salvatore",
		"actor": "Salvatore Cascio"
	}, {
		"name": "Alfredo",
		"actor": "Philippe Noiret"
	}],
	"genre": ["romance", "drama"]
}]

-- Listing 16.2: Tworzenie tabeli do przechowywania danych JSON oraz dodawanie indeksu

CREATE TABLE films (
    id integer GENERATED ALWAYS AS IDENTITY PRIMARY KEY,
    film jsonb NOT NULL
);

COPY films (film)
FROM 'C:\TwojKatalog\films.json';

CREATE INDEX idx_film ON films USING GIN (film);

SELECT * FROM films;


-- Operatory wyodrębniania typów danych json i jsonb

-- Listing 16.3: Uzyskiwanie wartości klucza formatu JSON przy użyciu operatorów wyodrębniania pól 

-- Zwraca wartość klucza jako typu danych JSON
SELECT id, film -> 'title' AS title
FROM films
ORDER BY id;

-- Zwraca wartość klucza jako tekst
SELECT id, film ->> 'title' AS title
FROM films
ORDER BY id;

-- Zwraca całą tablicę jako typu danych JSON
SELECT id, film -> 'genre' AS genre
FROM films
ORDER BY id;

-- Listing 16.4: Uzyskiwanie wartości tablicy danych JSON za pomocą operatorów wyodrębniania elementów 

-- Wyodrębnia pierwszy element tablicy JSON
-- (elementy tablicy są indeksowane od zera, ale ujemne liczby całkowite są odliczane od końca).
SELECT id, film -> 'genre' -> 0 AS genres
FROM films
ORDER BY id;

SELECT id, film -> 'genre' -> -1 AS genres
FROM films
ORDER BY id;

SELECT id, film -> 'genre' -> 2 AS genres
FROM films
ORDER BY id;

-- Zwraca element tablicy jako tekst
SELECT id, film -> 'genre' ->> 0 AS genres
FROM films
ORDER BY id;

-- Listing 16.5: Uzyskiwanie wartości klucza formatu JSON za pomocą operatorów wyodrębniania ze ścieżki

-- Pobranie oceny filmu uzyskanej od stowarzyszenia MPAA
SELECT id, film #> '{rating, MPAA}' AS mpaa_rating
FROM films
ORDER BY id;

-- Pobranie imienia pierwszej postaci
SELECT id, film #> '{characters, 0, name}' AS name
FROM films
ORDER BY id;

-- Tak samo jak wyżej, ale zwracany jest tekst
SELECT id, film #>> '{characters, 0, name}' AS name
FROM films
ORDER BY id;


-- Operatory ograniczania i obecności typu jsonb

-- Listing 16.6: Demonstrowanie użycia operatora ograniczenia @>

-- Czy wartość JSON film zawiera następującą parę klucz-wartość?
SELECT id, film ->> 'title' AS title,
       film @> '{"title": "The Incredibles"}'::jsonb AS is_incredible
FROM films
ORDER BY id;

-- Listing 16.7: Zastosowanie operatora ograniczania w klauzuli WHERE

SELECT film ->> 'title' AS title,
       film ->> 'year' AS year
FROM films
WHERE film @> '{"title": "The Incredibles"}'::jsonb; 

-- Listing 16-8: Demonstrowanie użycia operatora ograniczenia <@

SELECT film ->> 'title' AS title,
       film ->> 'year' AS year
FROM films
WHERE '{"title": "The Incredibles"}'::jsonb <@ film; 

-- Listing 16.9: Demonstrowanie operatorów obecności

-- Czy łańcuch tekstowy istnieje jako klucz najwyższego poziomu czy jako element tablicy w obrębie wartości JSON?
SELECT film ->> 'title' AS title
FROM films
WHERE film ? 'rating';

-- Czy dowolny z łańcuchów w tablicy tekstowej istnieje jako klucz najwyższego poziomu czy jako element tablicy?
SELECT film ->> 'title' AS title,
       film ->> 'rating' AS rating,
       film ->> 'genre' AS genre
FROM films
WHERE film ?| '{rating, genre}';

-- Czy wszystkie łańcuchy w tablicy tekstowej istnieją jako klucze najwyższego poziomu czy jako elementy tablicy?
SELECT film ->> 'title' AS title,
       film ->> 'rating' AS rating,
       film ->> 'genre' AS genre
FROM films
WHERE film ?& '{rating, genre}';


-- Analizowanie danych o trzęsieniach ziemi

-- Listing 16.10: Rekord z danymi JSON dotyczącymi jednego trzęsienia ziemi

{
	"type": "Feature",
	"properties": {
		"mag": 1.44,
		"place": "134 km W of Adak, Alaska",
		"time": 1612051063470,
		"updated": 1612139465880,
		"tz": null,
		"url": "https://earthquake.usgs.gov/earthquakes/eventpage/av91018173",
		"detail": "https://earthquake.usgs.gov/fdsnws/event/1/query?eventid=av91018173&format=geojson",
		"felt": null,
		"cdi": null,
		"mmi": null,
		"alert": null,
		"status": "reviewed",
		"tsunami": 0,
		"sig": 32,
		"net": "av",
		"code": "91018173",
		"ids": ",av91018173,",
		"sources": ",av,",
		"types": ",origin,phase-data,",
		"nst": 10,
		"dmin": null,
		"rms": 0.15,
		"gap": 174,
		"magType": "ml",
		"type": "earthquake",
		"title": "M 1.4 - 134 km W of Adak, Alaska"
	},
	"geometry": {
		"type": "Point",
		"coordinates": [-178.581, 51.8418333333333, 22.48]
	},
	"id": "av91018173"
}

-- Listing 16.11: Tworzenie tabeli earthquakes i ładowanie do niej danych

CREATE TABLE earthquakes (
    id integer GENERATED ALWAYS AS IDENTITY PRIMARY KEY,
    earthquake jsonb NOT NULL
);

COPY earthquakes (earthquake)
FROM 'C:\TwojKatalog\earthquakes.json';

CREATE INDEX idx_earthquakes ON earthquakes USING GIN (earthquake);

SELECT * FROM earthquakes;

-- Listing 16.12: Uzyskiwanie czasu wystąpienia trzęsienia ziemi
-- Zauważ, że czas zapisano w formacie epoki systemu Unix 

SELECT id, earthquake #>> '{properties, time}' AS time 
FROM earthquakes
ORDER BY id LIMIT 5;

-- Listing 16.13: Przekształcanie wartości klucza time w znacznik czasu

SELECT id, earthquake #>> '{properties, time}' as time,
       to_timestamp(
           (earthquake #>> '{properties, time}')::bigint / 1000
                   ) AS time_formatted
FROM earthquakes
ORDER BY id LIMIT 5;

-- Wyświetlanie i ustawianie w razie potrzeby strefy czasowej
SHOW timezone;
SET timezone TO 'US/Eastern';
SET timezone TO 'UTC';

-- Listing 16.14: Znajdowanie czasu wystąpienia najstarszego i najnowszego trzęsienia ziemi

SELECT min(to_timestamp(
           (earthquake #>> '{properties, time}')::bigint / 1000
                       )) AT TIME ZONE 'UTC' AS min_timestamp,
       max(to_timestamp(
           (earthquake #>> '{properties, time}')::bigint / 1000
                       )) AT TIME ZONE 'UTC' AS max_timestamp
FROM earthquakes;

-- Listing 16.15: Znajdowanie pięciu trzęsień ziemi o największej magnitudzie

SELECT earthquake #>> '{properties, place}' AS place,
       to_timestamp((earthquake #>> '{properties, time}')::bigint / 1000)
           AT TIME ZONE 'UTC' AS time,
       (earthquake #>> '{properties, mag}')::numeric AS magnitude
FROM earthquakes
ORDER BY (earthquake #>> '{properties, mag}')::numeric DESC NULLS LAST
LIMIT 5;

-- Dodatek: zamiast operatora wyodrębniania ze ścieżki (#>>), można też zastosować wyodrębnianie pola:
SELECT earthquake -> 'properties' ->> 'place' AS place,
       to_timestamp((earthquake -> 'properties' ->> 'time')::bigint / 1000)
           AT TIME ZONE 'UTC' AS time,
       (earthquake #>> '{properties, mag}')::numeric AS magnitude
FROM earthquakes
ORDER BY (earthquake #>> '{properties, mag}')::numeric DESC NULLS LAST
LIMIT 5;

-- Listing 16.16: Znajdowanie trzęsień ziemi z największą liczbą raportów na stronie internetowej Did You Feel It?
-- https://earthquake.usgs.gov/data/dyfi/

SELECT earthquake #>> '{properties, place}' AS place,
       to_timestamp((earthquake #>> '{properties, time}')::bigint / 1000)
           AT TIME ZONE 'UTC' AS time,
       (earthquake #>> '{properties, mag}')::numeric AS magnitude,
       (earthquake #>> '{properties, felt}')::integer AS felt
FROM earthquakes
ORDER BY (earthquake #>> '{properties, felt}')::integer DESC NULLS LAST
LIMIT 5;

-- Listing 16.17: Wyodrębnianie danych o lokalizacji trzęsienia ziemi

SELECT id,
       earthquake #>> '{geometry, coordinates}' AS coordinates,
       earthquake #>> '{geometry, coordinates, 0}' AS longitude,
       earthquake #>> '{geometry, coordinates, 1}' AS latitude
FROM earthquakes
ORDER BY id
LIMIT 5;

-- Listing 16.18: Przekształcanie danych JSON o lokalizacji w punkt typu geography rozszerzenia PostGIS
SELECT ST_SetSRID(
         ST_MakePoint(
            (earthquake #>> '{geometry, coordinates, 0}')::numeric,
            (earthquake #>> '{geometry, coordinates, 1}')::numeric
         ),
             4326)::geography AS earthquake_point
FROM earthquakes
ORDER BY id;

-- Listing 16.19: Przekształcanie współrzędnych danych JSON w kolumnę danych geometrycznych rozszerzenia PostGIS

-- Dodanie kolumny typu danych geography
ALTER TABLE earthquakes ADD COLUMN earthquake_point geography(POINT, 4326);

-- Zaktualizowanie tabeli earthquakes z użyciem elementu Point
UPDATE earthquakes
SET earthquake_point = 
        ST_SetSRID(
            ST_MakePoint(
                (earthquake #>> '{geometry, coordinates, 0}')::numeric,
                (earthquake #>> '{geometry, coordinates, 1}')::numeric
             ),
                 4326)::geography;

CREATE INDEX quake_pt_idx ON earthquakes USING GIST (earthquake_point);

-- Listing 16.20: Lokalizowanie trzęsień ziemi w promieniu 50 mil względem miasta Tulsa w stanie Oklahoma

SELECT earthquake #>> '{properties, place}' AS place,
       to_timestamp((earthquake -> 'properties' ->> 'time')::bigint / 1000)
           AT TIME ZONE 'UTC' AS time,
       (earthquake #>> '{properties, mag}')::numeric AS magnitude,
       earthquake_point
FROM earthquakes
WHERE ST_DWithin(earthquake_point,
                 ST_GeogFromText('POINT(-95.989505 36.155007)'),
                 80468)
ORDER BY time;

-- Generowanie i przetwarzanie danych formatu JSON

-- Listing 16.21: Przekształcanie wyników zapytania w dane JSON za pomocą funkcji to_json()

-- Przekształcenie całego wiersza z tabeli
SELECT to_json(employees) AS json_rows
FROM employees;

-- Listing 16.22: Określanie kolumn, które zostaną przekształcone w dane formatu JSON
-- Zwraca nazwy kluczy jako f1, f2 itd.
SELECT to_json(row(emp_id, last_name)) AS json_rows
FROM employees;

-- Listing 16.23: Generowanie nazw kluczy z użyciem podzapytania
SELECT to_json(employees) AS json_rows
FROM (
    SELECT emp_id, last_name AS ln FROM employees
) AS employees;

-- Listing 16.24: Agregowanie wierszy i przekształcanie ich w dane JSON
SELECT json_agg(to_json(employees)) AS json
FROM (
    SELECT emp_id, last_name AS ln FROM employees
) AS employees;

-- Listing 16.25: Dodawanie pary klucz-wartość najwyższego poziomu za pośrednictwem konkatenacji
-- Dwa przykłady

UPDATE films
SET film = film || '{"studio": "Pixar"}'::jsonb
WHERE film @> '{"title": "The Incredibles"}'::jsonb; 

UPDATE films
SET film = film || jsonb_build_object('studio', 'Pixar')
WHERE film @> '{"title": "The Incredibles"}'::jsonb; 

SELECT film FROM films -- check the updated data
WHERE film @> '{"title": "The Incredibles"}'::jsonb; 

-- Listing 16.26: Dodawanie za pomocą funkcji jsonb_set() wartości tablicy obecnej w ścieżce

UPDATE films
SET film = jsonb_set(film,
                 '{genre}',
                  film #> '{genre}' || '["World War II"]',
                  true)
WHERE film @> '{"title": "Cinema Paradiso"}'::jsonb; 

SELECT film FROM films -- sprawdzenie zaktualizowanych danych
WHERE film @> '{"title": "Cinema Paradiso"}'::jsonb; 

-- Listing 16.27: Usuwanie wartości z danych JSON

-- Usunięcie klucza studio wraz z jego wartością dla filmu The Incredibles
UPDATE films
SET film = film - 'studio'
WHERE film @> '{"title": "The Incredibles"}'::jsonb; 

-- Usunięcie trzeciego elementu tablicy genre dla filmu Cinema Paradiso
UPDATE films
SET film = film #- '{genre, 2}'
WHERE film @> '{"title": "Cinema Paradiso"}'::jsonb; 


-- Funkcje przetwarzające dane JSON

-- Listing 16.28: Ustalanie długości tablicy

SELECT id,
       film ->> 'title' AS title,
       jsonb_array_length(film -> 'characters') AS num_characters
FROM films
ORDER BY id;

-- Listing 16.29: Zwracanie elementów tablicy jako wierszy

SELECT id,
       jsonb_array_elements(film -> 'genre') AS genre_jsonb,
       jsonb_array_elements_text(film -> 'genre') AS genre_text
FROM films
ORDER BY id;

-- Listing 16.30: Zwracanie wartości kluczy z każdego elementu tablicy

SELECT id, 
       jsonb_array_elements(film -> 'characters')
FROM films
ORDER BY id;

WITH characters (id, json) AS (
    SELECT id,
           jsonb_array_elements(film -> 'characters')
    FROM films
)
SELECT id, 
       json ->> 'name' AS name,
       json ->> 'actor' AS actor
FROM characters
ORDER BY id;
