---------------------------------------------------------------------------
-- SQL w praktyce. Jak dzięki danym uzyskiwać cenne informacje. Wydanie II
-- Anthony DeBarros

-- Rozdział 14. Przykłady z kodem
----------------------------------------------------------------------------


-- Powszechnie używane funkcje łańcuchowe
-- Pełna lista: https://www.postgresql.org/docs/current/functions-string.html

-- Formatowanie dotyczące wielkości znaków
SELECT upper('Neal7');
SELECT lower('Randy');
SELECT initcap('at the end of the day');
-- Zwróć uwagę na to, funkcja initcap nie działa idealnie w przypadku skrótów
SELECT initcap('Practical SQL');

-- Informacje o znakach
SELECT char_length(' Pat ');
SELECT length(' Pat ');
SELECT position(', ' in 'Tan, Bella');

-- Usuwanie znaków
SELECT trim('s' from 'socks');
SELECT trim(trailing 's' from 'socks');
SELECT trim(' Pat ');
SELECT char_length(trim(' Pat ')); -- note the length change
SELECT ltrim('socks', 's');
SELECT rtrim('socks', 's');

-- Wyodrębnianie i zastępowanie znaków
SELECT left('703-555-1212', 3);
SELECT right('703-555-1212', 8);
SELECT replace('bat', 'b', 'c');


-- Tabela 14.2: Przykłady dopasowywania wyrażenia regularnego

-- Dowolny znak pojawiający się co najmniej raz
SELECT substring('The game starts at 7 p.m. on May 2, 2024.' from '.+');
-- Jedna lub dwie cyfry, po których w grupie przechwytywania następuje spacja oraz ciąg a.m. lub p.m.
SELECT substring('The game starts at 7 p.m. on May 2, 2024.' from '\d{1,2} (?:a.m.|p.m.)');
-- Jeden lub więcej znaków słowa znajdującego się na początku
SELECT substring('The game starts at 7 p.m. on May 2, 2024.' from '^\w+');
-- Jeden lub więcej znaków słowa, po których następuje na końcu dowolny znak
SELECT substring('The game starts at 7 p.m. on May 2, 2024.' from '\w+.$');
-- Jedno ze słów: May (maj) lub June (czerwiec)
SELECT substring('The game starts at 7 p.m. on May 2, 2024.' from 'May|June');
-- Cztery cyfry
SELECT substring('The game starts at 7 p.m. on May 2, 2024.' from '\d{4}');
-- Słowo May, po którym następuje spacja, cyfra, przecinek, spacja oraz 4 cyfry
SELECT substring('The game starts at 7 p.m. on May 2, 2024.' from 'May \d, \d{4}');

-- Listing 14.1: Użycie wyrażeń regularnych w klauzuli WHERE

SELECT county_name
FROM us_counties_pop_est_2019
WHERE county_name ~* '(lade|lare)'
ORDER BY county_name;

SELECT county_name
FROM us_counties_pop_est_2019
WHERE county_name ~* 'ash' AND county_name !~ 'Wash'
ORDER BY county_name;

-- Listing 14.2: Funkcje z wyrażeniami regularnymi służące do zastępowania lub dzielenia tekstu

SELECT regexp_replace('05/12/2024', '\d{4}', '2023');

SELECT regexp_split_to_table('Four,score,and,seven,years,ago', ',');

SELECT regexp_split_to_array('Phil Mike Tony Steve', ' ');

-- Listing 14.3: Ustalanie długości tablicy

SELECT array_length(regexp_split_to_array('Phil Mike Tony Steve', ' '), 1);


-- Przekształcanie tekstu w dane za pomocą funkcji wyrażeń regularnych

-- Listing 14.5: Tworzenie tabeli crime_reports i wczytywanie do niej danych
-- Dane pochodzące z witryny: https://sheriff.loudoun.gov/dailycrime

CREATE TABLE crime_reports (
    crime_id integer PRIMARY KEY GENERATED ALWAYS AS IDENTITY,
    case_number text,
    date_1 timestamptz,  -- uwaga: jest to zapis uproszczony systemu PostgreSQL dotyczący typu timestamp ze strefą czasową
    date_2 timestamptz,  -- uwaga: jest to zapis uproszczony systemu PostgreSQL dotyczący typu timestamp ze strefą czasową
    street text,
    city text,
    crime_type text,
    description text,
    original_text text NOT NULL
);

COPY crime_reports (original_text)
FROM 'C:\TwojKatalog\crime_reports.csv'
WITH (FORMAT CSV, HEADER OFF, QUOTE '"');

SELECT original_text FROM crime_reports;

-- Listing 14.6: Użycie funkcji regexp_match() do znalezienia pierwszej daty
SELECT crime_id,
       regexp_match(original_text, '\d{1,2}\/\d{1,2}\/\d{2}')
FROM crime_reports
ORDER BY crime_id;

-- Listing 14.7: Użycie funkcji regexp_matches() z flagą g
SELECT crime_id,
       regexp_matches(original_text, '\d{1,2}\/\d{1,2}\/\d{2}', 'g')
FROM crime_reports
ORDER BY crime_id;

-- Listing 14.8: Użycie funkcji regexp_match() do znalezienia drugiej daty
-- Zauważ, że wynik zawiera niepożądany łącznik
SELECT crime_id,
       regexp_match(original_text, '-\d{1,2}\/\d{1,2}\/\d{2}')
FROM crime_reports
ORDER BY crime_id;

-- Listing 14.9: Zastosowanie grupy przechwytywania do zwrócenia tylko daty
-- Eliminatuje łącznik
SELECT crime_id,
       regexp_match(original_text, '-(\d{1,2}\/\d{1,2}\/\d{2})')
FROM crime_reports
ORDER BY crime_id;

-- Listing 14.10: Dopasowywanie numeru sprawy, daty, typu przestępstwa i nazwy miasta

SELECT
    regexp_match(original_text, '(?:C0|SO)[0-9]+') AS case_number,
    regexp_match(original_text, '\d{1,2}\/\d{1,2}\/\d{2}') AS date_1,
    regexp_match(original_text, '\n(?:\w+ \w+|\w+)\n(.*):') AS crime_type,
    regexp_match(original_text, '(?:Sq.|Plz.|Dr.|Ter.|Rd.)\n(\w+ \w+|\w+)\n')
        AS city
FROM crime_reports
ORDER BY crime_id;

-- Dodatek: uzyskanie na raz wszystkich wyodrębnionych elementów

SELECT crime_id,
       regexp_match(original_text, '\d{1,2}\/\d{1,2}\/\d{2}') AS date_1,
       CASE WHEN EXISTS (SELECT regexp_matches(original_text, '-(\d{1,2}\/\d{1,2}\/\d{2})'))
            THEN regexp_match(original_text, '-(\d{1,2}\/\d{1,2}\/\d{2})')
            ELSE NULL
            END AS date_2,
       regexp_match(original_text, '\/\d{2}\n(\d{4})') AS hour_1,
       CASE WHEN EXISTS (SELECT regexp_matches(original_text, '\/\d{2}\n\d{4}-(\d{4})'))
            THEN regexp_match(original_text, '\/\d{2}\n\d{4}-(\d{4})')
            ELSE NULL
            END AS hour_2,
       regexp_match(original_text, 'hrs.\n(\d+ .+(?:Sq.|Plz.|Dr.|Ter.|Rd.))') AS street,
       regexp_match(original_text, '(?:Sq.|Plz.|Dr.|Ter.|Rd.)\n(\w+ \w+|\w+)\n') AS city,
       regexp_match(original_text, '\n(?:\w+ \w+|\w+)\n(.*):') AS crime_type,
       regexp_match(original_text, ':\s(.+)(?:C0|SO)') AS description,
       regexp_match(original_text, '(?:C0|SO)[0-9]+') AS case_number
FROM crime_reports
ORDER BY crime_id;

-- Listing 14.11: Pobieranie wartości z wnętrza tablicy

SELECT
    crime_id,
    (regexp_match(original_text, '(?:C0|SO)[0-9]+'))[1]
        AS case_number
FROM crime_reports
ORDER BY crime_id;

-- Listing 14.12: Aktualizowanie kolumny date_1 tabeli crime_reports

UPDATE crime_reports
SET date_1 = 
(
    (regexp_match(original_text, '\d{1,2}\/\d{1,2}\/\d{2}'))[1]
        || ' ' ||
    (regexp_match(original_text, '\/\d{2}\n(\d{4})'))[1] 
        ||' US/Eastern'
)::timestamptz
RETURNING crime_id, date_1, original_text;

-- Listing 14.13: Aktualizowanie wszystkich kolumn tabeli crime_reports

UPDATE crime_reports
SET date_1 = 
    (
      (regexp_match(original_text, '\d{1,2}\/\d{1,2}\/\d{2}'))[1]
          || ' ' ||
      (regexp_match(original_text, '\/\d{2}\n(\d{4})'))[1] 
          ||' US/Eastern'
    )::timestamptz,
             
    date_2 = 
    CASE 
    -- jeśli nie ma drugiej daty, lecz dostępna jest druga godzina
        WHEN (SELECT regexp_match(original_text, '-(\d{1,2}\/\d{1,2}\/\d{2})') IS NULL)
                     AND (SELECT regexp_match(original_text, '\/\d{2}\n\d{4}-(\d{4})') IS NOT NULL)
        THEN 
          ((regexp_match(original_text, '\d{1,2}\/\d{1,2}\/\d{2}'))[1]
              || ' ' ||
          (regexp_match(original_text, '\/\d{2}\n\d{4}-(\d{4})'))[1] 
              ||' US/Eastern'
          )::timestamptz 

    -- jeśli dostępna jest druga data i godzina
        WHEN (SELECT regexp_match(original_text, '-(\d{1,2}\/\d{1,2}\/\d{2})') IS NOT NULL)
              AND (SELECT regexp_match(original_text, '\/\d{2}\n\d{4}-(\d{4})') IS NOT NULL)
        THEN 
          ((regexp_match(original_text, '-(\d{1,2}\/\d{1,2}\/\d{2})'))[1]
              || ' ' ||
          (regexp_match(original_text, '\/\d{2}\n\d{4}-(\d{4})'))[1] 
              ||' US/Eastern'
          )::timestamptz 
    END,
    street = (regexp_match(original_text, 'hrs.\n(\d+ .+(?:Sq.|Plz.|Dr.|Ter.|Rd.))'))[1],
    city = (regexp_match(original_text,
                           '(?:Sq.|Plz.|Dr.|Ter.|Rd.)\n(\w+ \w+|\w+)\n'))[1],
    crime_type = (regexp_match(original_text, '\n(?:\w+ \w+|\w+)\n(.*):'))[1],
    description = (regexp_match(original_text, ':\s(.+)(?:C0|SO)'))[1],
    case_number = (regexp_match(original_text, '(?:C0|SO)[0-9]+'))[1];

-- Listing 14.14: Wyświetlanie wybranych danych o przestępstwach

SELECT date_1,
       street,
       city,
       crime_type
FROM crime_reports
ORDER BY crime_id;


-- Wyszukiwanie pełnotekstowe

-- Operatory wyszukiwania pełnotekstowego:
-- & (AND)
-- | (OR)
-- ! (NOT)

-- Uwaga: możesz wyświetlić dla zainstalowanego systemu PostgreSQL konfiguracje języków wyszukiwania za pomocą tej instrukcji:
SELECT cfgname FROM pg_ts_config;


-- Listing 14.15: Przekształcanie tekstu do postaci danych typu tsvector

SELECT to_tsvector('english', 'I am walking across the sitting room to sit with you.');

-- Listing 14.16: Przekształcanie terminów wyszukiwania w dane typu tsquery

SELECT to_tsquery('english', 'walking & sitting');

-- Listing 14.17: Wykonanie zapytania z funkcją ts_tsquery() względem danych typu tsvector

SELECT to_tsvector('english', 'I am walking across the sitting room') @@
       to_tsquery('english', 'walking & sitting');

SELECT to_tsvector('english', 'I am walking across the sitting room') @@ 
       to_tsquery('english', 'walking & running');

-- Listing 14.18: Tworzenie tabeli president_speeches i wypełnianie jej danymi

-- Źródła:
-- https://archive.org/details/State-of-the-Union-Addresses-1945-2006
-- https://www.presidency.ucsb.edu/documents/presidential-documents-archive-guidebook/annual-messages-congress-the-state-the-union
-- https://www.eisenhower.archives.gov/all_about_ike/speeches.html

CREATE TABLE president_speeches (
    president text NOT NULL,
    title text NOT NULL,
    speech_date date NOT NULL,
    speech_text text NOT NULL,
    search_speech_text tsvector,
    CONSTRAINT speech_key PRIMARY KEY (president, speech_date)
);

COPY president_speeches (president, title, speech_date, speech_text)
FROM 'C:\TwojKatalog\president_speeches.csv'
WITH (FORMAT CSV, DELIMITER '|', HEADER OFF, QUOTE '@');

SELECT * FROM president_speeches ORDER BY speech_date;

-- Listing 14.19: Przekształcanie tekstu przemówień do postaci danych typu tsvector kolumny search_speech_text

UPDATE president_speeches
SET search_speech_text = to_tsvector('english', speech_text);

-- Listing 14.20: Tworzenie indeksu GIN pod kątem wyszukiwania tekstowego

CREATE INDEX search_idx ON president_speeches USING gin(search_speech_text);

-- Listing 14.21: Znajdowanie przemówień zawierających słowo Vietnam

SELECT president, speech_date
FROM president_speeches
WHERE search_speech_text @@ to_tsquery('english', 'Vietnam')
ORDER BY speech_date;

-- Listing 14.22: Wyświetlanie wyników wyszukiwania przy użyciu funkcji ts_headline()

SELECT president,
       speech_date,
       ts_headline(speech_text, to_tsquery('english', 'tax'),
                   'StartSel = <,
                    StopSel = >,
                    MinWords=5,
                    MaxWords=7,
                    MaxFragments=1')
FROM president_speeches
WHERE search_speech_text @@ to_tsquery('english', 'tax')
ORDER BY speech_date;


-- Listing 14.23: Znajdowanie przemówień z użytym słowem transportation, lecz bez słowa roads

SELECT president,
       speech_date,
       ts_headline(speech_text,
                   to_tsquery('english', 'transportation & !roads'),
                   'StartSel = <,
                    StopSel = >,
                    MinWords=5,
                    MaxWords=7,
                    MaxFragments=1')
FROM president_speeches
WHERE search_speech_text @@
      to_tsquery('english', 'transportation & !roads')
ORDER BY speech_date;

-- Listing 14.24: Znajdowanie przemówień, w których słowo defence pojawia się po słowie military

SELECT president,
       speech_date,
       ts_headline(speech_text, 
                   to_tsquery('english', 'military <-> defense'),
                   'StartSel = <,
                    StopSel = >,
                    MinWords=5,
                    MaxWords=7,
                    MaxFragments=1')
FROM president_speeches
WHERE search_speech_text @@ 
      to_tsquery('english', 'military <-> defense')
ORDER BY speech_date;

-- Dodatek: przykład z odległością 2:
SELECT president,
       speech_date,
       ts_headline(speech_text, 
                   to_tsquery('english', 'military <2> defense'),
                   'StartSel = <,
                    StopSel = >,
                    MinWords=5,
                    MaxWords=7,
                    MaxFragments=2')
FROM president_speeches
WHERE search_speech_text @@ 
      to_tsquery('english', 'military <2> defense')
ORDER BY speech_date;

-- Listing 14.25: Ustalanie stopnia powiązania za pomocą funkcji ts_rank()

SELECT president,
       speech_date,
       ts_rank(search_speech_text,
               to_tsquery('english', 'war & security & threat & enemy'))
               AS score
FROM president_speeches
WHERE search_speech_text @@ 
      to_tsquery('english', 'war & security & threat & enemy')
ORDER BY score DESC
LIMIT 5;

-- Listing 14.26: Normalizacja funkcji ts_rank() z użyciem długości przemówienia

SELECT president,
       speech_date,
       ts_rank(search_speech_text,
               to_tsquery('english', 'war & security & threat & enemy'), 2)::numeric 
               AS score
FROM president_speeches
WHERE search_speech_text @@ 
      to_tsquery('english', 'war & security & threat & enemy')
ORDER BY score DESC
LIMIT 5;

-- Uwaga: aby zaznajomić się wyszukiwaniem pełnotekstowym systemu PostgreSQL zaimplementowanym w aplikacji internetowej za
-- pomocą zapytań podobnych do powyższych, zajrzyj na następującą stronę:
-- https://anthonydebarros.com/sotu/

