package tacos.restclient;

import java.util.Collection;
import java.util.List;

import org.springframework.core.ParameterizedTypeReference;
import org.springframework.hateoas.Resources;
import org.springframework.hateoas.client.Traverson;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import lombok.extern.slf4j.Slf4j;
import tacos.Ingredient;
import tacos.Taco;

@Service
@Slf4j
public class TacoCloudClient {

  private RestTemplate rest;
  private Traverson traverson;

  public TacoCloudClient(RestTemplate rest, Traverson traverson) {
    this.rest = rest;
    this.traverson = traverson;
  }

  //
  // Przykłady GET
  //

  /*
   * Określenie parametru jako argumentu varargs
   */
  public Ingredient getIngredientById(String ingredientId) {
    return rest.getForObject("http://localhost:8080/ingredients/{id}",
                             Ingredient.class, ingredientId);
  }

  /*
   * Implementacje alternatywne...
   * Następne trzy metody to alternatywne implementacje
   * getIngredientById() przedstawionej w rozdziale 6. Jeśli chcesz je wypróbować,
   * umieść w komentarzu poprzednią, a wyciągnij z komentarza
   * tę, która ma zostać użyta.
   */

  /*
   * Określenie parametrów za pomocą mapy
   */
  // public Ingredient getIngredientById(String ingredientId) {
  //   Map<String, String> urlVariables = new HashMap<>();
  //   urlVariables.put("id", ingredientId);
  //   return rest.getForObject("http://localhost:8080/ingredients/{id}",
  //       Ingredient.class, urlVariables);
  // }

  /*
   * Żądanie z użyciem adresu URI zamiast ciągu tekstowego
   */
  // public Ingredient getIngredientById(String ingredientId) {
  //   Map<String, String> urlVariables = new HashMap<>();
  //   urlVariables.put("id", ingredientId);
  //   URI url = UriComponentsBuilder
  //             .fromHttpUrl("http://localhost:8080/ingredients/{id}")
  //             .build(urlVariables);
  //   return rest.getForObject(url, Ingredient.class);
  // }

  /*
   * Używanie getForEntity() zamiast getForObject()
   */
  // public Ingredient getIngredientById(String ingredientId) {
  //   ResponseEntity<Ingredient> responseEntity =
  //       rest.getForEntity("http://localhost:8080/ingredients/{id}",
  //           Ingredient.class, ingredientId);
  //   log.info("Data i godzina pobrania zasobu: " +
  //           responseEntity.getHeaders().getDate());
  //   return responseEntity.getBody();
  // }

  public List<Ingredient> getAllIngredients() {
    return rest.exchange("http://localhost:8080/ingredients",
            HttpMethod.GET, null, new ParameterizedTypeReference<List<Ingredient>>() {})
        .getBody();
  }

  //
  // Przykłady PUT
  //

  public void updateIngredient(Ingredient ingredient) {
    rest.put("http://localhost:8080/ingredients/{id}",
          ingredient, ingredient.getId());
  }

  //
  // Przykłady POST
  //
  public Ingredient createIngredient(Ingredient ingredient) {
    return rest.postForObject("http://localhost:8080/ingredients",
        ingredient, Ingredient.class);
  }

  /*
   * Implementacje alternatywne...
   * Następne dwie metody to alternatywne implementacje
   * createIngredient() przedstawionej w rozdziale 6. Jeśli chcesz je wypróbować,
   * umieść w komentarzu poprzednią, a wyciągnij z komentarza
   * tę, która ma zostać użyta.
   */

  // public URI createIngredient(Ingredient ingredient) {
  //   return rest.postForLocation("http://localhost:8080/ingredients",
  //       ingredient, Ingredient.class);
  // }

  // public Ingredient createIngredient(Ingredient ingredient) {
  //   ResponseEntity<Ingredient> responseEntity =
  //          rest.postForEntity("http://localhost:8080/ingredients",
  //                             ingredient,
  //                             Ingredient.class);
  //   log.info("Nowy zasób został utworzony pod adresem " +
  //            responseEntity.getHeaders().getLocation());
  //   return responseEntity.getBody();
  // }

  //
  // Przykłady DELETE
  //

  public void deleteIngredient(Ingredient ingredient) {
    rest.delete("http://localhost:8080/ingredients/{id}",
        ingredient.getId());
  }

  //
  // Przykłady użycia Traverson wraz z RestTemplate
  //

  public Iterable<Ingredient> getAllIngredientsWithTraverson() {
    ParameterizedTypeReference<Resources<Ingredient>> ingredientType =
        new ParameterizedTypeReference<Resources<Ingredient>>() {};

    Resources<Ingredient> ingredientRes =
        traverson
          .follow("ingredients")
          .toObject(ingredientType);
    
    Collection<Ingredient> ingredients = ingredientRes.getContent();
          
    return ingredients;
  }

  public Ingredient addIngredient(Ingredient ingredient) {
    String ingredientsUrl = traverson
        .follow("ingredients")
        .asLink()
        .getHref();
    return rest.postForObject(ingredientsUrl,
                              ingredient,
                              Ingredient.class);
  }

  public Iterable<Taco> getRecentTacosWithTraverson() {
    ParameterizedTypeReference<Resources<Taco>> tacoType =
        new ParameterizedTypeReference<Resources<Taco>>() {};

    Resources<Taco> tacoRes =
        traverson
          .follow("tacos")
          .follow("recents")
          .toObject(tacoType);

      // Alternatywnie można użyć dwie świeżki w tym samym wywołaniu follow()
//    Resources<Taco> tacoRes =
//        traverson
//          .follow("tacos", "recents")
//          .toObject(tacoType);

    return tacoRes.getContent();
  }

}
