// Wersja końcowa
package com.apress.springrecipes.court.web;

import com.apress.springrecipes.court.domain.Player;
import com.apress.springrecipes.court.domain.Reservation;
import com.apress.springrecipes.court.domain.ReservationValidator;
import com.apress.springrecipes.court.domain.SportType;
import com.apress.springrecipes.court.service.ReservationService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.bind.support.SessionStatus;

import java.util.List;


@Controller
// Wiązanie kontrolera z adresem URL /reservationForm
// Początkowy widok jest określany na podstawie nazwy zwracanej przez domyślną metodę obsługi żądań GET.
@RequestMapping("/reservationForm")
// Dodawanie obiektu Reservation do sesji, ponieważ jest tworzony przy otwieraniu i używany po przesłaniu strony
@SessionAttributes("reservation")

public class ReservationFormController {
    private ReservationService reservationService;
    private ReservationValidator validator;

    // Podłączanie dostępnych w kontekście aplikacji usługi i walidatora w konstruktorze 
    @Autowired
    public ReservationFormController(ReservationService reservationService, ReservationValidator validator) {
        this.reservationService = reservationService;
        this.validator = validator;
    }

    // Create attribute for model 
    // Will be represented as drop box Sport Types in reservationForm
	// Tworzenie atrybutu dla modelu. Ten atrybut jest 
	// reprezentowany jako lista rozwijana z dyscyplinami w widoku reservationForm
    @ModelAttribute("sportTypes")
    public List<SportType> populateSportTypes() {
        return reservationService.getAllSportTypes();
    }

	// Kontroler zawsze najpierw szuka domyślnej metody obsługi żądań GET (niezależnie od jej nazwy).
	// Tu ta metoda to setupForm, co ułatwia jej identyfikację
    @RequestMapping(method = RequestMethod.GET)
    public String setupForm(@RequestParam(required = false, value = "username")
    String username, Model model) {
	    // Tworzenie początkowego obiektu Reservation
        Reservation reservation = new Reservation();
		// Dodawanie gracza do rezerwacji, ponieważ potrzebne będzie pole typu Player
        // UWAGA - Obiekt Player jest zapisywany w sesji w obiekcie Reservation. Gdyby
        //         obiekt Reservation nie działał w sesji, obiekt Player trzeba by 
        //         było odtworzyć po przesłaniu strony
        reservation.setPlayer(new Player(username, null));
		// Dodawanie rezerwacji do modelu, co pozwala ją wyświetlić w widoku
        model.addAttribute("reservation", reservation);

        // Return view as a string
        // Based on resolver configuration the reservationForm view
        // will be mapped to a JSP in /WEB-INF/jsp/reservationForm.jsp 
        // NOTE: If the method would have a void return value, by default the method would have 
        //       looked for the same reservationForm view, since the default URL for the 
        //       controller is this same name @RequestMapping("/reservationForm")
		// Zwracanie łańcucha znaków z widokiem.
        // Na podstawie konfiguracji widok reservationForm jest wiązany
        // z plikiem JSP /WEB-INF/jsp/reservationForm.jsp 
		// UWAGA: Gdyby metoda zwracała void, domyślnie szukałaby tego samego widoku
		//        reservationForm, ponieważ domyślny adres URL kontrolera to
		//        @RequestMapping("/reservationForm")
        return "reservationForm";
    }

    // Kontroler zawsze najpierw szuka domyślnej metody obsługi żądań POST (niezależnie od jej nazwy),
	// gdy dane są przesyłane pod dany adres URL (@RequestMapping(/reservationForm)).
	// Tu ta metoda to submitForm, co ułatwia jej identyfikację	
    @RequestMapping(method = RequestMethod.POST)
	// Parametry to obiekt Reservation z modelu i obiekty BindingResult oraz SessionStatus
    public String submitForm(@ModelAttribute("reservation")
    Reservation reservation, BindingResult result, SessionStatus status) {
		// Zakończono walidację obiektu Reservation
        validator.validate(reservation, result);

        if (result.hasErrors()) {
			// Wykryto błędy. Należy zwrócić widok reservationForm
            return "reservationForm";
        } else {
			// Brak błędów. Należy dodać rezerwację.
            reservationService.make(reservation);
			// Wywołanie setComplete() powoduje zakończenie przetwarzania sesji.
			// Pozwala to na usunięcie atrybutów sesji.
            status.setComplete();

			// Przekierowanie pod adres URL reservationSuccess zdefiniowany w kontrolerze ReservationSuccessController
            return "redirect:reservationSuccess";
        }
    }
}
