// Wersja końcowa 
package com.apress.springrecipes.court.web;

import com.apress.springrecipes.court.domain.Member;
import com.apress.springrecipes.court.service.MemberService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.annotation.*;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;
import java.util.Set;


@Controller
// Wiązanie kontrolera z wszystkimi adresami URL z członem /member/* 
// Początkowy widok jest określany na podstawie nazwy zwracanej przez domyślną metodę obsługi żądań GET.
@RequestMapping("/member/*")
@SessionAttributes("guests")
public class MemberController {
    // Walidator JSR-303
    private static Validator validator;
    private MemberService memberService;

	// Podłączanie dostępnych w kontekście aplikacji usługi i walidatora w konstruktorze
    @Autowired
    public MemberController(MemberService memberService) {
        this.memberService = memberService;

        // Inicjowanie walidacji JSR-303
        ValidatorFactory validatorFactory = Validation.buildDefaultValidatorFactory();
        validator = validatorFactory.getValidator();
    }

	// Kontroler zawsze najpierw szuka domyślnej metody obsługi żądań GET (niezależnie od jej nazwy).
	// Tu ta metoda to setupForm, co ułatwia jej identyfikację
    @RequestMapping(method = RequestMethod.GET)
    // Metoda przyjmuje parametr Model, aby ustawić obiekt typu Member
    public String setupForm(Model model) {
        model.addAttribute("member", new Member());
        model.addAttribute("guests", memberService.list());

        // Zwracanie widoku memberList. Jest on wiązany z plikiem
        // /WEB-INF/jsp/memberList.jsp
        return "memberList";
    }

	// Kontroler zawsze najpierw szuka domyślnej metody obsługi żądań POST (niezależnie od jej nazwy),
	// gdy dane są przesyłane pod dany adres URL (@RequestMapping(/member/*)).
	// Tu ta metoda to submitForm, co ułatwia jej identyfikację	
    @RequestMapping(method = RequestMethod.POST)
    public String submitForm(@ModelAttribute("member")
    Member member, BindingResult result, Model model) {
        // Pobieranie błędów JSR-303 (walidacja za pomocą adnotacji jest ustawiona w klasie Member) 
        Set<ConstraintViolation<Member>> violations = validator.validate(member);

        // Przejście w pętli po możliwych błędach
        for (ConstraintViolation<Member> violation : violations) {
            String propertyPath = violation.getPropertyPath().toString();
            String message = violation.getMessage();
            // Dodawanie błędów JSR-303 do BindingResult, aby Spring mógł je wyświetlić w widoku za pomocą obiektu FieldError
            result.addError(new FieldError("member", propertyPath, "Błąd: " + propertyPath + "(" + message + ")"));
        }

        // Wywołanie JSR-303 violations.size() == 0 lub standardowe wywołanie Springa 
		// result.hasErrors(), które obecnie uwzględnia błędy JSR-303
        if (!result.hasErrors()) {
            // Brak błędów
            memberService.add(member);
            model.addAttribute("guests", memberService.list());

            // Zauważ, że metoda SessionStatus.setComplete(); NIGDY nie jest wywoływana.
			// Pozwala to dodawać elementy do listy memberList, ponieważ jest ona przechowywana
			// w sesji. Wywołanie SessionStatus.setComplete() spowodowałoby usunięcie
			// wartości z listy memberList.
			// Zwracany jest widok memberList. Jest on wiązany z plikiem
            // /WEB-INF/jsp/memberList.jsp
            return "memberList";
        } else {
            // Zwracanie widoku memberList, dzięki czemu użytkownik może poprawić błędy. Ten widok
            // jest wiązany z plikiem /WEB-INF/jsp/memberList.jsp
            return "memberList";
        }
    }

    // Metoda powiązana z adresem URL /member/remove
    @RequestMapping("remove")
    public String removeMember(@RequestParam("memberName")
    String memberName) {
        memberService.remove(memberName);
		
		// Użycie przekierowania, co powoduje odświeżenie listy.
        // Ponieważ w kontrolerze używany jest symbol wieloznaczny, można
		// zastosować dowolny adres URL. Tu jest to ':', (czemu odpowiada /member/)
        return "redirect:";
    }
}
