package com.apress.springrecipes.calculator;

import java.util.Arrays;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.springframework.core.annotation.Order;

@Aspect
@Order(1)
public class CalculatorLoggingAspect {

    private Log log = LogFactory.getLog(this.getClass());

    @Before("CalculatorPointcuts.loggingOperation()")
    public void logBefore(JoinPoint joinPoint) {
        log.info("Metoda " + joinPoint.getSignature().getName()
                + "() rozpoczyna pracę. Argumenty: " + Arrays.toString(joinPoint.getArgs()));
    }

    @AfterReturning(
        pointcut = "CalculatorPointcuts.loggingOperation()",
        returning = "result")
    public void logAfterReturning(JoinPoint joinPoint, Object result) {
        log.info("Metoda " + joinPoint.getSignature().getName()
                + "() kończy pracę. Wynik: " + result);
    }

    @AfterThrowing(
        pointcut = "CalculatorPointcuts.loggingOperation()",
        throwing = "e")
    public void logAfterThrowing(JoinPoint joinPoint,
            IllegalArgumentException e) {
        log.error("Niedozwolony argument " + Arrays.toString(joinPoint.getArgs())
                + " w " + joinPoint.getSignature().getName() + "()");
    }

    @Around("execution(* *.*(..))")
    public Object logAround(ProceedingJoinPoint joinPoint) throws Throwable {
        log.info("Metoda " + joinPoint.getSignature().getName()
                + "() rozpoczyna pracę. Argumenty: " + Arrays.toString(joinPoint.getArgs()));
        try {
            Object result = joinPoint.proceed();
            log.info("Metoda " + joinPoint.getSignature().getName()
                    + "() kończy pracę. Wynik: " + result);
            return result;
        } catch (IllegalArgumentException e) {
            log.error("Niedozwolony argument "
                    + Arrays.toString(joinPoint.getArgs()) + " w "
                    + joinPoint.getSignature().getName() + "()");
            throw e;
        }
    }

    @Before("execution(* *.*(..))")
    public void logJoinPoint(JoinPoint joinPoint) {
        log.info("Rodzaj punktu złączenia: "
                + joinPoint.getKind());
        log.info("Typ z sygnatury: "
                + joinPoint.getSignature().getDeclaringTypeName());
        log.info("Nazwa z sygnatury: "
                + joinPoint.getSignature().getName());
        log.info("Argumenty: "
                + Arrays.toString(joinPoint.getArgs()));
        log.info("Docelowa klasa: "
                + joinPoint.getTarget().getClass().getName());
        log.info("Ta klasa: "
                + joinPoint.getThis().getClass().getName());
    }

    @Before("CalculatorPointcuts.parameterPointcut(target, a, b)")
    public void logParameter(Object target, double a, double b) {
        log.info("Docelowa klasa: " + target.getClass().getName());
        log.info("Argumenty: " + a + ", " + b);
    }
}
