/*
  Klient WWW Twitter RFID
  Kontekst: Arduino
*/

#include <SPI.h>
#include <Ethernet.h>
#include <TextFinder.h>
#include <Wire.h>
#include <LiquidCrystal.h>
#include <SonMicroReader.h>

SonMicroReader Rfid;       // instancja biblioteki czytnika
unsigned long tag = 0;     // adres bieżącego znacznika
unsigned long lastTag = 0; // adres poprzedniego znacznika
int addressBlock = 4;      // blok pamięci znacznika do odczytu 

int state = 0;             // stan, w którym jest szkic

// poniżej wprowadź adres MAC i adres IP  swojego kontrolera.
// Adres IP będzie zależny od Twojej sieci lokalnej:
byte mac[] = { 
  0x00, 0xAA, 0xBB, 0xCC, 0xDE, 0x01 };
IPAddress ip(192,168,1,20);       // będzie używany, tylko jeśli 
                                  // DHCP nie zadziała

IPAddress server(199,59,149,200); // adres API Twittera
EthernetClient client;            // połączenie klienta 

String twitterHandle = "";     // nazwa użytkownika Tweeter
String tweet = "";             // wpis użytkownika
int tweetBufferLength;         // miejsce zarezerwowane dla wpisu 
int tweetLength = 0;           // aktualna długość wpisu

long lastRequestTime = 0;     // czas ostatniego połączenia do serwera 
int requestDelay = 15 * 1000; // czas między żądaniami HTTP 


// zainicjuj bibliotekę z numerami pinów interfejsu 
LiquidCrystal lcd(9,8, 7, 6,5, 3);
const int screenWidth = 16;  // szerokość wyświetlacza LCD w znakach
long lastScrollTime = 0;     // ostatni czas przewijania LCD
int scrollDelay = 130;       // opóźnienie między przesunięciami LCD 
int cursorPosition = 0;      // pozycja kursora na LCD

const int potVoltage = A0;   // napięcie dla potencjometru 
                             // opóźnienia przewijania 
const int potGround = A2;    // masa dla potencjometru 
                             // opóźnienia przewijania 
const int potInput = A1;     // potencjometr opóźnienia przewijania 

void setup() {
  // inicjuj komunikację szeregową i czytnik:
  Serial.begin(9600);
  // uruchom połączenie Ethernet:
  if (Ethernet.begin(mac) == 0) {
    Serial.println(F("Nie udało się skonfigurować Ethernetu przez DHCP"));
    Ethernet.begin(mac, ip);
  }
  // ustaw piny A0 i A2 jako cyfrowe wyjście 
  // i użyj ich jako zasilania i masy 
  // potencjometru prędkości przewijania:
  pinMode(potGround, OUTPUT);
  pinMode(potVoltage, OUTPUT);
  digitalWrite(potGround, LOW);
  digitalWrite(potVoltage, HIGH);

  // zarezerwuj 140 * + 2 * szerokość 
  // ekranu + 3 dodatkowe bajty na wpis Tweetera:
  tweetBufferLength = 140 + 2*screenWidth + 3;
  tweet.reserve(tweetBufferLength);
  Rfid.begin();
  // definiuj liczbę kolumn i wierszy LCD:
  lcd.begin(screenWidth,2);
  lcd.clear();
  lcd.print(F("Gotowy"));
}


void loop() {
  switch(state) {
  case 0:             // pobierz znacznik 
    tag = Rfid.selectTag();
    if (tag != 0) {
      // masz znacznik, więc wyświetl go:
      Serial.println(tag, HEX);
      state++;          // przejdź do następnego stanu 
    }
    break;
  case 1:          // odczytaj blok
    if (Rfid.authenticate(addressBlock)) {
      Serial.print(F("uwierzytelniony "));
      // odczytaj nazwę użytkownika Twittera ze znacznika:
      Rfid.readBlock(addressBlock);
      twitterHandle = Rfid.getString();
      // pokaż nazwę:
      lcd.clear();              // wyczyść poprzednią zawartość 
      lcd.setCursor(0,0);       // przenieś kursor na początek
      lcd.print(twitterHandle); // nazwa Tweetera w górnym wierszu 
      Serial.println(twitterHandle);
      state++;              // przejdź do następnego stanu 
    }
    else state = 0;         // wróć do pierwszego stanu 
    break;
  case 2:                 // połączenie z serwerem 
    // jeśli jest to nowy znacznik, lub 
    // jeśli upłynął czas opóźnienia od ostatniego 
    // wysłanego żądania HTTP:
    if (tag != lastTag ||
      millis() - lastRequestTime > requestDelay) {
      // próba połączenia:
      if (connectToServer()) {
        state++;           // przejdź do następnego stanu 
      }
      else state = 0;      // wróć do pierwszego stanu 
    }
    else state = 0;        // wróć do pierwszego stanu 
    lastTag = tag;
    break;
  case 3:                // przeczytaj odpowiedź
    tweetLength = readResponse();
    state = 0;            // wróć do pierwszego stanu
    break;
  }


  // jeśli wyświetlacz LCD nie zostało ostatnio przesunięty:
  if (tweetLength > 0 && millis() - lastScrollTime > scrollDelay) {
    // przesunięcie wyświetlacza LCD:
    scrollLongString(cursorPosition);
    // inkrementuj położenie kursora LCD:
    if (cursorPosition < tweetLength) {
      cursorPosition++;
    }
    else {
      cursorPosition = 0;
    }
    // zapisz czas ostatniego przesunięcia wyświetlacza LCD:
    lastScrollTime = millis();
  }

  // aktualizuj szybkość przewijania z drugiego potencjometru:
  int sensorReading = analogRead(potInput);
  // mapuj do opóźnienia przewijania 100 - 300 ms:
  scrollDelay = map(sensorReading, 0, 1023, 100, 300);
}


// ta metoda bierze podciąg z długiego ciągu wpisu 
// Tweetera do wyświetlenia na ekranie 

void scrollLongString(int startPos) {
  String shortString = ""; // ciąg do wyświetlenia na ekranie 
  // upewnij się, że pozostał wystarczająco długi ciąg:
  if (startPos < tweetLength - screenWidth) {
    // pobierz podciąg 16-znakówowy:
    shortString = tweet.substring(startPos, startPos + screenWidth);
  }
  //odśwież ekran LCD:
  lcd.clear();              // wyczyść poprzednią zawartość 
  lcd.setCursor(0,0);       // przenieś kursor na początek 
                            // górnego wiersza 
  lcd.print(twitterHandle); // nazwa użytkownika Tweetera 
                            // do górnego wiersza 
  lcd.setCursor(0,1);       // przenieś kursor na początek 
                            // dolnego wiersza 
  lcd.print(shortString);   // wpis Tweetera przewijany na dole 
}


// ta metoda łączy się z serwerem 
// i tworzy żądanie HTTP:

boolean connectToServer() {
  // zapisz czas tej próby połączenia:
  lastRequestTime = millis();
  // próba połączenia:
  Serial.println(F("łączenie z serwerem"));
  if (client.connect(server, 80)) {
    Serial.println(F("tworzenie żądania HTTP"));
    // stwórz żądanie HTTP GET:
    client.print(F("GET /1/statuses/user_timeline.xml?screen_name="));
    client.print(twitterHandle);
    client.println(F(" HTTP/1.1"));
    client.println(F("Host:api.twitter.com"));
    client.println();
    return true;
  }
  else {
    Serial.print(F("próba połączenia nie powiodła się"));
    return false;
  }
}

int readResponse() {
  char tweetBuffer[141]; // 140 znaków + 1 ekstra

  int result = 0;
  // jeśli są dostępne bajty z serwera:
  while (client.connected()) {
    if (client.available()) {
      // stwórz instancję TextFinder do wyszukiwania odpowiedzi:
      TextFinder response(client);
      // jeśli odpowiedź z serwera zawiera <text>:
      response.getString("<text>", "</text>", tweetBuffer, 141);
      // wyświetl ciąg wpisu Tweetera:
      Serial.println(tweetBuffer);
      // stwórz ciąg znaków z odstępem na obu końcach:
      tweet = "                " + String(tweetBuffer)
        + "                ";
      result = tweet.length();
      // ważny jest tylko wpis Tweetera:
      client.stop();
    }
  }
  return result;
}

