/*
  Klient z joystickiem
  Kontekst: Arduino

  Ten program umożliwia Arduino sterowanie 
  jedną rakietką w sieciowej grze Pong.
*/

#include <SPI.h>
#include <Ethernet.h>

byte mac[] = { 0x00, 0xAA, 0xBB, 0xCC, 0xDE, 0x01 };
IPAddress ip(192,168,1,20);

// Wprowadź adres IP komputera, 
// na którym uruchomisz serwer Pong:
IPAddress server(192,168,1,100);

const int connectButton = 2;       // przycisk połączenia-rozłączenia 
const int connectionLED = 3;       // dioda LED wskazująca, czy 
                                   // jesteś podłączony
const int leftLED = 4;             // dioda LED wskazująca ruch w lewo 
const int rightLED = 5;            // dioda LED wskazująca ruch w prawo 
const int left = 880;              // próg wychylenia joysticka w lewo 
const int right = 891;             // próg wychylenia joysticka w prawo
const int sendInterval = 20;       // minimalny czas pomiędzy 
                                   // wiadomościami z serwera
const int debounceInterval = 15;   // używane do wygładzenia odczytów
                                   // przycisku 

EthernetClient client;            // instancja  klasy EthernetClient dla                 
                                  // połączenia
int lastButtonState = 0;          // poprzedni stan przycisku 
long lastTimeSent = 0;            // sygnatura czasowa ostatniej 
                                  // wiadomości przesłanej do serwera 
void setup()
{
   // zainicjuj port szeregowy i Ethernet:
   Ethernet.begin(mac, ip);
   Serial.begin(9600);
   // zainicjuj cyfrowe wejścia i wyjścia:
   pinMode(connectButton, INPUT);
   pinMode(connectionLED, OUTPUT);
   pinMode(leftLED, OUTPUT);
   pinMode(rightLED, OUTPUT);

   delay(1000);  // daj tarczy Ethernet czas na konfigurację 
   Serial.println("Uruchamianie");
}

void loop()
{
   // zapisz bieżący czas w milisekundach:
   long currentTime = millis();
   // sprawdź, czy przycisk jest wciśnięty:
   boolean buttonPushed = buttonRead(connectButton);

   // jeśli przycisk właśnie został wciśnięty:
   if (buttonPushed) {
      // jeśli klient jest podłączony, rozłącz:
      if (client.connected()) {
         Serial.println("rozłączanie");
         client.print("x");
         client.stop();
      } // jeśli klient jest rozłączony, spróbuj się połączyć:
      else {
         Serial.println("łączenie");
         client.connect(server, 8080);
      }
   }

// jeśli klient jest podłączony i upłynął interwał wysyłania:
   if (client.connected() && (currentTime - lastTimeSent > sendInterval)) {
      // odczytaj położenie joysticka 
      // i wyślij wiadomość, jeśli to konieczne:
      int sensorValue = analogRead(A0);
      if (sensorValue < left) {    // ruch w lewo
         client.print("l");
         digitalWrite(leftLED, HIGH);
      }

      if (sensorValue > right) {    // ruch w prawo
         client.print("p");
         digitalWrite(rightLED, HIGH);
      }
      // jeśli jesteś po środku, wyłącz diody LED:
      if (left < sensorValue && sensorValue < right) {
         digitalWrite(rightLED, LOW);
         digitalWrite(leftLED, LOW);
      }

      // zapisz bieżący czas jako czas wysłania ostatniej wiadomości:
      lastTimeSent = currentTime;
   }
   //ustaw diodę LED połączenia na podstawie stanu połączenia:
   digitalWrite(connectionLED, client.connected());
}
/*
 // jeśli są przychodzące dane z kllienta wypisz je szeregowo:
 if (client.available()) {
 char c = client.read();
 Serial.write(c);
 }

 */

// metoda odczytuje przycisk, aby zobaczyć, 
// czy jego stan właśnie się zmienił od niskiego 
// do wysokiego, i eliminuje odbicia przycisku 
// w przypadku zakłóceń elektrycznych:
boolean buttonRead(int thisButton) {
   boolean result = false;
   // tymczasowy stan przycisku:
   int currentState = digitalRead(thisButton);
   // końcowy stan przycisku:
   int buttonState = lastButtonState;
   // pobierz bieżący czas interwału eliminacji odbicia:
   long lastDebounceTime = millis();

   while ((millis() - lastDebounceTime) < debounceInterval) {
      // odczytaj stan przełącznika do zmiennej lokalnej:
      currentState = digitalRead(thisButton);
      // jeśli przycisk zmieni się z powodu zakłóceń:
      if (currentState != buttonState) {
         // resetuj czas eliminacji odbicia
         lastDebounceTime = millis();
      }

      // niezależnie od wartości odczytu, taki stan 
      // ma trwać dłużej niż opóźnienie eliminacji 
      // odbicia, więc ustaw ją jako faktyczny bieżący stan:
      buttonState = currentState;
   }
   // jeśli stan przycisku zmienił się na wysoki:
   if(buttonState != lastButtonState && buttonState == HIGH) {
      result = true;
   }

   // zapisz bieżący stan do następnego razu:
   lastButtonState = buttonState;
   return result;
}

