/*
  Serwer WWW czujnika gazu
  Kontekst: Arduino
*/

// dodaj biblioteki Ethernet i SPI:
#include <SPI.h>
#include <Ethernet.h>

// inicjuj instancję serwera:
Server server(80);

// adres Ethernet MAC i adres IP serwera:
byte mac[] = {
  0x00, 0xAA, 0xBB, 0xCC, 0xDE, 0x01 };
IPAddress ip(192,168,1,20);

String requestLine = "";       // przychodzące żądanie HTTP klienta

const int packetLength = 22;  // wiadomość danych XBee ma 
                              // 22 bajty długości
int dataPacket[packetLength]; // tablica do przechowywania danych XBee 
int byteCounter = 0;          // licznik bajtów odebranych z XBee 

//zmienne do obliczania średniej czujnika:
const int averageInterval = 10 * 1000; // czas między średnimi, 
                                       // w sekundach 
long sensorTotal = 0;          // używane do obliczania średniej             
                               // wartości czujnika 
float averageVoltage = 0.0;    // wartość średnia, w woltach 
long lastAverageTime = 0;      // kiedy była pobrana ostatnia średnia 
int readingCount = 0;          // odczyty od ostatniej średniej 

void setup() {
  // otwórz połączenie Ethernet i serwer:
  Ethernet.begin(mac, ip);
  server.begin();

  // rozpocznij komunikację szeregową:
  Serial.begin(9600);
}

void loop() {

  // nasłuchuj przychodzących danych szeregowych:
  if (Serial.available() > 0) {
    listenToSerial();
  }

  // nasłuchuj przychodzących klientów :
  Client client = server.available();
  if (client) {
    listenToClient(client);
  }

  // oblicz średnią odczytów po <averageInterval> sekund
  long now = millis();
  if (now - lastAverageTime > averageInterval) {
    averageVoltage = getAverageReading();
    Serial.println(averageVoltage);
  }
}


// ta metoda odbiera dane szeregowe i zapisuje je
// w tablicy:
void listenToSerial() {
  // odczytaj przychodzący bajt:
  int inByte = Serial.read();
  // początek nowego pakietu to 0x7E:
  if (inByte == 0x7E ) {
    // sparsuj ostatni pakiet i pobierz odczyt:
    int thisReading = parseData();
    // dodaj ten odczyt to sumy dla uśrednienia:
    sensorTotal = sensorTotal + thisReading;
    // zwiększ licznik odczytów:
    readingCount++;
    // opróżnij tablicę danych:
    for (int thisByte = 0; thisByte < packetLength; thisByte++) {
      dataPacket[thisByte] = 0;
    }
    // zresetuj licznik przychodzącyh bajtów:
    byteCounter = 0; 
  } 
  // jeśli licznik bajtow jest mniejszy niż długość pakietu danych,
  // dodaj ten bajt do tablicy danych:
  if (byteCounter < packetLength) {
    dataPacket[byteCounter] = inByte;
    // zwiększ licznik bajtów:
    byteCounter++;
  }  
}

// Ta metoda parsuje format danych XBee
// i zwraca średnią wartość zensora dla pakietu
int parseData() {
  int adcStart = 11;                     // odczyt ADC zaczyna się na bajcie 12
  int numSamples = dataPacket[8];        // liczba próbek w pakiecie
  int total = 0;                         // suma wszystkich odczytów ADC

  // odczytaj adres. Jest to wartość dwubajtowa, więc dodaj
  // te dwa bajty następująco:
  int address = dataPacket[5] + dataPacket[4] * 256;

  // odczytaj <numSamples> 10-bitowych wartości analogowych, po dwie na raz
  // ponieważ każdy odczyt ma długość dwóch bajtów:
  for (int thisByte = 0; thisByte < numSamples * 2;  thisByte=thisByte+2) {
    // wartość 10-bitowa = bajt wysoki * 256 + bajt niski:
    int thisSample = (dataPacket[thisByte + adcStart] * 256) + 
      dataPacket[(thisByte + 1) + adcStart];
    // dodaj wynik do sumy aby później ją uśrednić:
    total = total + thisSample;
  }
  // uśrednij wynik:
  int average = total / numSamples;
  return average;
}


// ta metoda nasłuchuje żądań od klienta HTTP:
void listenToClient( Client thisClient) {
  while (thisClient.connected()) {
    if (thisClient.available()) {
      // odczytaj bajt:
      char thisChar = thisClient.read();
      //dla wszystkich innych znaków 
      // zwiększ długość wiersza o 1:
      requestLine = requestLine + thisChar;
      // jeśli otrzymasz znak wysuwu wiersza 
      // i wiersz żądania zawiera tylko znak 
      // nowego wiersza, to wniosek zakończył się:
      if (thisChar == '\n' && requestLine.length() < 2) {
        // wyślij odpowiedź HTTP:
        makeResponse(thisClient);
        break;
      }
      // jeśli otrzymasz nowy wiersz lub powrót karetki,
      //jesteś na końcu wiersza:
      if (thisChar == '\n' || thisChar == '\r') {
        // wyczyść ostatni wiersz:
        requestLine = "";
      }
    }
  }
  // daj czas przeglądarce WWW na odebranie danych:
  delay(1);
  // zamknij połączenie:
  thisClient.stop();
}


// Ta metoda wysyła odpowiedź do klienta:
void makeResponse(Client thisClient) {
  // przeczytaj aktualne średnie napięcie:
  float thisVoltage = getAverageReading();

  // wyświetl nagłówek HTTP:
  thisClient.print("HTTP/1.1 200 OK\n");
  thisClient.print("Content-Type: text/html\n\n");
  // wyświetl dokument HTML:
  thisClient.print("<html><head><meta http-equiv=\"refresh\" content=\"3\">");
  thisClient.println("</head>");
  // jeśli odczyt jest dobry, wyświetl go:
  if (thisVoltage > 0.0) {
    thisClient.print("<h1>odczyt = ");
    thisClient.print(thisVoltage);
    thisClient.print(" woltów</h1>");
  } // jeśli odczyt nie jest dobry, wyświetl to:
  else {
    thisClient.print("<h1>Brak odczytu</h1>");
  }
  thisClient.println("</html>\n\n");
}

// ta metoda oblicza średnią odczytów
float getAverageReading() {
  int thisAverage = sensorTotal / readingCount;
  // konwertuj na napięcie:
  float voltage = 3.3 * (thisAverage / 1024.0);
  // resetuj licznik odczytu i sumy do następnego razu:
  readingCount = 0;
  sensorTotal = 0;
  // zapisz czas, kiedy została pobrana średnia:
  lastAverageTime = millis();
  // zwróć wynik:
  return voltage;
}

