/*
  Parser GPS
  Kontekst: Processing

  Ten program bierze dane szeregowe NMEA 0183 
  i parsuje datę, czas, szerokość i długość geograficzną 
  przy użyciu zdania GPRMC.
*/

// import biblioteki szeregowej:
import processing.serial.*;

Serial myPort;             // port szeregowy
float latitude = 0.0;      // szerokość geograficzna odczytywana 
                           // w stopniach 
String northSouth = "N";   // północ czy południe?
float longitude = 0.0;     // długość geograficzna odczytywana 
                           // w stopniach 
String eastWest = "W";     // wschód czy zachód?
float heading = 0.0;       // pozycja w stopniach 

int hrs, mins, secs;       // jednostki czasu
int currentDay, currentMonth, currentYear;

int satellitesInView = 0; // satelity w polu widzenia
int satellitesToFix = 0;  // satelity używane do obliczania poprawki 

float textX = 50;         // położenie tekstu na ekranie 
float textY = 30;

void setup() {
  size(400, 400);    // rozmiar okna 

  // ustawienia rysunku:
  noStroke();
  smooth();

  // lista wszystkich dostępnych portów szeregowych 
  println(Serial.list());

  // otwórz port, którego używasz dla urządzenia 
  // Bluetooth; może być bliżej końca Twojej listy 
  // portów szeregowych:
  String portName = Serial.list()[6];
  myPort = new Serial(this, portName, 9600);

  // odczytuj bajty do bufora, aż dotrzesz do 
  // znaku powrotu karetki (ASCII 13):
  myPort.bufferUntil('\r');
}

void draw() {
  // ciemnoniebieskie tło:
  background(#0D1133);

  // jasnoniebieski tekst:
  fill(#A3B5CF);

  // sklej cały tekst w jeden duży ciąg:

  // wyświetl datę i godzinę ze zdania GPS 
  // jako MM/DD/YYYY, HH:MM:SS GMT
  // — wszystkie liczby są formatowane za pomocą funkcji
  // nf(), tak aby były 2- lub 4-cyfrowe:
  String displayString = nf(currentMonth, 2)+ "/"+ nf(currentDay, 2)
    + "/"+ nf(currentYear, 4) + ", " + nf(hrs, 2)+ ":"
    + nf(mins, 2)+ ":"+ nf(secs, 2) + " GMT\n";

  // wyświetl położenie ze zdania GPS:
  displayString = displayString + latitude + " " + northSouth + ", "
    + longitude + " " + eastWest + "\n";

  // wyświetl nagłówek:
  displayString = displayString + "kierunek " + heading + " stopni\n";

  // pokaż informacje na temat używanych satelitów:
  displayString = displayString + "satelity w polu widzenia: "
    + satellitesInView + "\n";
  displayString = displayString + "satelity używane do obliczania pozycji: "
    + satellitesToFix;
  text(displayString, textX, textY);

  // narysuj strzałkę przy użyciu pozycji:
  drawArrow(heading);
}


void serialEvent(Serial myPort) {
  // przeczytaj bufor szeregowy:
  String myString = myPort.readStringUntil('\n');

  // jeśli masz jakieś bajty inne niż wysuw wiersza, 
  // parsuj je:
  if (myString != null) {
    print(myString);
    parseString(myString);
  }
}

void parseString (String serialString) {
  // podziel ciąg w miejscach przecinków:
  String items[] = (split(serialString, ','));

  // jeśli pierwszy element w zdaniu jest 
  // identyfikatorem, parsuje resztę
  if (items[0].equals("$GPRMC")) {
    // $GPRMC zwraca czas, datę, pozycję, 
    // kurs i prędkość
    getRMC(items);
  }

  if (items[0].equals("$GPGGA")) {
    // $GPGGA zwraca godzinę, datę, pozycję, 
    // używane satelity
    getGGA(items);
  }

  if (items[0].equals("$GPGSV")) {
    // $GPGSV zwraca liczbę satelitów w polu widzenia
    satellitesInView = getGSV(items);
  }
}

void getRMC(String[] data) {
  // przenieś elementy z ciągu do zmiennych:
  int time = int(data[1]);
  // pierwsze dwie cyfry czasu to godzina:
  hrs = time/10000;
  //drugie dwie cyfry czasu to minuty:
  mins = (time % 10000)/100;
  // ostatnie dwie cyfry czasu to sekundy:
  secs = (time%100);

  // jeśli masz prawidłowy odczyt, parsuj resztę:
  if (data[2].equals("A")) {
    latitude = minutesToDegrees(float(data[3]));

    northSouth = data[4];
    longitude = minutesToDegrees(float(data[5]));
    eastWest = data[6];
    heading = float(data[8]);
    int date = int(data[9]);
    // ostatnie dwie cyfry daty to rok. Dodaj także wiek:
    currentYear = date % 100 + 2000;
    //drugie dwie cyfry daty to miesiące:
    currentMonth = (date % 10000)/100;
    // pierwsze dwie cyfry daty to dzień:
    currentDay = date/10000;
  }
}

void getGGA(String[] data) {
  // przenieś elementy z ciągu do zmiennych:
  int time = int(data[1]);
  // pierwsze dwie cyfry czasu to godziny:
  hrs = time/10000;
  // drugie dwie cyfry czasu to minuty:
  mins = (time % 10000)/100;
  // ostatnie dwie cyfry czasu to sekundy:
  secs = (time % 100);

  // jeśli masz prawidłowy odczyt, parsuje resztę:
  if (data[6].equals("1")) {
    latitude = minutesToDegrees(float(data[2]));
    northSouth = data[3];
    longitude = minutesToDegrees(float(data[4]));
    eastWest = data[5];
    satellitesToFix = int(data[7]);
  }
}

int getGSV(String[] data) {
  int satellites = int(data[3]);
  return satellites;
}

float minutesToDegrees(float thisValue) {
  // uzyskaj całkowitą część pomiaru stopni:
  int wholeNumber = (int)(thisValue / 100);
  // uzyskaj częścić ułamkową i przekonwertuj 
  // minuty na ułamek dziesiętny:
  float fraction = (thisValue - ( wholeNumber ) * 100) / 60;
  // połącz obie części i zwróć je:
  float result = wholeNumber + fraction;
  return result;
}


void drawArrow(float angle) {
  // przesuń wszystko, co zostanie narysowane dalej 
  // tak, że (0,0) jest wyśrodkowany na ekranie:
  translate(width/2, height/2);

  // narysuj okrąg w jasnoniebieskim kolorze:
  fill(80,200,230);
  ellipse(0,0,50,50);

  // nadaj strzałce czarny kolor:
  fill(0);
  // obróć za pomocą kierunku:
  rotate(radians(angle));

  // narysuj strzałkę, środek strzałki jest w punkcie (0,0):
  triangle(-10, 0, 0, -20, 10, 0);
  rect(-2,0, 4,20);
}

