/*
   Wydobywanie danych ze strony WWW AirNow
   Kontekst: Arduino
*/

#include <SPI.h>
#include <Ethernet.h>
#include <TextFinder.h>

const int connectedLED = 2;        // sygnalizuje połączenie TCP 
const int successLED = 3;          // sygnalizuje pomyślny odczyt 
const int resetLED = 4;            // sygnalizuje resetowanie Arduino 
const int disconnectedLED = 5;     // sygnalizuje połączenie z serwerem 
const int meterPin = 9;            // kontroluje miernik VU
const int meterMin = 0;            // minimalna wartość miernika
const int meterMax = 200;          // maksymalna wartość miernika
const int AQIMax = 200;            // maksymalna wartość indeksu 
                                   //zanieczyszczenia powietrza 
const int requestInterval = 10000; // opóźnienie między 
                                   // aktualizacjami serwera 

// Zmień adresy MAC i IP tak, aby pasowały do Twojego 
// urządzenia i serwera. Adres IP zależy od
// Twojej sieci lokalnej:
byte mac[] = { 0x00, 0xAA, 0xBB, 0xCC, 0xDE, 0x01 };
IPAddress ip(192,168,1,20);
IPAddress server(208,201,239,101);

// Zainicjuj bibliotekę klienta Ethernet 
EthernetClient client;

boolean requested;           // czy wysłano żądanie 
long lastAttemptTime = 0;    // czas ostatniego połączenia z serwerem 
int airQuality = 0;          // wartość AQI 
boolean meterIsSet = false;  // czy miernik jest ustawiony

void setup() {
   // uruchom połączenie Ethernet:
   Ethernet.begin(mac, ip);
   // uruchom bibliotekę szeregową:
   Serial.begin(9600);
   // ustaw piny wszystkich diod LED:
   pinMode(connectedLED, OUTPUT);
   pinMode(successLED, OUTPUT);
   pinMode(resetLED, OUTPUT);
   pinMode(disconnectedLED, OUTPUT);
   pinMode(meterPin, OUTPUT);

   // poczekaj sekundę na zainicjowanie tarczy Ethernet:
   delay(1000);
   // zamigaj diodą resetowania:
   blink(resetLED, 3);
   // spróbuj się połączyć:
   connectToServer();
}

void loop()
{
   // jeśli jesteś połączony, zapisz wszystkie 
   // przychodzące bajty do ciągu wejściowego:
   if (client.connected()) {
      if (!requested) {
         requested = makeRequest();
      }
      else {
         // stwórz instancję TextFinder do przeszukania odpowiedzi:
         TextFinder response(client);
         // sprawdź, czy odpowiedź z serwera zawiera wartość AQI:
         if(response.find("Air Quality:")) {
            // skonwertuj pozostałą część na liczbę całkowitą:
            airQuality = response.getValue();
            // ustaw miernik:
            meterIsSet = setMeter(airQuality);
         }
      }
   }
   else if (millis() - lastAttemptTime > requestInterval) {
      // jeśli nie masz połączenia i od ostatniego
      // połączenia upłynęły dwie minuty, wówczas 
      // spróbuj połączyć się ponownie:
      client.stop();
      connectToServer();
   }
   // ustaw diody LED sygnalizujące stan:
   setLeds();
}

void connectToServer() {
   // wyczyść stan miernika:
   meterIsSet = false;
   // spróbuj się połączyć i odczekaj milisekundę:
   Serial.println("łączenie...");
   if (client.connect(server, 80)) {
      requested = false;
   }
   // zapisz czas tej próby połączenia:
   lastAttemptTime = millis();
}

boolean makeRequest() {
   Serial.println("żądanie...");
   // utwórz żądanie HTTP GET i uzupełnij ścieżkę do 
   // skryptu PHP na Twoim serwerze:
   client.println("GET /~myaccount/scraper.php HTTP/1.1");
   // wprowadź nazwę swojego serwera:
   client.print("HOST:example.com\n\n");
   // aktualizacja stanu programu:
   client.println();
   return true;
}

boolean setMeter(int thisLevel) {
   Serial.println("setting meter...");
   boolean result = false;
   // skaluj wynik do zakresu, jakiego używa miernik:
   int meterSetting = map(thisLevel, 0, AQIMax, meterMin, meterMax);
   // ustaw miernik:
   analogWrite(meterPin, meterSetting);
   if (meterSetting > 0) {
      result = true;
   }
   return result;
}

void setLeds() {
   // diody sygnalizacji połączenia 
   // i braku połączenia mogą po prostu
   // użyć statusu connected() klienta:
   digitalWrite(connectedLED, client.connected());
   digitalWrite(disconnectedLED, !client.connected());
   // stan diody sukcesu zależy od tego, 
   // czy odczyt zakończył się sukcesem:
   digitalWrite(successLED, meterIsSet);
}

void blink(int thisPin, int howManyTimes) {
   // zamigaj diodą resetowania:
   for (int blinks=0; blinks< howManyTimes; blinks++) {
      digitalWrite(thisPin, HIGH);
      delay(200);
      digitalWrite(thisPin, LOW);
      delay(200);
   }
}
