package sample.spring.chapter11.web;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.propertyeditors.CustomDateEditor;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.SessionAttributes;
import org.springframework.web.bind.support.SessionStatus;
import org.springframework.web.servlet.ModelAndView;

import sample.spring.chapter11.domain.FixedDepositDetails;
import sample.spring.chapter11.service.FixedDepositService;

@Controller
@RequestMapping(value = "/fixedDeposit")
@SessionAttributes(value = { "newFixedDepositDetails",
		"editableFixedDepositDetails" })
public class FixedDepositController {
	private static Logger logger = Logger
			.getLogger(FixedDepositController.class);

	@Autowired
	private FixedDepositService fixedDepositService;

	@RequestMapping(value = "/list", method = RequestMethod.GET)
	@ModelAttribute(value = "fdList")
	public List<FixedDepositDetails> listFixedDeposits() {
		logger.info("Metoda listFixedDeposits(): pobranie listy lokat.");
		return fixedDepositService.getFixedDeposits();
	}

	@ModelAttribute(value = "newFixedDepositDetails")
	public FixedDepositDetails getNewFixedDepositDetails() {
		FixedDepositDetails fixedDepositDetails = new FixedDepositDetails();
		fixedDepositDetails.setEmail("Należy podać prawidłowy adres e-mail.");
		logger.info("Metoda getNewFixedDepositDetails(): zwrot nowego egzemplarza FixedDepositDetails.");
		return fixedDepositDetails;
	}

	@RequestMapping(params = "fdAction=createFDForm", method = RequestMethod.POST)
	public String showOpenFixedDepositForm() {
		logger.info("Metoda showOpenFixedDepositForm(): wyświetlenie formularza umożliwiającego założenie nowej lokaty.");
		return "createFixedDepositForm";
	}

	@RequestMapping(params = "fdAction=create", method = RequestMethod.POST)
	public String openFixedDeposit(
			@ModelAttribute(value = "newFixedDepositDetails") FixedDepositDetails fixedDepositDetails,
			BindingResult bindingResult, SessionStatus sessionStatus) {

		if (bindingResult.hasErrors()) {
			logger.info("Metoda openFixedDeposit(): błędy w trakcie weryfikacji - ponowne wyświetlenie formularza zakładania nowej lokaty.");
			return "createFixedDepositForm";
		} else {
			fixedDepositService.saveFixedDeposit(fixedDepositDetails);
			sessionStatus.setComplete();
			logger.info("Metoda openFixedDeposit(): zapisano informacje o lokacie. Przekierowanie w celu wyświetlenia listy lokat.");
			return "redirect:/fixedDeposit/list";
		}
	}

	@RequestMapping(params = "fdAction=edit", method = RequestMethod.POST)
	public String editFixedDeposit(
			@ModelAttribute("editableFixedDepositDetails") FixedDepositDetails fixedDepositDetails,
			BindingResult bindingResult, SessionStatus sessionStatus) {
		System.out.println("error " + bindingResult.getAllErrors());
		new FixedDepositDetailsValidator().validate(fixedDepositDetails,
				bindingResult);

		if (bindingResult.hasErrors()) {
			logger.info("Metoda editFixedDeposit(): błędy w trakcie weryfikacji - ponowne wyświetlenie formularza zakładania nowej lokaty.");
			return "editFixedDepositForm";
		} else {
			fixedDepositService.editFixedDeposit(fixedDepositDetails);
			sessionStatus.setComplete();
			logger.info("Metoda editFixedDeposit(): lokata została zmodyfikowana. Przekierowanie w celu wyświetlenia listy lokat.");
			return "redirect:/fixedDeposit/list";
		}
	}

	@RequestMapping(params = "fdAction=close", method = RequestMethod.GET)
	public String closeFixedDeposit(
			@RequestParam(value = "fixedDepositId") int fdId) {
		fixedDepositService.closeFixedDeposit(fdId);
		logger.info("Metoda closeFixedDeposit(): lokata została zamknięta. Przekierowanie w celu wyświetlenia listy lokat.");
		return "redirect:/fixedDeposit/list";
	}

	@RequestMapping(params = "fdAction=view", method = RequestMethod.GET)
	public ModelAndView viewFixedDepositDetails(
			@RequestParam(value = "fixedDepositId") int fixedDepositId) {
		FixedDepositDetails fixedDepositDetails = fixedDepositService
				.getFixedDeposit(fixedDepositId);
		Map<String, Object> modelMap = new HashMap<String, Object>();
		modelMap.put("editableFixedDepositDetails", fixedDepositDetails);
		logger.info("Metoda viewFixedDepositDetails(): informacje o lokacie zostały wczytane ze źródła danych. Wyświetlenie formularza pozwalającego na edycję lokaty.");
		return new ModelAndView("editFixedDepositForm", modelMap);
	}

	@InitBinder(value = "newFixedDepositDetails")
	public void initBinder_New(WebDataBinder webDataBinder) {
		logger.info("Metoda initBinder_New(): rejestracja CustomDateEditor.");
		webDataBinder.registerCustomEditor(Date.class, new CustomDateEditor(
				new SimpleDateFormat("MM-dd-yyyy"), false));
	}

	@InitBinder(value = "editableFixedDepositDetails")
	public void initBinder_Edit(WebDataBinder webDataBinder) {
		logger.info("Metoda initBinder_Edit(): rejestracja CustomDateEditor.");
		webDataBinder.registerCustomEditor(Date.class, new CustomDateEditor(
				new SimpleDateFormat("MM-dd-yyyy"), false));
		webDataBinder.setDisallowedFields("id");
	}
}