package sample.spring.chapter12.web;

import java.util.List;

import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import sample.spring.chapter12.domain.FixedDepositDetails;
import sample.spring.chapter12.exception.ValidationException;
import sample.spring.chapter12.service.FixedDepositService;

@Controller
@RequestMapping(value = "/fixedDeposits")
public class FixedDepositController {
	private static Logger logger = Logger
			.getLogger(FixedDepositController.class);
	
	@Autowired
	private FixedDepositService fixedDepositService;

	@RequestMapping(method = RequestMethod.GET)
	public ResponseEntity<List<FixedDepositDetails>> getFixedDepositList() {
		logger.info("Metoda listFixedDeposits(): pobranie listy lokat.");
		return new ResponseEntity<List<FixedDepositDetails>>(
				fixedDepositService.getFixedDeposits(), HttpStatus.OK);
	}

	@RequestMapping(method = RequestMethod.GET, params = "id")
	public ResponseEntity<FixedDepositDetails> getFixedDeposit(
			@RequestParam("id") int id) {
		logger.info("Metoda getFixedDeposit(): pobranie lokaty o identyfikatorze "
				+ id);
		return new ResponseEntity<FixedDepositDetails>(
				fixedDepositService.getFixedDeposit(id), HttpStatus.OK);
	}

	@RequestMapping(method = RequestMethod.POST)
	public ResponseEntity<FixedDepositDetails> openFixedDeposit(
			@RequestBody FixedDepositDetails fixedDepositDetails,
			BindingResult bindingResult) {

		new FixedDepositDetailsValidator().validate(fixedDepositDetails,
				bindingResult);

		if (bindingResult.hasErrors()) {
			logger.info("Metoda openFixedDeposit(): Wystąpiły błędy podczas weryfikacji.");
			throw new ValidationException("Wystąpiły błędy podczas weryfikacji.");
		} else {
			fixedDepositService.saveFixedDeposit(fixedDepositDetails);
			logger.info("Metoda openFixedDeposit(): zapisano informacje o lokacie.");
			return new ResponseEntity<FixedDepositDetails>(fixedDepositDetails,
					HttpStatus.CREATED);
		}
	}

	@RequestMapping(method = RequestMethod.PUT)
	public ResponseEntity<FixedDepositDetails> editFixedDeposit(
			@RequestBody FixedDepositDetails fixedDepositDetails,
			@RequestParam("id") int fixedDepositId, BindingResult bindingResult) {

		fixedDepositDetails.setId(fixedDepositId);
		new FixedDepositDetailsValidator().validate(fixedDepositDetails,
				bindingResult);

		if (bindingResult.hasErrors()) {
			logger.info("Metoda editFixedDeposit(): Wystąpiły błędy podczas weryfikacji.");
			throw new ValidationException("Wystąpiły błędy podczas weryfikacji.");
		} else {
			fixedDepositService.editFixedDeposit(fixedDepositDetails);
			logger.info("Metoda editFixedDeposit(): zmodyfikowano dane lokaty.");
			return new ResponseEntity<FixedDepositDetails>(fixedDepositDetails,
					HttpStatus.OK);
		}
	}

	@RequestMapping(method = RequestMethod.DELETE)
	@ResponseStatus(value = HttpStatus.OK)
	public HttpEntity<String> closeFixedDeposit(
			@RequestParam(value = "id") int fdId) {
		fixedDepositService.closeFixedDeposit(fdId);
		logger.info("Metoda closeFixedDeposit(): zamknięto lokatę.");
		return new HttpEntity<String>("Zamknięto lokatę.");
	}

	@ExceptionHandler(ValidationException.class)
	@ResponseBody
	@ResponseStatus(value = HttpStatus.BAD_REQUEST)
	public String handleException(Exception ex) {
		logger.info("Obsługa ValidationException " + ex.getMessage());
		return ex.getMessage();
	}
}