/**
 * Licencja Apache Software Foundation (ASF) oparta na jednej lub kilku
 * licencjach udziałowców. Dodatkowe informacje dotyczące praw autorskich
 * zawarte są w pliku NOTICE dołączonym to tego materiału. ASF udziela Ci
 * licnecji na ten plik na warunkach licencji Apache License, Version 2.0
 * ("Licencja"). Kopia licencji jest dostępna pod adresem: 
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * O ile nie jest to określone przez właściwe przepisy prawa lub uzgodnione
 * na piśmie, oprogramowanie w ramach Licencji jest rozpowszechniane w takiej postaci,
 * w jakiej jest, bez jakichkolwiek gwarancji ani warunków, jawnie wyrażonych,
 * ani domniemanych. Terminy opisujące uprawnienia i ograniczenia są zawarte w 
 * treści Licencji.
 */

package com.cloudera.datascience.common;

import java.io.IOException;
import java.nio.charset.StandardCharsets;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.InputSplit;
import org.apache.hadoop.mapreduce.RecordReader;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.apache.hadoop.mapreduce.lib.input.FileSplit;
import org.apache.hadoop.mapreduce.lib.input.TextInputFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * UWAGA: ta część jest skopiowana z Mahout. Jest w całości samodzielna, więc skopiowanie oznacza,
 * że można uniknąć zależności Mahout.
 *
 * Klasa odczytująca rekordy rozdzielone okreśłonymi znacznikami początkowym i końcowym.
 */
public class XmlInputFormat extends TextInputFormat {

  private static final Logger log = LoggerFactory.getLogger(XmlInputFormat.class);

  public static final String START_TAG_KEY = "xmlinput.start";
  public static final String END_TAG_KEY = "xmlinput.end";

  @Override
  public RecordReader<LongWritable, Text> createRecordReader(InputSplit split, TaskAttemptContext context) {
    try {
      return new XmlRecordReader((FileSplit) split, context.getConfiguration());
    } catch (IOException ioe) {
      log.warn("Błąd podczas tworzenia XmlRecordReader", ioe);
      return null;
    }
  }

  /**
   * Klasa XMLRecordReader odczytująca dokument XML w celu zwrócenia bloków XML w postaci rekordów'
   * określonych przez znaczniki początkowy i końcowy
   * 
   */
  public static class XmlRecordReader extends RecordReader<LongWritable, Text> {

    private final byte[] startTag;
    private final byte[] endTag;
    private final long start;
    private final long end;
    private final FSDataInputStream fsin;
    private final DataOutputBuffer buffer = new DataOutputBuffer();
    private LongWritable currentKey;
    private Text currentValue;

    public XmlRecordReader(FileSplit split, Configuration conf) throws IOException {
      startTag = conf.get(START_TAG_KEY).getBytes(StandardCharsets.UTF_8);
      endTag = conf.get(END_TAG_KEY).getBytes(StandardCharsets.UTF_8);

      // otwarcie pliku i wyszukanie znacznika początkowego
      start = split.getStart();
      end = start + split.getLength();
      Path file = split.getPath();
      FileSystem fs = file.getFileSystem(conf);
      fsin = fs.open(split.getPath());
      fsin.seek(start);
    }

    private boolean next(LongWritable key, Text value) throws IOException {
      if (fsin.getPos() < end && readUntilMatch(startTag, false)) {
        try {
          buffer.write(startTag);
          if (readUntilMatch(endTag, true)) {
            key.set(fsin.getPos());
            value.set(buffer.getData(), 0, buffer.getLength());
            return true;
          }
        } finally {
          buffer.reset();
        }
      }
      return false;
    }

    @Override
    public void close() throws IOException {
      fsin.close();
    }

    @Override
    public float getProgress() throws IOException {
      return (fsin.getPos() - start) / (float) (end - start);
    }

    private boolean readUntilMatch(byte[] match, boolean withinBlock) throws IOException {
      int i = 0;
      while (true) {
        int b = fsin.read();
        // koniec pliku:
        if (b == -1) {
          return false;
        }
        // zapis do bufora:
        if (withinBlock) {
          buffer.write(b);
        }

        // sprawdzene zgodności:
        if (b == match[i]) {
          i++;
          if (i >= match.length) {
            return true;
          }
        } else {
          i = 0;
        }
        // sprawdzenie, czy został minięty punkt końcowy:
        if (!withinBlock && i == 0 && fsin.getPos() >= end) {
          return false;
        }
      }
    }

    @Override
    public LongWritable getCurrentKey() throws IOException, InterruptedException {
      return currentKey;
    }

    @Override
    public Text getCurrentValue() throws IOException, InterruptedException {
      return currentValue;
    }

    @Override
    public void initialize(InputSplit split, TaskAttemptContext context) throws IOException, InterruptedException {
    }

    @Override
    public boolean nextKeyValue() throws IOException, InterruptedException {
      currentKey = new LongWritable();
      currentValue = new Text();
      return next(currentKey, currentValue);
    }
  }
}
