/*
 * chat.js — moduł zapewniający wymianę wiadomości na czacie.
*/

/*jslint         node    : true, continue : true,
  devel  : true, indent  : 2,    maxerr   : 50,
  newcap : true, nomen   : true, plusplus : true,
  regexp : true, sloppy  : true, vars     : false,
  white  : true
*/
/*global */

// ------------ ROZPOCZĘCIE SEKCJI ZMIENNYCH ZAKRESU MODUŁU --------------
'use strict';
var
  emitUserList, signIn, signOut, chatObj,
  socket = require( 'socket.io' ),
  crud   = require( './crud'    ),

  makeMongoId = crud.makeMongoId,
  chatterMap  = {};
// ------------- ZAKOŃCZENIE SEKCJI ZMIENNYCH ZAKRESU MODUŁU ---------------

// ---------------- ROZPOCZĘCIE SEKCJI METOD NARZĘDZIOWYCH -----------------
// emitUserList — rozgłaszanie listy użytkowników do wszystkich podłączonych klientów.
//
emitUserList = function ( io ) {
  crud.read(
    'user',
    { is_online : true },
    {},
    function ( result_list ) {
      io
        .of( '/chat' )
        .emit( 'listchange', result_list );
    }
  );
};

// signIn — aktualizacja właściwości is_online i obiektu chatterMap.
//
signIn = function ( io, user_map, socket ) {
  crud.update(
    'user',
    { '_id'     : user_map._id },
    { is_online : true         },
    function ( result_map ) {
      emitUserList( io );
      user_map.is_online = true;
      socket.emit( 'userupdate', user_map );
    }
  );

  chatterMap[ user_map._id ] = socket;
  socket.user_id = user_map._id;
};

// signOut — aktualizacja właściwości is_online i obiektu chatterMap.
//
signOut = function ( io, user_id ) {
  crud.update(
    'user',
    { '_id'     : user_id },
    { is_online : false   },
    function ( result_list ) { emitUserList( io ); }
  );
  delete chatterMap[ user_id ];
};
// ----------------- ZAKOŃCZENIE SEKCJI METOD NARZĘDZIOWYCH  ------------------

// ---------------- ROZPOCZĘCIE SEKCJI METOD PUBLICZNYCH ------------------
chatObj = {
  connect : function ( server ) {
    var io = socket.listen( server );

    // Rozpoczęcie konfiguracji io.
    io
      .set( 'blacklist' , [] )
      .of( '/chat' )
      .on( 'connection', function ( socket ) {

		// Rozpoczęcie procedury obsługi komunikatu /adduser/.
        // Podsumowanie: zapewnia funkcjonalność logowania.
        // Argumenty: pojedynczy obiekt user_map.
        //   Obiekt user_map powinien mieć następujące właściwości:
        //     name = nazwa użytkownika;
        //     cid  = identyfikator klienta.
        // Akcje:
        //   Jeśli użytkownik z dostarczoną nazwą użytkownika już istnieje w bazie danych MongoDB, użycie istniejącego obiektu użytkownika i zignorowanie pozostałych danych wejściowych.
        //   Jeśli użytkownik z dostarczoną nazwą użytkownika nie istnieje w bazie danych MongoDB, utworzenie i użycie obiektu tego użytkownika.
        //   Wysłanie do nadawcy komunikatu 'userupdate', aby można było zakończyć cykl logowania. Upewnienie się, że identyfikator klienta został przekazany z powrotem, aby klient mógł powiązać się z użytkownikiem, ale bez zapisywania tego w bazie danych MongoDB.
        //   Ustawienie dla użytkownika statusu online i wysłanie zaktualizowanej listy użytkowników online do wszystkich klientów, w tym do klienta, od którego pochodzi komunikat 'adduser'.
        //
        socket.on( 'adduser', function ( user_map ) {
          crud.read(
            'user',
            { name : user_map.name },
            {},
            function ( result_list ) {
              var
                result_map,
                cid = user_map.cid;

              delete user_map.cid;

              // Użycie istniejącego użytkownika z dostarczoną nazwą.
              if ( result_list.length > 0 ) {
                result_map     = result_list[ 0 ];
                result_map.cid = cid;
                signIn( io, result_map, socket );
              }

              // Utworzenie użytkownika z nową nazwą.
              else {
                user_map.is_online = true;
                crud.construct(
                  'user',
                  user_map,
                  function ( result_list ) {
                    result_map     = result_list[ 0 ];
                    result_map.cid = cid;
                    chatterMap[ result_map._id ] = socket;
                    socket.user_id = result_map._id;
                    socket.emit( 'userupdate', result_map );
                    emitUserList( io );
                  }
                );
              }
            }
          );
        });
        // Zakończenie procedury obsługi komunikatu /adduser/. 

        // Rozpoczęcie procedury obsługi komunikatu /updatechat/.
        // Podsumowanie: obsługa wiadomości dla czatu.
        // Argumenty: pojedynczy obiekt chat_map.
        //   Obiekt chat_map powinien mieć następujące właściwości:
        //     dest_id  = identyfikator odbiorcy;
        //     dest_name = nazwa odbiorcy;
        //     sender_id = identyfikator nadawcy;
        //     msg_text  = treść wiadomości.
        // Akcje :
        //   Jeśli odbiorca jest online, wysyłany jest do niego obiekt chat_map.
        //   Jeśli odbiorca nie jest online, do nadawcy wysyłany jest komunikat 'użytkownik jest w trybie offline.'.
        //
		
        socket.on( 'updatechat', function ( chat_map ) {
          if ( chatterMap.hasOwnProperty( chat_map.dest_id ) ) {
            chatterMap[ chat_map.dest_id ]
              .emit( 'updatechat', chat_map );
          }
          else {
            socket.emit( 'updatechat', {
              sender_id : chat_map.sender_id,
              msg_text  : chat_map.dest_name + ' jest w trybie offline.'
            });
          }
        });
        // Zakończenie procedury obsługi komunikatu /updatechat/.

        // Rozpoczęcie metod rozłączania.
        socket.on( 'leavechat', function () {
          console.log(
            '** Użytkownik %s wylogowany **', socket.user_id
          );
          signOut( io, socket.user_id );
        });

        socket.on( 'disconnect', function () {
          console.log(
            '** Użytkownik %s zamknął okno lub kartę przeglądarki **',
            socket.user_id
          );
          signOut( io, socket.user_id );
        });
        // Zakończenie metod rozłączania.

        socket.on( 'updateavatar', function () {} );
      }
    );
    // Zakończenie konfiguracji io.

    return io;
  }
};

module.exports = chatObj;
// ----------------- ZAKOŃCZENIE SEKCJI METOD PUBLICZNYCH -------------------
