#!/usr/local/bin/perl
###########################################################
# Plik: rankingfactors.pl                                 #
# Opis: Skrypt wykonuje analiz kilku czynnikw           #
#       wpywajcych na pozycj strony w rankingu,        #
#       w tym analizuje:                                  #
# 1) Sowa kluczowe w tytuach stron                      #
# 2) Sowa kluczowe w nazwach domen                       #
# 3) Sowa kluczowe w treci strony                       #
# 4) Sowa kluczowe w nagwkach                          #
# 5) Sowa kluczowe w opisach meta                        #
# 6) Blisko sw kluczowych                             #
# 7) Sowa kluczowe w linkach zewntrznych                #
# 8) Rozmiar strony                                       #
# 9) Liczb sw na stron                                #
# 10) Rozmiar witryny                                     #
# i wiele innych czynnikw...                             #
#                                                         #
# Format: perl rankingfactors.pl 10|100 sowo(a) kluczowe #
###########################################################

use LWP::Simple;
use LWP::UserAgent;
use HTML::TokeParser;
use HTML::TreeBuilder;
use File::Path;
use Math::Round qw(:all);

my $keyphrase = "";

my @googleLinks = ( );
my @googleTitles = ( );
my @yahooLinks = ( );
my @yahooTitles = ( );
my @bingLinks = ( );
my @bingTitles = ( );

#w razie potrzeby stwrz fraz/sowo kluczowe
foreach $argnum (1 .. $#ARGV) {
    $keyphrase = $keyphrase . "$ARGV[$argnum] ";
}
my $numres = $ARGV[0];
$keyphrase =~ s/^\s+//;
$keyphrase =~ s/\s+$//;
$keyphrase =~ s/'//g;
$keyphrase =~ s/"//g;

print "\nUruchamianie..";
#czyszczenie plikw tymczasowych
rmtree( './serptemp', {keep_root => 1} );
print "\n.. czyszczenie zakoczone ";
#incicjalizacja zmiennych
initializeKeyVariables($keyphrase, \@googleLinks,
                    \@googleTitles, \@yahooLinks,
                    \@yahooTitles, \@bingLinks,
                    \@bingTitles);
#zapisywanie wszytskich docelowych linkw znalezionych na stronach z wynikami wyszukiwania
print "\n..pobieranie stron z wynikami wyszukiwania";
getSERPResults($#googleLinks, \@googleLinks, "google");
getSERPResults($#yahooLinks,\@yahooLinks, "yahoo");
getSERPResults($#bingLinks,\@bingLinks, "bing");
print "\n..pobrano strony z wynikami wyszukiwania";


#-------------------Analiza tytuw-----------------------
#pobieranie rzeczywistych tytuw
my @googleRealTitles = ( );
my @yahooRealTitles = ( );
my @bingRealTitles = ( );
getRealTitles($#googleLinks, \@googleRealTitles, "google");
getRealTitles($#yahooLinks,\@yahooRealTitles, "yahoo");
getRealTitles($#bingLinks,\@bingRealTitles, "bing");
print "\n..pobrano rzeczywiste tytuy";

#pobranie rzeczywistych tytuw z tytuami na stronach z wynikami wyszukiwania
my @googleTitleComp = ( );
my @yahooTitleComp = ( );
my @bingTitleComp = ( );
my $percentMatchTitlesGoogle = compareArrays($#googleTitles,\@googleRealTitles,
\@googleTitles,

\@googleTitleComp);
my $percentMatchTitlesYahoo = compareArrays($#yahooTitles,\@yahooRealTitles,
\@yahooTitles, \@yahooTitleComp);
my $percentMatchTitlesBing = compareArrays($#bingTitles,\@bingRealTitles,
\@bingTitles, \@bingTitleComp);
print "\n..zakoczono czciowe porwnanie tytuw";

#szukanie dopasowa sw kluczowych
my @googleKeywordTitleMatch = ( );
my @yahooKeywordTitleMatch = ( );
my @bingKeywordTitleMatch = ( );
getKeywordsTitleMatch($keyphrase, \@googleRealTitles,$#googleRealTitles,
\@googleKeywordTitleMatch );

getKeywordsTitleMatch($keyphrase, \@yahooRealTitles, $#yahooRealTitles,
\@yahooKeywordTitleMatch);
getKeywordsTitleMatch($keyphrase, \@bingRealTitles, $#bingRealTitles,
\@bingKeywordTitleMatch);
print "\n..zakoczono porwnywanie sw kluczowych tytuw";

#sprawdzanie czy sowa kluczowe znalezione w tytule znajduj si w treci strony
my @googleKeywordTitlePageCopy = ( );
my @yahooKeywordTitlePageCopy = ( );
my @bingKeywordTitlePageCopy = ( );
compareTitlePageCopy($#googleRealTitles,\@googleRealTitles,
\@googleKeywordTitlePageCopy, "google");
compareTitlePageCopy($#yahooRealTitles,\@yahooRealTitles,
\@yahooKeywordTitlePageCopy, "yahoo");
compareTitlePageCopy($#bingRealTitles,\@bingRealTitles,
\@bingKeywordTitlePageCopy, "bing");
print "\n..zakoczono porwnywanie tytuu i treci strony";


#-------------------Analiza nazwy domeny-----------------------

#dopasowanie dokadne
my @googleDomainKeywordExactMatch = ( );
my @yahooDomainKeywordExactMatch = ( );
my @bingDomainKeywordExactMatch = ( );
my $percentDomainKeywordExactMatchGoogle = keywordDomainExactMatch($keyphrase,
\@googleLinks, $#googleLinks,
\@googleDomainKeywordExactMatch);
my $percentDomainKeywordExactMatchYahoo = keywordDomainExactMatch($keyphrase,
\@yahooLinks, $#yahooLinks,
\@yahooDomainKeywordExactMatch);
my $percentDomainKeywordExactMatchBing = keywordDomainExactMatch($keyphrase,
\@bingLinks, $#bingLinks,
\@bingDomainKeywordExactMatch);
print "\n..zakoczono analiz dokadnych dopasowa sw kluczowych w nazwie domeny";

#dopasowanie czciowe
my @googleDomainKeywordPartialMatch = ( );
my @yahooDomainKeywordPartialMatch = ( );
my @bingDomainKeywordPartialMatch = ( );
$percentDomainKeywordPartialMatchGoogle = keywordDomainPartialMatch($keyphrase,
\@googleLinks, $#googleLinks,
\@googleDomainKeywordPartialMatch);
$percentDomainKeywordPartialMatchYahoo = keywordDomainPartialMatch($keyphrase,
\@yahooLinks, $#yahooLinks,
\@yahooDomainKeywordPartialMatch);
$percentDomainKeywordPartialMatchBing = keywordDomainPartialMatch($keyphrase,
\@bingLinks, $#bingLinks,
\@bingDomainKeywordPartialMatch);
print "\n..zakoczono analiz czciowych dopasowa sw kluczowych w nazwie domeny";


#-------------------Analiza treci strony----------------------------
my @googleKeywordDensity = ( );
my @yahooKeywordDensity = ( );
my @bingKeywordDensity = ( );
my $googleAvgDensity = keywordDensity($#googleLinks, $keyphrase,
\@googleKeywordDensity, "google");
my $yahooAvgDensity = keywordDensity($#yahooLinks, $keyphrase,
\@yahooKeywordDensity, "yahoo");
my $bingAvgDensity = keywordDensity($#bingLinks, $keyphrase,
\@bingKeywordDensity, "bing");

#-------------------Analiza znacznikw opisowych META------------------
my @googleDescriptionMetaExact = ( );
my @yahooDescriptionMetaExact = ( );
my @bingDescriptionMetaExact = ( );

checkExactDescriptionMeta($#googleLinks, \@googleDescriptionMetaExact,
$keyphrase, "google");
checkExactDescriptionMeta($#yahooLinks, \@yahooDescriptionMetaExact,
$keyphrase, "yahoo");
checkExactDescriptionMeta($#bingLinks, \@bingDescriptionMetaExact,
$keyphrase, "bing");

my @googleDescriptionMetaPartial = ( );
my @yahooDescriptionMetaPartial = ( );
my @bingDescriptionMetaPartial = ( );

checkPartialDescriptionMeta($#googleLinks, \@googleDescriptionMetaPartial,
$keyphrase, "google");
checkPartialDescriptionMeta($#yahooLinks, \@yahooDescriptionMetaPartial,
$keyphrase, "yahoo");
checkPartialDescriptionMeta($#bingLinks, \@bingDescriptionMetaPartial,
$keyphrase, "bing");
print "\n..zakoczono analiz opisw META";

#-------------------Analiza znacznikw nagwkw----------------------------
my @googleNumberOfHeaderTags = ( );
my @yahooNumberOfHeaderTags = ( );
my @bingNumberOfHeaderTags = ( );
my @googleHeaderTagsKeywords = ( );
my @yahooHeaderTagsKeywords = ( );
my @bingHeaderTagsKeywords = ( );

checkHeaderTags($#googleLinks, \@googleNumberOfHeaderTags,
\@googleHeaderTagsKeywords, "google", $keyphrase);
checkHeaderTags($#yahooLinks, \@yahooNumberOfHeaderTags,
\@yahooHeaderTagsKeywords, "yahoo", $keyphrase);
checkHeaderTags($#bingLinks, \@bingNumberOfHeaderTags,
\@bingHeaderTagsKeywords, "bing", $keyphrase);
print "\n..zakoczono analiz znacznikw nagwkw";

#-------------------Analiza zagszczenia sw kluczowych---------------------
my @googleKeywordPositions = ( );
my @yahooKeywordPositions = ( );
my @bingKeywordPositions = ( );
my @googleKeywordPositionsList = ( );
my @yahooKeywordPositionsList = ( );
my @bingKeywordPositionsList = ( );
analyzeKeywordPositions($#googleLinks, \@googleKeywordPositions,
\@googleKeywordPositionsList, "google",
$keyphrase);
analyzeKeywordPositions($#yahooLinks, \@yahooKeywordPositions,
\@yahooKeywordPositionsList, "yahoo", $keyphrase);
analyzeKeywordPositions($#bingLinks, \@bingKeywordPositions,
\@bingKeywordPositionsList, "bing", $keyphrase);
print "\n..zakoczono analiz zagszczenia sw kluczowych";

#-------------------Analiza linkw zewntrznych--------------------------
my @googleOutboundLinkKeywords = ( );
my @yahooKOutboundLinkKeywords = ( );
my @bingOutboundLinkKeywords = ( );
outboundLinkKeywordAnalysis($#googleLinks, \@googleLinks,
\@googleOutboundLinkKeywords, "google", $keyphrase);
outboundLinkKeywordAnalysis($#yahooLinks, \@yahooLinks,
\@yahooKOutboundLinkKeywords, "yahoo", $keyphrase);
outboundLinkKeywordAnalysis($#bingLinks, \@bingLinks,
\@bingOutboundLinkKeywords, "bing", $keyphrase);
print "\n..zakoczono analiz linkw zewntrznych";

#-------------------Analiza PR linkw zewntrznych--------------------------
my @googleOutboundLinksPR = ( );
my @yahooKOutboundLinksPR = ( );
my @bingOutboundLinksPR = ( );
outboundLinkPRAnalysis($#googleLinks, \@googleLinks,
\@googleOutboundLinksPR, "google", $keyphrase);
outboundLinkPRAnalysis($#yahooLinks, \@yahooLinks,
\@yahooKOutboundLinksPR, "yahoo", $keyphrase);
outboundLinkPRAnalysis($#bingLinks, \@bingLinks,
\@bingOutboundLinksPR, "bing", $keyphrase);
print "\n..zakoczono analiz PR linkw zewntrznych";

#-------------------Analiza redniego rozmiaru strony--------------------------
my @googlePageSize = ( );
my @yahooPageSize = ( );
my @bingPageSize = ( );
my $googleAvgPageSize = averagePageSize($#googleLinks, \@googlePageSize, "google");
my $yahooAvgPageSize = averagePageSize($#yahooLinks, \@yahooPageSize, "yahoo");
my $bingAvgPageSize = averagePageSize($#bingLinks, \@bingPageSize, "bing");
print "\n..zakoczono analiz redniego rozmiaru strony";

#-------------------Analiza optymalnej liczby sw kluczowych--------------------
my @googleWords = ( );
my @yahooWords = ( );
my @bingWords = ( );
my $googleWordsPerPage = optimumWordsPerPage($#googleLinks, \@googleWords, "google");
my $yahooWordsPerPage = optimumWordsPerPage($#yahooLinks, \@yahooWords, "yahoo");
my $bingWordsPerPage = optimumWordsPerPage($#bingLinks, \@bingWords, "bing");
print "\n..zakoczono analiz optymalnej liczby sw kluczowych na stron";

#-------------------Analiza rozmiaru strony -------------------------------
my @googleResultsWebsiteSizes = ( );
my @yahooResultsWebsiteSizes = ( );
my @bingResultsWebsiteSizes = ( );
my $googleAverageWebSize = analyzeWebsiteSize($#googleLinks, \@googleLinks,
\@googleResultsWebsiteSizes);
my $yahooAverageWebSize = analyzeWebsiteSize($#yahooLinks, \@yahooLinks,
\@yahooResultsWebsiteSizes);
my $bingAverageWebSize = analyzeWebsiteSize($#bingLinks, \@bingLinks,
\@bingResultsWebsiteSizes);
print "\n..zakoczono analiz rozmiaru strony internetowej";

#-------------------Analiza wieku strony-----------------------------------
my @googlePageAge = ( );
my @yahooPageAge = ( );
my @bingPageAge = ( );
pageAgeAnalysis($#googleLinks, \@googleLinks, \@googlePageAge);
pageAgeAnalysis($#yahooLinks, \@yahooLinks, \@yahooPageAge);
pageAgeAnalysis($#bingLinks, \@bingLinks, \@bingPageAge);


#-------------------Tworzenie raportu HTML---------------------------------
#tworzenie pliku index
createIndexHTML($keyphrase);

my $numberOfLinesGoogle = $#googleLinks;
my $numberOfLinesYahoo = $#yahooLinks;
my $numberOfLinesBing = $#bingLinks;

createGoogleHTMLReport();
createYahooHTMLReport();
createBingHTMLReport();

#---------------------------PODPROGRAMY---------------------------
# Podprogram:
#   createGoogleHTMLReport
# Opis:
#   Ten podprogram tworzy plik google.html 
#   z podsumowaniem wynikw dla wyszukiwarki Google
# Zmienne wejciowe:
#   Brak
# Zmienne wyjciowe:
#   Tworzy plik google.html
# Zwraca:
#   Nic
sub createGoogleHTMLReport {
    #tworzenie tabeli podsumowania
    my $googleFile = "<html><head><title>Raport podsumowujcy dla wyszukiwarki Google</title>";
    $googleFile .= "<style>";
    $googleFile .=
"body, td, tr{font-family: \"Trebuchet ms\", verdana, sans-serif; font-size:9px;}";
    $googleFile .=
"b{font-family: \"Trebuchet ms\", verdana, sans-serif;font-size:10px;}";
    $googleFile .= "</style>";
    $googleFile .= "</head>";
    $googleFile .= "<body><h1>Raport podsumowujcy</h1>";
    $googleFile .= "<br>";
    $googleFile .=
"<table border=\"1\" width=\"500\" cellspacing=\"2\" cellpadding=\"2\">";
    $googleFile .= "<tr><td colspan=2><b>Wartoci rednie</b></td>";
    $googleFile .= "</tr>";
    $googleFile .= "<tr>";
    $googleFile .= "<td><b>% dopasowania tytuu</b></td>";
    my $tmp = sprintf "%.1f", $percentMatchTitlesGoogle;
    $googleFile .= "<td>$tmp</td>";
    $googleFile .= "</tr>";
    $googleFile .= "<tr>";
    $googleFile .= "<td><b>% dokadnego dopasowania sw kluczowych w nazwie domeny</b></td>";
    $tmp = sprintf "%.1f", $percentDomainKeywordExactMatchGoogle;
    $googleFile .= "<td>$tmp</td>";
    $googleFile .= "</tr>";
    $googleFile .= "<tr>";
    $googleFile .= "<td><b>% czciowego dopasowania sw kluczowych w nazwie domeny</b></td>";
    $tmp = sprintf "%.1f", $percentDomainKeywordPartialMatchGoogle;
    $googleFile .= "<td>$tmp</td>";
    $googleFile .= "</tr>";
    $googleFile .= "<tr>";
    $googleFile .= "<td><b>% zagszczenie sw kluczowych</b></td>";
    $tmp = sprintf "%.1f", $googleAvgDensity;
    $googleFile .= "<td>$tmp</td>";
    $googleFile .= "</tr>";
    $googleFile .= "<tr>";
    $googleFile .= "<td><b>Rozmiar strony [bajty]</b></td>";
    $tmp = sprintf "%.0f", $googleAvgPageSize;
    $googleFile .= "<td>$tmp</td>";
    $googleFile .= "</tr>";
    $googleFile .= "<tr>";
    $googleFile .= "<td><b>Liczba sw na stron</b></td>";
    $tmp = sprintf "%.0f", $googleWordsPerPage;
    $googleFile .= "<td>$tmp</td>";
    $googleFile .= "</tr>";
    $googleFile .= "<tr>";
    $googleFile .= "<td><b>Rozmiar witryny [bazowych adresw URL]</b></td>";
    $tmp = round($googleAverageWebSize);
    $googleFile .= "<td>$tmp</td>";
    $googleFile .= "</tr>";
    $googleFile .= "</table><br><br>";
    $googleFile .= "<b>Tabela szczegowa</b> <br>";
    $googleFile .= "<table border=1 cellpadding=2 cellspacing=2>";
    $googleFile .= "<tr>";
    $googleFile .= "<td nowrap>#</td>";
    $googleFile .= "<td width='100'><b>URL</b></td>";
    $googleFile .= "<td nowrap width='150'><b>Tytu Google</b></td>";
    $googleFile .= "<td nowrap width='150'><b>Tytu strony</b></td>";
    $googleFile .= "<td nowrap><b>Sowa kluczowe<br> znalezione w tytule? [Y|N]</b></td>";
    $googleFile .= "<td nowrap><b>Sowa kluczowe <br>w treci strony[%]</b></td>";
    $googleFile .= "<td nowrap><b>Dokadne dopasowania <br>w nazwie domeny</b></td>";
    $googleFile .= "<td nowrap><b>Czciowe dopasowania <br>w nazwie domeny</b></td>";
    $googleFile .= "<td nowrap><b>Zagszczenie sw<br> kluczowych</b></td>";
    $googleFile .= "<td nowrap><b>Dokadne dopasowania <br>w opisach META</b></td>";
    $googleFile .= "<td nowrap><b>Czciowe dopasowania <br>w opisach META</b></td>";
    $googleFile .= "<td nowrap><b>Znaczniki nagwkw</b></td>";
    $googleFile .= "<td nowrap><b>Sowa kluczowe<br>w znacznikach nagwkw</b></td>";
    $googleFile .= "<td nowrap width='350'><b>Pozycje sw kluczowych <br> na stronie</b></td>";
    $googleFile .= "<td nowrap><b>Mapa zagszczenia <br>sw kluczowych</b></td>";
    $googleFile .= "<td nowrap><b>Linki zewntrzne <br> ze sowami kluczowymi</b></td>";
    $googleFile .= "<td nowrap width='150'><b>Wskanik PR<br> linkw zewntrznych</b></td>";
    $googleFile .= "<td nowrap><b>Rozmiar strony<br>[bajty]</b></td>";
    $googleFile .= "<td nowrap><b>Liczba sw<br> na stronie</b></td>";
    $googleFile .= "<td nowrap><b>Rozmiar witryny <br> internetowej</b></td>";
    $googleFile .= "<td nowrap><b>Wiek strony</b></td>";
    $googleFile .= "</tr>";

    for (my $i=0; $i < $numberOfLinesGoogle; $i++) {
        $googleFile .= "<tr>";
        $googleFile .= "<td align=left>$i&nbsp;</td>";
        $googleFile .= "<td align=left>$googleLinks[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googleTitles[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googleRealTitles[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googleKeywordTitleMatch[$i]&nbsp;</td>";
        $tmp = sprintf "%.1f", $googleKeywordTitlePageCopy[$i];

        $googleFile .= "<td align=left>$tmp&nbsp;</td>";
        $googleFile .= "<td align=left>$googleDomainKeywordExactMatch[$i]&nbsp;</td>";
        $googleFile .=
"<td align=left>$googleDomainKeywordPartialMatch[$i]&nbsp;</td>";
        $tmp = sprintf "%.3f", $googleKeywordDensity[$i];
        $googleFile .= "<td align=left>$tmp&nbsp;</td>";
        $googleFile .= "<td align=left>$googleDescriptionMetaExact[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googleDescriptionMetaPartial[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googleNumberOfHeaderTags[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googleHeaderTagsKeywords[$i]&nbsp;</td>";
        $tmp = $googleKeywordPositionsList[$i];
        $tmp =~ s/\|/\, /g;
        $googleFile .= "<td align=left>$tmp&nbsp;</td>";
        $googleFile .=
"<td align=left><a href='./maps/google".$i.".html'>Map</a></td>";
        printIndividualKeywordProminenceMap($i, \@googleKeywordPositions, "google");
        $googleFile .= "<td align=left>$googleOutboundLinkKeywords[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googleOutboundLinksPR[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googlePageSize[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googleWords[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googleResultsWebsiteSizes[$i]&nbsp;</td>";
        $googleFile .= "<td align=left>$googlePageAge[$i]&nbsp;</td>";
        $googleFile .= "</tr>";
    }
    my $filename = "./report/google.html";
    open FILE, ">", "$filename" or die $!;
    print FILE $googleFile;
    close FILE;
}

# Podprogram:
#   createYahooHTMLReport
# Opis:
#   Ten podprogram tworzy plik yahoo.html
#   z podsumowaniem wynikw dla wyszukiwarki Yahoo
# Zmienne wejciowe:
#   Brak
# Zmienne wyjciowe:
#   Tworzy plik yahoo.html
# Zwraca:
#   Nic
sub createYahooHTMLReport {
    #Tworzenie tabeli podsumowania
    my $yahooFile = "<html><head><title>Raport podsumowujcy dla wyszukiwarki Yahoo</title>";
    $yahooFile .= "<style>";
    $yahooFile .=
"body, td, tr{font-family: \"Trebuchet ms\", verdana, sans-serif; font-size:9px;}";
    $yahooFile .=
"b{font-family: \"Trebuchet ms\", verdana, sans-serif;font-size:10px;}";
    $yahooFile .= "</style>";
    $yahooFile .= "</head>";
    $yahooFile .= "<body><h1>Raport podsumowujcy</h1>";
    $yahooFile .= "<br>";
    $yahooFile .=
"<table border=\"1\" width=\"500\" cellspacing=\"2\" cellpadding=\"2\">";
    $yahooFile .= "<tr><td colspan=2><b>Wartoci rednie</b></td>";
    $yahooFile .= "</tr>";
    $yahooFile .= "<tr>";
    $yahooFile .= "<td><b>% dopasowania tytuu</b></td>";
    my $tmp = sprintf "%.1f", $percentMatchTitlesYahoo;
    $yahooFile .= "<td>$tmp</td>";
    $yahooFile .= "</tr>";
    $yahooFile .= "<tr>";
    $yahooFile .= "<td><b>% dokadnego dopasowania sw kluczowych w nazwie domeny </b></td>";
    $tmp = sprintf "%.1f", $percentDomainKeywordExactMatchYahoo;
    $yahooFile .= "<td>$tmp</td>";
    $yahooFile .= "</tr>";
    $yahooFile .= "<tr>";
    $yahooFile .= "<td><b>% czciowego dopasowania sw kluczowych w nazwie domeny </b></td>";
    $tmp = sprintf "%.1f", $percentDomainKeywordPartialMatchYahoo;
    $yahooFile .= "<td>$tmp</td>";
    $yahooFile .= "</tr>";
    $yahooFile .= "<tr>";
    $yahooFile .= "<td><b>% zagszczenie sw kluczowych </b></td>";
    $tmp = sprintf "%.1f", $yahooAvgDensity;
    $yahooFile .= "<td>$tmp</td>";
    $yahooFile .= "</tr>";
    $yahooFile .= "<tr>";
    $yahooFile .= "<td><b>Rozmiar strony [bajty]</b></td>";
    $tmp = sprintf "%.0f", $yahooAvgPageSize;
    $yahooFile .= "<td>$tmp</td>";
    $yahooFile .= "</tr>";
    $yahooFile .= "<tr>";
    $yahooFile .= "<td><b> Liczba sw na stron</b></td>";
    $tmp = sprintf "%.0f", $yahooWordsPerPage;
    $yahooFile .= "<td>$tmp</td>";
    $yahooFile .= "</tr>";
    $yahooFile .= "<tr>";
    $yahooFile .= "<td><b> Rozmiar witryny [bazowych adresw URL]</b></td>";
    $tmp = round($yahooAverageWebSize);
    $yahooFile .= "<td>$tmp</td>";
    $yahooFile .= "</tr>";
    $yahooFile .= "</table><br><br>";
    $yahooFile .= "<b>Tabela szczegowa</b> <br>";
    $yahooFile .= "<table border=1 cellpadding=2 cellspacing=2>";
    $yahooFile .= "<tr>";
    $yahooFile .= "<td nowrap>#</td>";
    $yahooFile .= "<td width='100'><b>URL</b></td>";
    $yahooFile .= "<td nowrap width='150'><b>Tytu Yahoo</b></td>";
    $yahooFile .= "<td nowrap width='150'><b>Tytu strony</b></td>";
    $yahooFile .= "<td nowrap><b>Sowa kluczowe<br> znalezione w tytule? [Y|N]</b></td>";
    $yahooFile .= "<td nowrap><b>Sowa kluczowe <br>w treci strony[%]</b></td>";
    $yahooFile .= "<td nowrap><b>Dokadne dopasowania <br>w nazwie domeny</b></td>";
    $yahooFile .= "<td nowrap><b>Czciowe dopasowania <br>w nazwie domeny</b></td>";
    $yahooFile .= "<td nowrap><b>Zagszczenie sw<br> kluczowych</b></td>";
    $yahooFile .= "<td nowrap><b>Dokadne dopasowania <br>w opisach META</b></td>";
    $yahooFile .= "<td nowrap><b>Czciowe dopasowania <br>w opisach META</b></td>";
    $yahooFile .= "<td nowrap><b>Znaczniki nagwkw</b></td>";
    $yahooFile .= "<td nowrap><b>Sowa kluczowe<br>w znacznikach nagwkw</b></td>";
    $yahooFile .= "<td nowrap width='350'><b>Pozycje sw kluczowych <br> na stronie</b></td>";
    $yahooFile .= "<td nowrap><b>Mapa zagszczenia <br>sw kluczowych</b></td>";
    $yahooFile .= "<td nowrap><b>Linki zewntrzne <br> ze sowami kluczowymi</b></td>";
    $yahooFile .= "<td nowrap width='150'><b>Wskanik PR<br> linkw zewntrznych</b></td>";
    $yahooFile .= "<td nowrap><b>Rozmiar strony<br>[bajty]</b></td>";
    $yahooFile .= "<td nowrap><b>Liczba sw<br> na stronie</b></td>";
    $yahooFile .= "<td nowrap><b>Rozmiar witryny <br> internetowej</b></td>";
    $yahooFile .= "<td nowrap><b>Wiek strony</b></td>";
    $yahooFile .= "</tr>";

    for (my $i=0; $i < $numberOfLinesYahoo; $i++) {
        $yahooFile .= "<tr>";
        $yahooFile .= "<td align=left>$i&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooLinks[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooTitles[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooRealTitles[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooKeywordTitleMatch[$i]&nbsp;</td>";
        $tmp = sprintf "%.1f", $yahooKeywordTitlePageCopy[$i];

        $yahooFile .= "<td align=left>$tmp&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooDomainKeywordExactMatch[$i]&nbsp;</td>";
        $yahooFile .=
"<td align=left>$yahooDomainKeywordPartialMatch[$i]&nbsp;</td>";
        $tmp = sprintf "%.3f", $yahooKeywordDensity[$i];
        $yahooFile .= "<td align=left>$tmp&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooDescriptionMetaExact[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooDescriptionMetaPartial[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooNumberOfHeaderTags[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooHeaderTagsKeywords[$i]&nbsp;</td>";
        $tmp = $yahooKeywordPositionsList[$i];
        $tmp =~ s/\|/\, /g;
        $yahooFile .= "<td align=left>$tmp&nbsp;</td>";
        $yahooFile .=
"<td align=left><a href='./maps/yahoo".$i.".html'>Map</a></td>";
        printIndividualKeywordProminenceMap($i, \@yahooKeywordPositions, "yahoo");
        $yahooFile .= "<td align=left>$yahooOutboundLinkKeywords[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooOutboundLinksPR[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooPageSize[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooWords[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooResultsWebsiteSizes[$i]&nbsp;</td>";
        $yahooFile .= "<td align=left>$yahooPageAge[$i]&nbsp;</td>";
        $yahooFile .= "</tr>";
    }
    my $filename = "./report/yahoo.html";
    open FILE, ">", "$filename" or die $!;
    print FILE $yahooFile;
    close FILE;
}

# Podprogram:
#   createBingHTMLReport
# Opis:
#   Ten podprogram tworzy plik bing.html
#   z podsumowaniem wynikw dla wyszukiwarki Bing
# Zmienne wejciowe:
#   Brak
# Zmienne wyjciowe:
#   Tworzy plik bing.html
# Zwraca:
#   Nic
sub createBingHTMLReport {
    #tworzenie tabeli podsumowania
          my $bingFile = "<html><head><title>Raport podsumowujcy dla wyszukiwarki Bing</title>";
    $bingFile .= "<style>";
    $bingFile .=
"body, td, tr{font-family: \"Trebuchet ms\", verdana, sans-serif; font-size:9px;}";
    $bingFile .=
"b{font-family: \"Trebuchet ms\", verdana, sans-serif;font-size:10px;}";
    $bingFile .= "</style>";
    $bingFile .= "</head>";
    $bingFile .= "<body><h1>Raport podsumowujcy</h1>";
    $bingFile .= "<br>";
    $bingFile .=
"<table border=\"1\" width=\"500\" cellspacing=\"2\" cellpadding=\"2\">";
    $bingFile .= "<tr><td colspan=2><b>Wartoci rednie</b></td>";
    $bingFile .= "</tr>";
    $bingFile .= "<tr>";
    $bingFile .= "<td><b>% dopasowania tytuu</b></td>";
    my $tmp = sprintf "%.1f", $percentMatchTitlesBing;
    $bingFile .= "<td>$tmp</td>";
    $bingFile .= "</tr>";
    $bingFile .= "<tr>";
    $bingFile .= "<td><b>% dokadnego dopasowania sw kluczowych w nazwie domeny</b></td>";
    $tmp = sprintf "%.1f", $percentDomainKeywordExactMatchBing;
    $bingFile .= "<td>$tmp</td>";
    $bingFile .= "</tr>";
    $bingFile .= "<tr>";
    $bingFile .= "<td><b>% czciowego dopasowania sw kluczowych w nazwie domeny</b></td>";
    $tmp = sprintf "%.1f", $percentDomainKeywordPartialMatchBing;
    $bingFile .= "<td>$tmp</td>";
    $bingFile .= "</tr>";
    $bingFile .= "<tr>";
    $bingFile .= "<td><b>% zagszczenie sw kluczowych</b></td>";
    $tmp = sprintf "%.1f", $bingAvgDensity;
    $bingFile .= "<td>$tmp</td>";
    $bingFile .= "</tr>";
    $bingFile .= "<tr>";
    $bingFile .= "<td><b>Rozmiar strony [bajty]</b></td>";
    $tmp = sprintf "%.0f", $bingAvgPageSize;
    $bingFile .= "<td>$tmp</td>";
    $bingFile .= "</tr>";
    $bingFile .= "<tr>";
    $bingFile .= "<td><b>Liczba sw na stron</b></td>";
    $tmp = sprintf "%.0f", $bingWordsPerPage;
    $bingFile .= "<td>$tmp</td>";
    $bingFile .= "</tr>";
    $bingFile .= "<tr>";
    $bingFile .= "<td><b>Rozmiar witryny [bazowych adresw URL]</b></td>";
    $tmp = round($bingAverageWebSize);
    $bingFile .= "<td>$tmp</td>";
    $bingFile .= "</tr>";
    $bingFile .= "</table><br><br>";
    $bingFile .= "<b>Tabela szczegowa</b> <br>";
    $bingFile .= "<table border=1 cellpadding=2 cellspacing=2>";
    $bingFile .= "<tr>";
    $bingFile .= "<td nowrap>#</td>";
    $bingFile .= "<td width='100'><b>URL</b></td>";
    $bingFile .= "<td nowrap width='150'><b>Tytu Bing</b></td>";
    $bingFile .= "<td nowrap width='150'><b>Tytu strony</b></td>";
    $bingFile .= "<td nowrap><b>Sowa kluczowe<br> znalezione w tytule? [Y|N]</b></td>";
    $bingFile .= "<td nowrap><b>Sowa kluczowe <br>w treci strony[%]</b></td>";
    $bingFile .= "<td nowrap><b>Dokadne dopasowania <br>w nazwie domeny</b></td>";
    $bingFile .= "<td nowrap><b>Czciowe dopasowania <br>w nazwie domeny</b></td>";
    $bingFile .= "<td nowrap><b>Zagszczenie sw<br> kluczowych</b></td>";
    $bingFile .= "<td nowrap><b>Dokadne dopasowania <br>w opisach META</b></td>";
    $bingFile .= "<td nowrap><b>Czciowe dopasowania <br>w opisach META</b></td>";
    $bingFile .= "<td nowrap><b>Znaczniki nagwkw</b></td>";
    $bingFile .= "<td nowrap><b>Sowa kluczowe<br>w znacznikach nagwkw</b></td>";
    $bingFile .= "<td nowrap width='350'><b>Pozycje sw kluczowych <br> na stronie</b></td>";
    $bingFile .= "<td nowrap><b>Mapa zagszczenia <br>sw kluczowych</b></td>";
    $bingFile .= "<td nowrap><b>Linki zewntrzne <br> ze sowami kluczowymi</b></td>";
    $bingFile .= "<td nowrap width='150'><b>Wskanik PR<br> linkw zewntrznych</b></td>";
    $bingFile .= "<td nowrap><b>Rozmiar strony<br>[bajty]</b></td>";
    $bingFile .= "<td nowrap><b>Liczba sw<br> na stronie</b></td>";
    $bingFile .= "<td nowrap><b>Rozmiar witryny <br> internetowej</b></td>";
    $bingFile .= "<td nowrap><b>Wiek strony</b></td>";
    $bingFile .= "</tr>";
    
    for (my $i=0; $i < $numberOfLinesBing; $i++) {
        $bingFile .= "<tr>";
        $bingFile .= "<td align=left>$i&nbsp;</td>";
        $bingFile .= "<td align=left>$bingLinks[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingTitles[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingRealTitles[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingKeywordTitleMatch[$i]&nbsp;</td>";
        $tmp = sprintf "%.1f", $bingKeywordTitlePageCopy[$i];

        $bingFile .= "<td align=left>$tmp&nbsp;</td>";
        $bingFile .= "<td align=left>$bingDomainKeywordExactMatch[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingDomainKeywordPartialMatch[$i]&nbsp;</td>";
        $tmp = sprintf "%.3f", $bingKeywordDensity[$i];
        $bingFile .= "<td align=left>$tmp&nbsp;</td>";
        $bingFile .= "<td align=left>$bingDescriptionMetaExact[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingDescriptionMetaPartial[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingNumberOfHeaderTags[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingHeaderTagsKeywords[$i]&nbsp;</td>";
        $tmp = $bingKeywordPositionsList[$i];
        $tmp =~ s/\|/\, /g;
        $bingFile .= "<td align=left>$tmp&nbsp;</td>";
        $bingFile .= "<td align=left><a href='./maps/bing".$i.".html'>Map</a></td>";
        printIndividualKeywordProminenceMap($i, \@bingKeywordPositions, "bing");
        $bingFile .= "<td align=left>$bingOutboundLinkKeywords[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingOutboundLinksPR[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingPageSize[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingWords[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingResultsWebsiteSizes[$i]&nbsp;</td>";
        $bingFile .= "<td align=left>$bingPageAge[$i]&nbsp;</td>";
        $bingFile .= "</tr>";
    }
    my $filename = "./report/bing.html";
    open FILE, ">", "$filename" or die $!;
    print FILE $bingFile;
    close FILE;
}

# Podprogram:
#   createIndexHTML
# Opis:
#   Ten podprogram tworzy fragment kodu HTML dla pliku index 
#   szukajc ostatnio zmodyfikowanego acucha znakw
# Zmienne wejciowe:
#   $keyword => sowo kluczowe
# Zmienne wyjciowe:
#   Tworzy plik index.html
# Zwraca:
#   Nic
sub createIndexHTML {
    my $keyword = shift;

    my $indexFile = "<html><head><title>Raport podsumowujcy</title></head>";
    $indexFile .= "<body><center><strong>Raport podsumowujcy";
    $indexFile .= " (for \"$keyword\") <br><br>";
    $indexFile .=
"<a href=\"#\" onclick=\"document.all.myiframe.src='google.html\'\">";
    $indexFile .= "Google</a> |";
    $indexFile .=
"<a href=\"#\" onclick=\"document.all.myiframe.src='yahoo.html'\">";
    $indexFile .= "Yahoo!</a> |";
    $indexFile .=
"<a href=\"#\" onclick=\"document.all.myiframe.src='bing.html'\">";
    $indexFile .= "Bing</a><br><br>";
    $indexFile .= "Kliknij na link, aby wywietli raport...<br><br>";
    $indexFile .=
"<iframe name=\"myiframe\" width=5000 height=6000 border=\"0\" frameborder=\"0\">";
    $indexFile .= "</iframe></center></body></html>";
    
my $filename = "./report/index.html";
    open FILE, ">", "$filename" or die $!;
    print FILE $indexFile;
    close FILE;
}

# Podprogram:
#   pageAgeAnalysis
# Opis:
#   Ten podprogram przeglda wszystkie adresy URL znalezione
#   na stronie z wynikami wyszukiwania szukajc ostatnio modyfikowanego acucha
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $destArr => macierz (odwoanie) do macierzy linkw
#   $srcArr => macierz (odwoanie) do macierzy linkw
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic
sub pageAgeAnalysis {
    my ($numberOfElements, $srcArr, $destArr) = @_;

    for(my $i=0; $i<$numberOfElements; $i++) {
        #print "\nprzetwarzanie: $srcArr->[$i]";
        my $ua = new LWP::UserAgent;
        $ua->agent("Mozilla/3.0 (compatible)");
        my $request = new HTTP::Request("GET", "$srcArr->[$i]");
        my $response = $ua->request($request);
        my $code=$response->code;
        $destArr->[$i]= scalar(localtime($response->last_modified)),
        #print "\n$destArr->[$i]";
    }
}

# Podprogram:
#   analyzeWebsiteSize
# Opis:
#   Ten podprogram przeglda strony z wynikami wyszukiwania Google, 
#   aby sprawdzi rozmiar rnych witryn
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $destArr => macierz (odwoanie) do macierzy linkw
#   $srcArr => macierz (odwoanie) do macierzy linkw
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Zwraca redni rozmiar witryny
sub analyzeWebsiteSize {
    my ($numberOfElements, $srcArr, $destArr) = @_;
    # tworzy linki za pomoc polecenia "site:" 
    my $ua = new LWP::UserAgent;
    my $res;
    $ua->timeout(25);
    $ua->agent("Mozilla/3.0 (compatible)");
    my $total = 0;
    
    for($i=0; $i<$numberOfElements; $i++){

        my $filename = "./serptemp/temp.txt";
        my $url = $srcArr->[$i];
        #Najpierw pobieramy bazowe adresy URL
        
        if($url =~ /^http/) {
            my @tmparr1 = split (/\/\//,$url);
            my @tmparr2 = split (/\//,$tmparr1[1]);
            my $baseurl = "";
            if($#tmparr2>0) {
                $baseurl = $tmparr2[0];
            }else {
                $baseurl = $tmparr1[1];
            }
            $baseurl =~ s/\/$//;
            $url = $baseurl;
        }

        my $tmpurl =
'http://www.google.com/search?hl=en&q=site%3A' . $url . '&btnG=Wyszukaj';
        my $randNum = int(rand(5));
        #print "\nOczekiwanie przez $randNum sekund.\n";
        sleep($randNum);
        $res = $ua->get("$tmpurl",':content_file' => "$filename");
        #Pobieranie treci strony z wynikami wyszukiwania Google
        my $pageCopy = "";
        if (-e "$filename"){
            my $p = HTML::TokeParser->new($filename);
            #Pobieranie treci strony dla tego pliku
            while (my $token = $p->get_tag("body")) {
                $pageCopy = $p->get_text("/body");
            }
        }else {
            print "\nplik nie istnieje";
        }
        #Dzielenie pliku za pomoc tekstu "z okoo <b>"
        my $separator1 = 'z okoo ';

        my @tempArr1 = split(/$separator1/, $pageCopy);
        my $separator2 = 'b';
        my @tempArr2 = split(/$separator2/, $tempArr1[1]);
        my $separator3 = ' for';
        my @tempArr3 = split(/$separator3/, $tempArr2[0]);

        my $size = $tempArr3[0];
        
        #Usunicie przecinku z liczby
        $size =~ s/,//g;
        
        #Zapisywanie dla tego adresu URL
        $destArr->[$i] = $size;
        $total = $total + $size;
    }
    #Obliczanie i zwracanie wartoci redniej
    if ($total>0) {
        return ($total/$numberOfElements);
    } else {
        return 0;
    }
}

# Podprogram:
#   optimumWordsPerPage
# Opis:
#   Ten podprogram wykonuje ptl po wszystkich plikach, aby 
#   w macierzy wynikowej zapisa rozmiary stron.
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $destArr => macierz (odwoanie) do macierzy linkw
#   $prefix => Prefiks pliku wyszukiwarki
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Zwraca redni liczb sw na stron
sub optimumWordsPerPage {
    my ($numberOfElements, $destArr, $prefix) = @_;
    my $total = 0;
    for(my $i=0; $i< $numberOfElements; $i++) {
        my $filename = './serptemp/' . $prefix . "$i.txt";
        my $tree = HTML::TreeBuilder->new;
        $tree->parse_file("$filename");
        my $non_html = $tree->as_text();
        $non_html =~ s/^s+/ /g;
        my @tempsizearr = split(/ /,$non_html);
        $destArr->[$i]= $#tempsizearr;
        $total = $total + $#tempsizearr;
    }
    return ($total/$numberOfElements);
}

# Podprogram:
#   averagePageSize
# Opis:
#   Ten podprogram wykonuje ptl po wszystkich plikach, aby
#   w macierzy wynikowej zapisa rozmiary stron.
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $destArr => macierz (odwoanie) do macierzy linkw
#   $prefix => Prefiks pliku wyszukiwarki
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Zwraca redni rozmiar strony
sub averagePageSize {
    my ($numberOfElements, $destArr, $prefix) = @_;
    my $total = 0;
    for(my $i=0; $i< $numberOfElements; $i++) {
        my $filename = './serptemp/' . $prefix . "$i.txt";
        my $filesize = -s "$filename";
        $destArr->[$i] = $filesize;
        $total = $total + $destArr->[$i];
    }
    return ($total/$numberOfElements);
}

# Podprogram:
#   outboundLinkPRAnalysis
# Opis:
#   Ten podprogram przeprowadza analiz skadniow wartoci PR 
#   z gwnych domen wszystkich zewntrznych linkw
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $srcLinksArr => macierz (odwoanie) do macierzy linkw
#   $prefix => Prefiks pliku wyszukiwarki
# Zmienne wyjciowe:
#   Drukuje map sw kluczowych
# Zwraca:
#   Nic
sub outboundLinkPRAnalysis {
    my ($numberOfElements, $srcLinksArr, $destArr, $prefix) = @_;
    my $PRURL = 'http://www.seowarrior.net/scripts/pr.php?pr=';
    my $range = 2;
    #Ptla przez wszystkie pliki
    for(my $i=0; $i< $numberOfElements; $i++) {
        my $filename = './serptemp/' . $prefix . "$i.txt";
        my %linkHash = ();
        my $PRs = "";
        #Sprawdzanie, czy plik istnieje
        if (-e "$filename") {
            my $p = HTML::TokeParser->new($filename);
            while (my $token = $p->get_tag("a")) {
                #Pobieranie linka i tekstu kotwiczki
                my $url = $token->[1]{href} || "-";
                my $text = $p->get_trimmed_text("/a");
                #Sprawdzanie, czy link jest zewntrzny, czy wewntrzny
                if($url =~ /^http/) {
                    my @tmparr1 = split (/\/\//,$url);
                    my @tmparr2 = split (/\./,$tmparr1[1]);
                    my $tmpbaseURLChild = $tmparr2[0] . $tmparr2[1];

                    my @tmparr3 = split (/\/\//,$srcLinksArr->[$i]);
                    my @tmparr4 = split (/\./,$tmparr3[1]);
                    my $tmpbaseURLParent = $tmparr4[0] . $tmparr4[1];

                    my @tmparr5 = split (/\//,$tmparr1[1]);
                    my $baseurl = "";
                    if($#tmparr5>0) {
                        $baseurl = $tmparr5[0];
                    }else {
                        $baseurl = $tmparr1[1];
                    }
                    $baseurl =~ s/\/$//;

                    if($tmpbaseURLChild ne $tmpbaseURLParent) {
                        #Praca na linku zewntrznym
                        if( !(exists $linkHash{$baseurl}) ){
                            #Pobieranie wartoci PR / losowy czas upienia
                            my $randNum = int(rand($range));
                            #print "\nOczekiwanie przez $randNum sekund.\n";
            
                            sleep($randNum);
                            my $tmpurl = $PRURL . $baseurl;
                            my $PR = get $tmpurl;
                            #print "$PR:";
                            $PR =~ s/\n//g;
                            $PRs = $PRs . $PR . "|";
                            $linkHash{$baseurl} = 1;
                        }
                    }
                }
            }
        }
        else {
            #print "\nPlik: $filename nie zosta znaleziony!";
        }
        $destArr->[$i] = $PRs;
        #print "\n$PRs";
    }
}

# Podprogram:
#   outboundLinkKeywordAnalysis
# Opis:
#   Ten podprogram analizuje sowa kluczowe w linkach zewntrznych
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $srcLinksArr => macierz (odwoanie) do macierzy linkw
#   $prefix => Prefiks pliku wyszukiwarki
#   $keyword => sowo kluczowe
# Zmienne wyjciowe:
#   Drukuje map sw kluczowych
# Zwraca:
#   Nic
sub outboundLinkKeywordAnalysis {
    my ($numberOfElements, $srcLinksArr, $destArr, $prefix, $keyword) = @_;
    my @keywordFragments = split(/ /,$keyword);
    #Ptla przez kady z plikw
    for(my $i=0; $i< $numberOfElements; $i++) {
        my $filename = './serptemp/' . $prefix . "$i.txt";
        my $keywordMatchPercent = "";
        my $foundCount = 0;
        my $total = 0;
        #Sprawdzanie, czy plik istnieje
        if (-e "$filename") {
            my $p = HTML::TokeParser->new($filename);
            while (my $token = $p->get_tag("a")) {
                #Pobieranie linka i tekstu kotwiczki
                my $url = $token->[1]{href} || "-";
                my $text = $p->get_trimmed_text("/a");
                $text =~ s/"//;
                $text =~ s/'//;

                #Sprawdzenie, czy link jest wewntrzny, czy zewntrzny
                if($url =~ /^http/) {
                    @tmparr1 = split (/\/\//,$url);
                    @tmparr2 = split (/\./,$tmparr1[1]);
                    $tmpbaseURLChild = $tmparr2[0] . $tmparr2[1];
    
                    @tmparr3 = split (/\/\//,$srcLinksArr->[$i]);
                    @tmparr4 = split (/\./,$tmparr3[1]);
                    $tmpbaseURLParent = $tmparr4[0] . $tmparr4[1];
                    if($tmpbaseURLChild ne $tmpbaseURLParent) {
                        #Link zewntrzny...przetwarzanie
                        if($#keywordFragments >0){
                            #Obsuga wielu sw kluczowych
                            for(my $j=0; $j <= $#keywordFragments; $j++){
                                #Sprawdzenie dopasowania
                                if($text =~ /$keywordFragments[$j]/i) {
                                #Dopasowanie znalezione
                                $foundCount++;
                                last;
                            }
                        }
                    } else {
        
                            if($text =~ /$keyword/i) {
                                #Dopasowanie znalezione
                                $foundCount++;
                            }
                        }
                    }
                }
                $total++;
            }
        }
        else {
        #print "\nPlik: $filename nie zosta znaleziony!";
        }
        if($total>0) {
            $destArr->[$i] = ( $foundCount);
        } else {
            $destArr->[$i] = 0;
        }
        #print "\n$destArr->[$i]";
    }
}

# Podprogram:
#   printKeywordProminenceMap
# Opis:
#   Ten podprogram drukuje map kadego adresu URL
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $srcArr => macierz (odwoanie) do macierzy linkw
# Zmienne wyjciowe:
#   Drukuje map sw kluczowych
# Zwraca:
#   Nic
sub printKeywordProminenceMap {
    my ($srcArr, $numberOfElements) = @_;
    for(my $i; $i<$numberOfElements; $i++){
        print "$srcArr->[$index]\n";
    }
}

# Podprogram:
#   printIndividualKeywordProminenceMap
# Opis:
#   Ten podprogram drukuje map kadego adresu URL
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $srcArr => macierz (odwoanie) do macierzy wynikw
# Zmienne wyjciowe:
#   Drukuje map sw kluczowych
# Zwraca:
#   Nic
sub printIndividualKeywordProminenceMap {
    my ($index, $srcArr, $prefix) = @_;
    my $filename = "./report/maps/$prefix".$index.".html";
    open FILE, ">", "$filename" or die $!;
    print FILE "<html><head><title>\n";
    print FILE "Mapa wag sw kluczowych\n";
    print FILE "</title></head>\n";
    print FILE "<body><table width=400 cellpading=2 cellspacing=0><tr><td width=400>";
    print FILE $srcArr->[$index];
    print FILE "</td></tr></table></body></html>";
    close FILE;
}

# Podprogram:
#   analyzeKeywordPositions
# Opis:
#   Ten podprogram analizuje relatywne pozycje sw kluczowych w treci witryny
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $destArr => macierz (odwoanie) do macierzy wynikw
#   $keyword => sowo kluczowe do analizy
#   $prefix => prefiks pliku
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach

sub analyzeKeywordPositions {
    my ($numberOfElements, $destArr, $destArr2, $prefix, $keyword) = @_;
    my @keywordFragments = split(/ /,$keyword);
    #Ptla przez wszystkie pliki do pobrania
    for(my $i=0; $i< $numberOfElements; $i++) {
        my $pageCopy = "";
        my $tmpMap = ":";
        my $filename = './serptemp/' . $prefix . "$i.txt";
        #Sprawdzanie, czy plik istnieje
        if (-e "$filename"){
            my $p = HTML::TokeParser->new($filename);
            #Pobranie treci strony dla tego pliku
            while (my $token = $p->get_tag("body")) {
                $pageCopy = $p->get_trimmed_text("/body");
                $pageCopy = cleanText($pageCopy);
            }
            $pageCopy =~ s/s+/ /g;
            my @tempArr = split(/ /, $pageCopy);
            $totalWords = $#tempArr;
            #print "\nSuma wyrazw na tej stronie: $totalWords";
            #Ptla przez wszystkie wyrazy
            for(my $j=0; $j < $totalWords; $j++){
                my $flag = "N";
                if($#keywordFragments >0){
                    #Obsuga wielu sw kluczowych
                    for(my $k=0; $k <= $#keywordFragments; $k++){
                    #Sprawdzanie dopasowania
                    if($tempArr[$j] =~ /$keywordFragments[$k]/i) {
                        #Aktualizacja zmiennej docelowej indeksem macierzy sw kluczowych
                        $destArr->[$i] .= "$k ";
                        #Aktualizacja zmiennej docelowej pozycj wzgldn
                        $destArr2->[$i] = $destArr2->[$i] ."$j" . "|";
                        $flag = "T";
                        last;
                    } else {
                        if( ($k == $#keywordFragments) && ($flag ne "T") ) {
                                $destArr->[$i] .= "* ";
                            }
                        }
                    }
                } else {
                    #Obsuga pojedynczego sowa kluczowego
                    $tempArr[$j] =~ s/"//;
                    $tempArr[$j] =~ s/'//;

                    if($tempArr[$j] =~ /$keyword/i){
                        $destArr->[$i] .= "0 ";
                        $destArr2->[$i] = $destArr2->[$i] . "$j" . "|";
                        $flag = "T";
                    } else {
                        $destArr->[$i] .= "* ";
                    }
                }
                if($flag ne "N") {
                    $destArr->[$i] .= "* ";
                }
            }
            #print "\n\n$destArr->[$i]";
        } else {
            print "\nPlik nie istnieje";
        }
    }
}

# Podprogram:
#   checkHeaderTags
# Opis:
#   Ten podprogram sprawdza, czy uywane s znaczniki nagwkw, a jeli s, to czy 
#   w nagwkach uywane s sowa kluczowe.
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $destArr1 => macierz (odwoanie) do macierzy wynikw
#   $destArr2 => macierz (odwoanie) do macierzy wynikw
#   $keyword => sowo kluczowe do analizy
#   $prefix => prefiks pliku
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach
sub checkHeaderTags {
    my ($numberOfElements, $destArr1, $destArr2, $prefix, $keyword) = @_;
    my @keywordFragments = split(/ /,$keyword);
    
    for(my $i=0; $i < $numberOfElements; $i++) {
        my $filename = './serptemp/' . $prefix . "$i.txt";
        if (-e "$filename"){
            my $p = HTML::TokeParser->new($filename);
            my $h1Text = "";
            my $h2Text = "";
            my $h3Text = "";
            my $h4Text = "";
            my $h5Text = "";
            my $h6Text = "";
            my $separator = '|s|e|p|a|r|a|t|o|r';
            while(my $token = $p->get_token) {
                if($token->[0] eq 'S' and $token->[1] eq 'h1') {
                    $h1Text = $h1Text . $separator . $p->get_text("/h1");
                }
                if($token->[0] eq 'S' and $token->[1] eq 'h2') {
                    $h2Text = $h2Text . $separatpr . $p->get_text("/h2");
                }
                if($token->[0] eq 'S' and $token->[1] eq 'h3') {
                    $h3Text = $h3Text . $separator . $p->get_text("/h3");
                }
                if($token->[0] eq 'S' and $token->[1] eq 'h4') {
                    $h4Text = $h4Text . $separator . $p->get_text("/h4");
                }
                if($token->[0] eq 'S' and $token->[1] eq 'h5') {
                    $h5Text = $h5Text . $separator . $p->get_text("/h5");
                }
                if($token->[0] eq 'S' and $token->[1] eq 'h6') {
                    $h6Text = $h6Text . $separator . $p->get_text("/h6");
                }
            }
            $h1Text = cleanText($h1Text);
            $h2Text = cleanText($h2Text);
            $h3Text = cleanText($h3Text);
            $h4Text = cleanText($h4Text);
            $h5Text = cleanText($h5Text);
            $h6Text = cleanText($h6Text);

            my @h1Arr = split($separator, $h1Text);
            my @h2Arr = split($separator, $h2Text);
            my @h3Arr = split($separator, $h3Text);
            my @h4Arr = split($separator, $h4Text);
            my @h5Arr = split($separator, $h5Text);
            my @h6Arr = split($separator, $h6Text);
        
            my $h1Cnt = ($#h1Arr == -1) ? 0 : $#h1Arr;
            my $h2Cnt = ($#h2Arr == -1) ? 0 : $#h2Arr;
            my $h3Cnt = ($#h3Arr == -1) ? 0 : $#h3Arr;
            my $h4Cnt = ($#h4Arr == -1) ? 0 : $#h4Arr;
            my $h5Cnt = ($#h5Arr == -1) ? 0 : $#h5Arr;
            my $h6Cnt = ($#h6Arr == -1) ? 0 : $#h6Arr;

            my $h1Flag = "N";
            my $h2Flag = "N";
            my $h3Flag = "N";
            my $h4Flag = "N";
            my $h5Flag = "N";
            my $h6Flag = "N";
            
            $destArr1->[$i] =
"".$h1Cnt."|".$h2Cnt."|".$h3Cnt."|".$h4Cnt."|".$h5Cnt."|".$h6Cnt;
            if($#keywordFragments > 0) {
                #Obsuga wielu sw kluczowych
                for(my $j=0; $j<=$#keywordFragments; $j++) {

                    if( $keywordFragments[$j] =~ /$h1Text/i ) {
                        $h1Flag = "T";
                    }
                    if( $keywordFragments[$j] =~ /$h2Text/i ) {
                        $h2Flag = "T";
                    }
                    if( $keywordFragments[$j] =~ /$h3Text/i ) {
                        $h3Flag = "T";
                    }
                    if( $keywordFragments[$j] =~ /$h4Text/i ) {
                        $h4Flag = "T";
                    }
                    if( $keywordFragments[$j] =~ /$h5Text/i ) {
                        $h5Flag = "T";
                    }
                    if( $keywordFragments[$j] =~ /$h6Text/i ) {
                        $h6Flag = "T";
                    }
                }
            } else {
                #Obsuga sowa kluczowego
                if($keyword =~ /$h1Text/i) {
                    $h1Flag = "T";
                }
                if($keyword =~ /$h2Text/i) {
                    $h2Flag = "T";
                }
                if($keyword =~ /$h3Text/i) {
                    $h3Flag = "T";
                }
                if($keyword =~ /$h4Text/i) {
                    $h4Flag = "T";
                }
                if($keyword =~ /$h5Text/i) {
                    $h5Flag = "T";
                }
                if($keyword =~ /$h6Text/i) {
                    $h6Flag = "Y";
                }
            }
            $destArr2->[$i] =
"".$h1Flag."|".$h2Flag."|".$h3Flag."|".$h4Flag."|".$h5Flag."|".$h6Flag;
        } else {
            # Brak pliku => wstaw wartoci domylne;
            $destArr1->[$i] = "0|0|0|0|0|0|";
            $destArr2->[$i] = "N|N|N|N|N|N|";
        }
        #print "\n".$destArr1->[$i]."\n".$destArr2->[$i];
    }
}

# Podprogram:
#   checkExactDescriptionMeta
# Opis:
#   Ten podprogram sprawdza, czy wystpuj dokadne dopasowania w opisie sw 
#   kluczowych.
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $destArr => macierz (odwoanie) do macierzy wynikw
#   $keyword => sowo kluczowe do analizy
#   $prefix => prefiks pliku
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach
sub checkExactDescriptionMeta {
    my ($numberOfElements, $destArr, $keyword, $prefix) = @_;
    for(my $i=0; $i<$numberOfElements; $i++){
        $filename = './serptemp/' . $prefix . "$i.txt";
        if (-e "$filename"){
            my $p = HTML::TokeParser->new($filename);
            while (my $token=$p->get_tag("meta")) {
                if ($token->[1]{name}=~/description/i) {
                    my $metaDescription = $token->[1]{content};
                    $metaDescription =~ s/"//;
                    $metaDescription =~ s/'//;

                    if($metaDescription =~ /$keyword/i) {
                        $destArr->[$i] = "T";
                    } else {
                        $destArr->[$i] = "N";
                    }
                }
            }
        }
        if ( !(exists $destArr->[$i])) {
            $destArr->[$i] = "N";
        }
    }
}

# Podprogram:
#   checkExactDescriptionMeta
# Opis:
#   Ten podprogram sprawdza, czy wystpuj dokadne dopasowania w opisie sw 
#   kluczowych.
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $destArr => macierz (odwoanie) do macierzy wynikw
#   $keyword => sowo kluczowe do analizy
#   $prefix => prefiks pliku
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach
sub checkPartialDescriptionMeta {
    my ($numberOfElements, $destArr, $keyword, $prefix) = @_;
    my @keywordFragments = split(/ /, $keyword);
    
    for(my $i=0; $i<$numberOfElements; $i++){
        $filename = './serptemp/' . $prefix . "$i.txt";
        if (-e "$filename"){
            my $p = HTML::TokeParser->new($filename);
            while (my $token=$p->get_tag("meta")) {
                if ($token->[1]{name}=~/description/i) {
                    my $metaDescription = $token->[1]{content};

                    if($#keywordFragments >0) {
                        for (my $j=0; $j<=$#keywordFragments; $j++){
                            if($metaDescription =~ /$keywordFragments[$j]/i) {
                                $destArr->[$i] = "T";
                                last;
                            } else {
                                $destArr->[$i] = "N";
                            }
                        }
                    } else {
                        if($metaDescription =~ /$keyword/i) {
                            $destArr->[$i] = "T";
                            last;
                        } else {
                            $destArr->[$i] = "N";
                        }
                    }
                }
            }
        }
        if ( !(exists $destArr->[$i])) {
            $destArr->[$i] = "N";
        }
    }
}

# Podprogram:
#   keywordDensity
# Opis:
#   Ten podprogram oblicza gsto wystpowania danego sowa kluczowego.
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $destArr => macierz (odwoanie) do macierzy wynikw
#   $keyword => sowo kluczowe do analizy
#   $prefix => prefiks pliku
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach

sub keywordDensity {
    my ($numberOfElements, $keyword, $destArr, $prefix) = @_;
    my $total = 0;
    #Ptla przez wszystkie pliki
    
    for(my $i=0; $i<$numberOfElements; $i++) {
        my $pageCopy = "";
        my $filename = './serptemp/' . $prefix . "$i.txt";
        if (-e "$filename"){
            my $p = HTML::TokeParser->new($filename);
            while (my $token = $p->get_tag("body")) {
                $pageCopy = $p->get_trimmed_text("/body");
            }
        } else {
            print "\nW czasie obliczania gstoci wystpowania sowa kluczowego nie odnaleziono pliku.";
        }    
        #Porwnanie treci i macierzy (funkcja sep)
        $pageCopy =~ s/"//g;
        $pageCopy =~ s/'//g;

        $total = $total + calculateKD($i, $pageCopy, $destArr, $keyword);
    }
    return ($total/$numberOfElements);
}

# Podprogram:
#   calcualteKD
# Opis:
#   Pomocniczy podprogram do obliczania gstoci wystpowania sw kluczowych
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $destArr => macierz (odwoanie) do macierzy wynikw
#   $keyword => sowo kluczowe do analizy
#   $prefix => prefiks pliku
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach
sub calculateKD {
    my ($index, $pageCopy, $destArr, $keyword) = @_;

    my @keywordFragments = split (/ /,$keyword);
    if ($#keywordFragments>0) {
        for (my $i=0; $i<= $#keywordFragments; $i++){
            my @tempArr = split(/$keywordFragments[$i]/,$pageCopy);
            my @tempArr2 = split(/ /, $pageCopy);
            if( ($#tempArr == -1) || ($#tempArr2 == -1)) {
                $destArr->[$index] = 0;
            }else {
                $destArr->[$index] = $destArr->[$index] + ($#tempArr/$#tempArr2)*100;
            }
        }
        return $destArr->[$index];

    } else {
        my @tempArr = split(/$keyword/,$pageCopy);
        my @tempArr2 = split(/ /, $pageCopy);
        $destArr->[$index] = ($#tempArr/$#tempArr2)*100;
        return $destArr->[$index];
    }
}

# Podprogram:
#   keywordDomainExactMatch
# Opis:
#   Ten podprogram analizuje sowa kluczowe w nazwach domen. Sprawdza, 
#   czy sowo kluczowe jest czci nazwy domeny.
#   Potencjalne udoskonalenie wie si z wyszukiwaniem odmian sowa kluczowego.
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $linksArr => macierz (odwoanie) do macierzy linkw
#   $destArr => macierz (odwoanie) do macierzy wynikw
#   $keyword => prefiks pliku dla trzech wyszukiwarek
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach
sub keywordDomainExactMatch {
    my ($keyword, $linksArr, $numberOfElements, $destArr) = @_;
    my $matchCnt=0;
    my @keywordFragments = split(/ /, $keyword);
    my $numberOfKeywordFragments = $#keywordFragments;
    my $total = 0;
    for (my $i=0; $i<=$numberOfElements; $i++) {
        $matchCnt=0;
        my $tmp = $linksArr->[$i];
        $tmp =~ s/^http:\/\///g;
        $tmp =~ s/^https:\/\///g;
        my @linkFragments = split(/\//,$tmp);
        my $link = $linkFragments[0];

        if($numberOfKeywordFragments>0) {
            for(my $j=0; $j<=$numberOfKeywordFragments; $j++) {
            if ($link =~ /$keywordFragments[$j]/i) {
                $matchCnt++;
            }
        }
    } else {
        if($link =~ /$keyword/i) {
            $matchCnt++;
        }
    }
    if($matchCnt>0) {
        if($numberOfKeywordFragments>0) {
            if($matchCnt == ($numberOfKeywordFragments+1)) {
                $destArr->[$i] = "T";
                } else {
                    $destArr->[$i] = "N";
                }
            } else {
                # Pojedyncze sowo kluczowe
                $destArr->[$i] = "T";
            }
        } else {
            $destArr->[$i] = "N";
        }
        if($destArr->[$i] eq "T") {
            $total++;
        }
    }
    return ( ($total/$numberOfElements)* 100);
}

# Podprogram:
#   keywordDomainPartialMatch
# Opis:
#   Ten podprogram analizuje sowa kluczowe w nazwach domen. Szuka 
#   czciowych dopasowa midzy sowami kluczowymi, a nazw domeny.
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $linksArr => macierz (odwoanie) do macierzy linkw
#   $destArr => macierz (odwoanie) do macierzy wynikw
#   $keyword => sowo kluczowe do analizy
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach
sub keywordDomainPartialMatch {
    my ($keyword, $linksArr, $numberOfElements, $destArr) = @_;
    my $totalNumber = $numberOfElements;
    my $matchCnt=0;
    my @keywordFragments = split (/ /, $keyword);
    my $numOfKeywordFragments = $#keywordFragments;

    my $keywordHyphen = $keyword;
    my $keywordUnderscore = $keyword;
    my $keywordNoSpace = $keyword;
    $keywordHyphen =~ s/ /-/g;
    $keywordNoSpace =~ s/ //g;

    #Ptla przez wszystkie linki
    if($numOfKeywordFragments >0) {
        for(my $i=0; $i<$numberOfElements; $i++) {
            my $tmp = $linksArr->[$i];
            $tmp =~ s/^http:\/\///gi;
            $tmp =~ s/^https:\/\///gi;
            my @linkFragments = split(/\//,$tmp);
            my $link = $linkFragments[0];
            for(my $j=0; $j<=$numOfKeywordFragments; $j++) {
                if($link =~ /$keywordFragments[$j]/i) {
                    $destArr->[$i] = "T";
                    $j = $numOfKeywordFragments;
                    $matchCnt++;
                } else {
                    $destArr->[$i] = "N";
                }
            }
        }
    } else {

        for(my $i=0; $i<$numberOfElements; $i++) {
            my $tmp = $linksArr->[$i];
            $tmp =~ s/^http:\/\///g;
            $tmp =~ s/^https:\/\///g;
            my @linkFragments = split(/\//,$tmp);
            my $link = $linkFragments[0];

            if( ($link =~ /$keyword/) ||
                ($link =~ /$keywordHyphen/) ||
                ($link =~ /$keywordNoSpace/) ) {
                $destArr->[$i] = "T";
                $matchCnt++;
            } else {
                $destArr->[$i] = "N";
            }
        }

    }
    return ( ($matchCnt/$totalNumber)* 100);
}

# Podprogram:
#   compareTitlePageCopy
# Opis:
#   Ten program porwnuje tytu strony z jej treci
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $titlesArr => macierz (odwoanie) do macierzy tytuw
#   $destArr => macierz (odwoanie) do macierzy wynikw
#   $prefix => prefiks pliku dla trzech wyszukiwarek
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach
sub compareTitlePageCopy {
    my ($numberOfElements, $titlesArr, $destArr, $prefix) = @_;
    #Ptla przez wszystkie pliki
    for(my $i=0; $i<=$numberOfElements; $i++) {
        #split up current title into token words
        my $title = $titlesArr->[$i];

        $title = cleanText($title);
        $title =~ s/\'//g;
        $title =~ s/\"//g;
        
        my @titleFragments = split(/ /,$title);
        #Pobieranie treci dla kadego pliku
        my $pageCopy = "";
        my $filename = './serptemp/' . $prefix . "$i.txt";
        if (-e "$filename"){
            my $p = HTML::TokeParser->new($filename);
            while (my $token = $p->get_tag("body")) {
                $pageCopy = $p->get_trimmed_text("/body");
                $pageCopy =~ s/\'//g;
                $pageCopy =~ s/\"//g;

                last;
            }
        }
        #Porwnanie treci i macierzy (funkcja sep)
        compareTitlePageCopyHelper($i, $#titleFragments,
\@titleFragments, $pageCopy, $destArr);
    }
}

# Podprogram:
#   compareTitlePageCopyHelper
# Opis:
#   Ten podprogram uywany jest przez podprogram compareTitlePageCopy 
#   do porwnywania tytuu strony i jej treci
# Zmienne wejciowe:
#   $index => reprezentuje indeks numeryczny macierzy
#   $numberOfElements => liczba plikw do przetworzenia
#   $titleFragments => macierz (odwoanie) do macierzy fragmentw tytuw
#   $pageCopy => tre strony
#   $pageCopyTitleArr => macierz (odwoanie) do macierzy wynikw
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach
sub compareTitlePageCopyHelper {
    my ($index, $numberOfElements, $titleFragments, $pageCopy, $pageCopyTitleArr) = @_;
    my $foundCnt = 0;
    my $totalTitleFragments = $numberOfElements;

    for(my $j=0; $j<=$numberOfElements; $j++) {
        my $tmpfragment = $titleFragments->[$j];
        
        if( $pageCopy =~ /$tmpfragment/i ){
            $foundCnt++;
        }
    }
    if($foundCnt == 0){
        $pageCopyTitleArr->[$index] = 0;
    } else {
        $pageCopyTitleArr->[$index] = ( ($foundCnt/($totalTitleFragments+1)) * 100);
    }
}

# Podprogram:
#   compareArrays
# Opis:
#   Ten podprogram porwnuje ze sob elementy dwch macierzy, aby sprawdzi
#   czy si w sobie nawzajem znajduj.
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $realArr => macierz (odwoanie) do pierwszej macierzy rdowej
#   $foundArr => macierz (odwoanie) do drugiej macierzy rdowej
#   $destArr => macierz (odwoanie) do macierzy wynikw
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazywanych macierzach
sub compareArrays {
    my ($numOfElements, $realArr, $foundArr, $destArr) = @_;
    my $found = 0;
    my $percentMatch = 0;

    for(my $i=0; $i<$numOfElements; $i++){
        $tmpVar = $foundArr->[$i];
        $tmpVar =~ s/\(/\\\(/g;
        $tmpVar =~ s/\)/\\\)/g;
        $tmpVar =~ s/\-/\\\-/g;
        $tmpVar =~ s/\+/\\\+/g;
        $tmpVar =~ s/\$/\\\$/g;
        $tmpVar =~ s/\^/\\\^/g;
        $tmpVar =~ s/\[/\\\[/g;
        $tmpVar =~ s/\]/\\\]/g;
        $tmpVar =~ s/\}/\\\}/g;
        $tmpVar =~ s/\{/\\\{/g;

        if ($realArr->[$i] =~ /$tmpVar/i) {
            $destArr[$i] = "T";
            $found++;
        }else {
            $destArr[$i] = "N";
        }
    }
    return ( ($found/$numOfElements)*100);
}

# Podprogram:
#   getRealTitles
# Opis:
#   Ten podprogram pobiera rzeczywiste tytuy
# Zmienne wejciowe:
#   $numberOfElements => liczba plikw do przetworzenia
#   $titlesArr => macierz (odwoanie) do macierzy, ktra bdzie zawiera
#   rzeczywiste tytuy
#   $prefix => prefik pliku, ktry ma by uyty
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Podprogram operuje na macierzy zdefiniowanej ju wczeniej poza funkcj.
#   Podprogram nic nie zwraca.
sub getRealTitles {
    my ($numberOfElements, $titlesArr, $prefix) = @_;
    for(my $i=0; $i<$numberOfElements; $i++){
        $filename = './serptemp/' . $prefix . "$i.txt";
        if (-e "$filename"){
            my $p = HTML::TokeParser->new($filename);
            while (my $token = $p->get_token) {
                if ($token->[0] eq "S" and lc $token->[1] eq 'title') {
                    my $title = $p->get_text() || "not found";
                    $title =~ s/^\s+//;
                    $title =~ s/\s+$//;
                    $titlesArr->[$i]=$title;
                    last;
                }
            }
        }else {
            $titlesArr->[$i]="not found";
        }
    }
}

# Podprogram:
#   getKeywordsTitleMatch
# Opis:
#   Ten podprogram porwnuje dane sowo kluczowe z wpisami macierzy
#   budujc jednoczenie trzeci macierz z wynikami tego porwnania.
# Zmienne wejciowe:
#   $keyword => sowo lub fraza kluczowa, na ktrej przeprowadzona bdzie analiza
#   $sourceArr => macierz (odwoanie), ktra ma zosta uyta do porwnania
#   $numOfElements => rozmiar macierzy odwoania
#   $destArr => macierz (odwoanie)zawierajca wyniki porwnania
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Podprogram operuje na macierzy zdefiniowanej ju wczeniej poza funkcj.
#   Podprogram nic nie zwraca.
sub getKeywordsTitleMatch {
    my ($keyword, $sourceArr, $numOfElements, $destArr) = @_;
    $keyword = cleanText($keyword);
    $keyword =~ s/\'//g;
    $keyword =~ s/\"//g;
    @keywordFragments = split(/ /, $keyword);
    my $numberOfKeywordTokens = $#keywordFragments;
    
    for(my $i=0; $i<= $numOfElements; $i++) {
    my $tmp = $sourceArr->[$i];
    $tmp = cleanText($tmp);
    $tmp =~ s/'//;
    $tmp =~ s/"//;
    my $foundCnt = 0;
    if ($numberOfKeywordTokens >0) {
        for(my $j=0; $j<=$#keywordFragments; $j++){
            if ($tmp =~ /$keywordFragments[$j]/i) {
                $foundCnt++;
                
                last;
            }
        }
        } else {
            if ($tmp =~ /$keyword/i) {
                $foundCnt++;
        
            }
        }
        if($foundCnt > 0) {
            $destArr->[$i] = "T";
        } else {
            $destArr->[$i] = "N";
        }
    }
}

# Podprogram:
#   initializeKeyVariables
# Opis:
#   Gwnym celem jest zdefiniowanie macierzy linkw i tytuw, ktre bd 
#   uywane w rnych czciach skryptu.
# Zmienne wejciowe:
#   $keyword => sowo lub fraza kluczowa, na ktrej przeprowadzona bdzie analiza
#   $googleLinksArr => macierz (odwoanie)z linkami Google
#   $googleTitlesArr => macierz (odwoanie)z tytuami Google
#   $yahooLinksArr => macierz (odwoanie)z linkami Yahoo!
#   $yahooTitlesArr => macierz (odwoanie)z tytuami Yahoo!
#   $bingLinksArr => macierz (odwoanie)z linkami Bing
#   $bingTitlesArr => macierz (odwoanie)z tytuami Bing
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Podprogram operuje na przekazywanych macierzach zdefiniowanych ju wczeniej poza funkcj.
#   Podprogram nic nie zwraca.
sub initializeKeyVariables {
    my ($keyword, $googleLinksArr,$googleTitlesArr,
        $yahooLinksArr, $yahooTitlesArr, $bingLinksArr,
        $bingTitlesArr) = @_;
    #Tworzenie agentw
    my $uaGoogle = new LWP::UserAgent;
    my $uaYahoo = new LWP::UserAgent;
    my $uaBing = new LWP::UserAgent;

    #Ustawienie maksymalnego czasu wykonania polecenia na 25 sekund
    $uaGoogle->timeout(25);
    $uaYahoo->timeout(25);
    $uaBing->timeout(25);

    #Konfiguracja agenta
    my $useragent = "Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0)";
    $uaGoogle->agent("$useragent");
    $uaYahoo->agent("$useragent");
    $uaBing->agent("$useragent");

    #Konfigurowanie i pobieranie 100 wynikw wyszukiwania z kadej wyszukiwarki
    my $gurl=
"http://www.google.com/search?num=$numres&hl=en&safe=off&q=$keyword&sa=N";
    my $yurl=
"http://search.yahoo.com/search?p=$keyword&ei=UTF-8&fr=sfp&n=$numres&b=1";
    my $lurl=
"http://search.bing.com/results.aspx?q=$keyword&first=1&count=$numres&";
    
    my $reqGoogle = new HTTP::Request GET => "$gurl";
    my $reqYahoo = new HTTP::Request GET => "$yurl";
    my $reqBing = new HTTP::Request GET => "$lurl";
    
    my $resGoogle = $uaGoogle->request($reqGoogle);
    my $resYahoo = $uaYahoo->request($reqYahoo);
    my $resBing = $uaBing->request($reqBing);

    #Przypisanie stron z wynikami wyszukiwania do specjalnych zmiennych
    my $ghtml = $resGoogle->content;
    my $yhtml = $resYahoo->content;
    my $lhtml = $resBing->content;

    #Pobranie linkw dla kadej strony z wynikami wyszukiwania
    my $streamGoogle = HTML::TokeParser->new(\$ghtml);
    my $streamYahoo = HTML::TokeParser->new(\$yhtml);
    my $streamBing = HTML::TokeParser->new(\$lhtml);

    # Przetwarzanie linkw Google
    my $cnt=0;
    my $threeDots = '...';
    while (my $token = $streamGoogle->get_token) {
        if ($token->[0] eq 'S' && $token->[1] eq 'a') {
            if( ($token->[2]{'href'} !~ /cache/i) &&
            !($token->[2]{'href'} !~ /^http/i) &&
             ($token->[2]{'href'} !~ /^https/i) &&
             ($token->[2]{'href'} !~ /google/i) &&
             ($token->[2]{'href'} !~ /aclk/i) &&
             ($token->[2]{'href'} !~ /youtube/i)&&
             ($token->[2]{'href'} !~ /wikipedia/i) ) {
             $googleLinksArr->[$cnt] = $token->[2]{'href'};
             $googleTitlesArr->[$cnt] = $streamGoogle->get_trimmed_text("/a");
             $googleTitlesArr->[$cnt] =~ s/$threeDots$//;
             $cnt++;
            }
        }
    }
    # Przetwarzanie linkw Yahoo
    my $cnt2=0;
    while (my $token = $streamYahoo->get_token) {
        if ($token->[0] eq 'S' && $token->[1] eq 'a') {
            @tmpurl= split (/\*\*/, $token->[2]{'href'});
            $tmpurl[1] =~ s/%3f/?/g;
            $tmpurl[1] =~ s/%26/&/g;
            
            if( ($tmpurl[1] !~ /cache/i) &&
                 ($tmpurl[1] !~ /^https/i) &&
                 ($tmpurl[1] !~ /yahoo/i) &&
                 ($tmpurl[1] !~ /wikipedia/i) &&
                 ($tmpurl[1] !~ /overture/i) ){
                $tmpurl[1] =~ s/%3a/:/g;
                $tmpurl[1] =~ s/^s+//g;
                if( $tmpurl[1] ne "") {
                    $yahooLinksArr->[$cnt2] = $tmpurl[1];
                $yahooTitlesArr->[$cnt2] = $streamYahoo->get_trimmed_text("/a");
                    $yahooTitlesArr->[$cnt2] =~ s/$threeDots$//;
                    $cnt2++;
                }
            }
        }
    }
    # Przetwarzanie linkw Bing
    my $cnt3=0;
    while (my $token = $streamBing->get_token) {
        if ($token->[0] eq 'S' && $token->[1] eq 'a') {
            if( !($token->[2]{'href'} !~ /^http/i) &&
                 ($token->[2]{'href'} !~ /^https/i) &&
                 ($token->[2]{'href'} !~ /cache/i) &&
                 ($token->[2]{'href'} !~ /wikipedia/i) &&
                 ($token->[2]{'href'} !~ /msn/i) &&
                 ($token->[2]{'href'} !~ /hotmail/i) &&
                 ($token->[2]{'href'} !~ /microsoft/i) &&
                 ($token->[2]{'href'} !~ /bing\.com/i) ) {
                  $token->[2]{'href'} =~ s/^s+//g;
                  if($token->[2]{'href'} ne "") {
                    $bingLinksArr->[$cnt3] = $token->[2]{'href'};
                $bingTitlesArr->[$cnt3] = $streamBing->get_trimmed_text("/a");
                    $bingTitlesArr->[$cnt3] =~ s/$threeDots$//;
                    $cnt3++;
                }
            }
        }
    }
}

# Podprogram:
#   getSERPResults
# Opis:
#   Ten podprogram pobiera wszystkie pliki htmls wszystkich zdefiniowanych adresw urls 
#   zdefiniowanych w macierzy, do ktrej odwouje si zmienna $urlArr
# Zmienne wejciowe:
#   $numberOfElements => rozmiar macierzy odwoania
#   $urlArr => macierz (odwoanie)z adresami URL do przetworzenia
#   $name => prefiks pliku, ktry ma by uyty
# Zmienne wyjciowe:
#   pliki tekstowe z kodem html z pobranych linkw
# Zwraca:
#   Podprogram operuje na macierzy zdefiniowanej ju wczeniej poza funkcj.
#   Podprogram nic nie zwraca.
sub getSERPResults {
    my ($numberOfElements, $urlArr, $name) = @_;
    my $ua = new LWP::UserAgent;
    my $res;

    $ua->timeout(25);
    $ua->agent("My Crawler");
    
    for($i=0;$i<$numberOfElements;$i++){
        $filename = "./serptemp/". $name . $i . ".txt";
        $res = $ua->get("$urlArr->[$i]",':content_file' => "$filename");
    }
}

# Podprogram:
#   cleanText
# Opis:
#   Ten podprogram uywany jest to wyczyszczenia fragemntw kodu HTML.
# Zmienne wejciowe:
#   $text => tre pliku do wyczyszczenia
# Zmienne wyjciowe:
#   Brak
# Zwraca:
#   Nic, caa praca wykonywana na przekazanej macierzy
sub cleanText {
    my $text = shift;
       $text =~ s/\(/ /g;
       $text =~ s/\)/ /g;
       $text =~ s/\[/ /g;
       $text =~ s/\]/ /g;
       $text =~ s/\./ /g;
       $text =~ s/\-/ /g;
       $text =~ s/\=/ /g;
       $text =~ s/\|/ /g;
       $text =~ s/\!/ /g;
       $text =~ s/\,/ /g;
       $text =~ s/\?/ /g;
       $text =~ s/\^/ /g;
       $text =~ s/\:/ /g;
       $text =~ s/\;/ /g;
       $text =~ s/\&/ /g;
       $text =~ s/\*/ /g;
       $text =~ s/\$/ /g;
       $text =~ s/\s+/ /g;
    return $text;
}
