package scalax.functional

/** Ta klasa pozwala na 'spłaszczenie' (flatten) morfizmu funktora. 
  * Jeśli funktor F[_] został zastosowany dwukrotnie, np. List[List[Int]],
  * to ta klasa potrafi zredukować to do jednego zastosowania,
  * np. List[Int].
 */
trait Monad[M[_]] {
  /**
   * Ta metoda pobiera dwa zastosowania funktora i redukuje je do jednego zastosowania.
   * 
   * Zasadniczo, metoda ta pobiera kontenery zagnieżdżone w kontenerach i usuwa kontener zewnętrzny. 
   */
  def flatten[A](m: M[M[A]]): M[A]
  /**
   * Ta metoda została dodana, ponieważ często bezpośrednia implementacja flatMap
   * jest bardziej efektywna niż wykorzystanie map i flatten.   
   */
  def flatMap[A,B](m: M[A])(f: A=>M[B])(implicit functor : Functor[M]): M[B] = flatten(functor.map(m)(f))
}

object Monad {
  implicit object TraversableFunctor extends Monad[Traversable] {
    override def flatten[A](m: Traversable[Traversable[A]]): Traversable[A] = m.flatten
    override def flatMap[A,B](m: Traversable[A])(f: A=>Traversable[B])(implicit functor : Functor[Traversable]): Traversable[B] =
      m flatMap f
  }
  implicit object OptionFunctor extends Monad[Option] {
    override def flatten[A](m: Option[Option[A]]): Option[A] = m flatMap identity
    override def flatMap[A,B](m: Option[A])(f: A=>Option[B])(implicit functor : Functor[Option]): Option[B] =
      m.flatMap(f)
  }
}

final class MonadOps[M[_], A](val m : M[A], val monad: Monad[M]) {
  @inline final def flatMap[B](f: A=>M[B])(implicit functor: Functor[M]): M[B] = monad.flatMap(m)(f)
  @inline final def flatten[B](implicit ev : M[A] <:< M[M[B]]): M[B] = monad.flatten(m)
}

trait MonadImplicits {
  implicit def monadOps[M[_], A](value: M[A])(implicit monad: Monad[M]) : MonadOps[M, A] =
    new MonadOps[M,A](value, monad)
}