package library

import java.util.concurrent.{Callable, Executors}
import collection.mutable.ArrayBuffer

/** Ta klasa definiuje intefejs określający zasady wykonywania funkcji. */
trait ThreadStrategy {
  // Przekaż funkcję zwracającą wartość i otrzymaj funkcję zwracającą wartość.
  // Sama funkcja może zostać wykonana w innym wątku.
  def execute[A](func : Function0[A]) : Function0[A]
}

/** Ta klasa przechowuje gęstą dwuwymiarową macierz o skończonym rozmiarze. */
class Matrix(private val repr : Array[Array[Double]]) {
  /** Dostęp do wiersza o numerze idx (liczone od 0). Zwraca sekwencję wartości w kolejnych kolumnach tego wiersza. */  
  def row(idx : Int) : Seq[Double] = {
    repr(idx)
  }
  /** Dostęp do kolumny o numerze idx (liczone od 0). Zwraca sekwencję wartości w kolejnych wierszach tej kolumny. */  
  def col(idx : Int) : Seq[Double] = {
    repr.foldLeft(ArrayBuffer[Double]()) {
      (buffer, currentRow) =>
        buffer.append(currentRow(idx))
        buffer
    } toArray 
  }
  /** Liczba wierszy macierzy. */
  lazy val rowRank = repr.size
  /** Liczba kolumn macierzy. */
  lazy val colRank = if(rowRank > 0) repr(0).size else 0
  /** Eleganckie wypisanie zawartości macierzy */
  override def toString = "Macierz" + repr.foldLeft("") { (msg, row) => msg + row.mkString("\n|", " | ", "|")}
}

/** Definiuje usługę mnożącą dwie macierze i pozwala na zmianę strategii wątków. */
object MatrixService {
  /** Ta metoda mnoży dwie macierze. Pobiera domniemany parametr, pozwalający na zmianę strategii wątków. */
  def  multiply(a : Matrix, b : Matrix)(implicit threading : ThreadStrategy = ThreadStrategy.SameThreadStrategy) : Matrix = {
	// Upewnij się, że rozmiar macierzy pozwala na mnożenie. 
    assert(a.colRank == b.rowRank)
	// Stwórz bufor do przechowywania wyników. 
	// Rozmiar jest określany przez liczbę wierszy a i liczbę kolumny w b.     
    val buffer = new Array[Array[Double]](a.rowRank)
    for ( i <- 0 until a.rowRank ) {
      buffer(i) = new Array[Double](b.colRank)
    }
	// Ta funkcja pomocnicza wyliczy wartość przechowywaną na pozycji index(row,col) macierzy wynikowej
	// i umieści ją w buforze.     
    def computeValue(row : Int, col : Int) : Unit = {
       // Tworzy listę par elementów z dwóch macierzy.
       val pairwiseElements =
         a.row(row).zip(b.col(col))
	   // Mnoży wartości z wiersza przez wartości z kolumny. Suma tych iloczynów to wynikowa wartość dla danej pozycji macierzy.
       val products = 
         for((x,y) <- pairwiseElements) 
         yield x*y
       val result = products.sum
       buffer(row)(col) = result
    }
	// Utwórz listę obliczeń dla każdego wyniku (row,col) w macierzy.
    val computations = for {
      i <- 0 until a.rowRank
      j <- 0 until b.colRank
    } yield threading.execute { () => computeValue(i,j) }
	// Wykonaj wszystkie obliczenia *lub* poczekaj na zakończenie strategii
    computations.foreach(_())
    new Matrix(buffer)
  }
}

/** Obiekt towarzyszacy cechy ThreadStrategy.
 * Definiuje różne strategie puli wątków, z których nikt nie korzysta. */
object ThreadStrategy {
  /** Ta strategia wykona wszystkie funkcje w wątku lokalnym. */
  object SameThreadStrategy extends ThreadStrategy {
    def execute[A](func : Function0[A]) = func
  }

  /** Ta strategia wykona wszystkie funkcje w puli wątków. */
  object ThreadPoolStrategy extends ThreadStrategy {
    val pool = Executors.newFixedThreadPool(java.lang.Runtime.getRuntime.availableProcessors)
    def execute[A](func : Function0[A] ) = {
	  // Dodaj do puli wątków klasę Callabale.
      val future = pool.submit(new Callable[A] {
        def call() : A = {
          Console.println("Wykonanie funkcji w wątku: : " + Thread.currentThread.getName)
          func()
        }
      })
      // Stwórz funkcję, która po wywołaniu zablokuje kod i poczeka, aż opóźnione wątki zakończą działanie.
      () => future.get()
    }
  }
}
