/** Definiuje bazową cechę logującą, której klasy mogą używać do logowania komunikatów o aktualnym stanie. */
trait Logger {
  def log(category : String, msg : String) : Unit = {
       println(msg)
  }
}
/** Logger przesyłający komunikaty przez zdalne gniazdo (niezaimplementowany) */
trait RemoteLogger extends Logger {
  //val socket = ..
  override def log(category : String, msg : String) : Unit = {
    //Wyślij przez gniazdo
  }
}

/** Logger, który nic nie robi. */
trait NullLogger extends Logger {
   override def log(category : String, msg : String) : Unit = {}
}

/** Ta cecha może zostać rozszerzona przez klasę, która korzysta z usługi Logger.
  * W ten sposób można 'konfigurować' klasy poprzez domieszki cech 
  * i tworzenie odpowiednich instancji loggerów.  
 */
trait HasLogger {
  val logger : Logger = new Logger {}
}

/**
 * Ta usługa pozwala na dostęp do danych za pomocą zapytań. 
 * Rozszerza ona klasę HasLogger i może zostać skonfigurowana podczas tworzenia instancji. 
 */
trait DataAccess extends HasLogger {

   def query[A](in : String) : A = {
     logger.log("ZAPYTANIE", in)     
     null.asInstanceOf[A]
   }
}

/** Tę cechę można wmieszać do dowolnej klasy HasLogger w celu dodania zdalnego loggera. */
trait HasRemoteLogger extends HasLogger {
  override val logger : Logger = new RemoteLogger {}
}

/** Tę cechę można wmieszać do dowolnej klasy HasLogger w celu wyłączenia logowania. */
trait HasNullLogger extends HasLogger {
  override val logger : Logger = new NullLogger {}
}


// Foo to instancja DataAccess skonfigurowana z loggerem, z którego chcemy korzystać.
object foo extends DataAccess with HasNullLogger {}


// Mechanizm definiowania zagnieżdżonych usług, których elementy nadal są konfigurowalne.
trait HasDataAccess extends HasLogger {
  
}
