import {
  generateAssertionOptions,
  generateAttestationOptions,
  verifyAttestationResponse,
  verifyAssertionResponse,
} from '@simplewebauthn/server'
import express from 'express'
import cookieParser from 'cookie-parser'

const rpName = 'Physical Token Server'
const rpID = 'localhost'
const origin = `https://${rpID}:3000`

const users = [
  {
    id: '1234',
    username: 'franek',
    password: 'tajnehaslo',
    devices: [],
    authenticators: {},
  },
]

const sessions = {}

const app = express()
app.use(cookieParser())

app.use(express.json())

app.post('/login', (request, response) => {
  const { username, password } = request.body
  const user = users.find(
    (u) => u.username === username && u.password === password
  )
  if (!user) {
    return response
      .status(403)
      .send({ message: 'Nieprawidłowa nazwa użytkownika lub hasło' })
  }
  if (Object.keys(user.authenticators).length) {
    return response.status(200).send({
      userID: user.id,
      message: 'Wymagany klucz bezpieczeństwa',
      twoFactorNeeded: true,
    })
  }
  const newSession = Math.random()
  sessions[newSession] = { user }
  return response
    .status(200)
    .cookie('__session', newSession)
    .send({ userID: user.id, message: 'Zalogowano' })
})

app.post('/logout', (request, response) => {
  if (request.cookies && request.cookies['__session']) {
    delete sessions[request.cookies['__session']]
  }

  return response
    .status(200)
    .clearCookie('__session')
    .send({ message: 'Wylogowano' })
})

const secured = (callback) => async (req, res) => {
  if (!req.cookies || !req.cookies['__session']) {
    res.status(401).send('Brak autoryzacji')
    return
  }

  const isValid =
    req.cookies['__session'] && sessions[req.cookies['__session']]

  if (!isValid) {
    res.status(401).send('Brak autoryzacji')
    return
  }

  await callback(req, res)
}

app.get(
  '/account',
  secured((request, response) => {
    const sessionID = request.cookies['__session']
    const { user } = sessions[sessionID]
    response.send({ user })
  })
)

app.post('/startRegister', (request, response) => {
  const sessionID = request.cookies['__session']
  const { user } = sessions[sessionID]
  const userID = user.id

  const options = generateAttestationOptions({
    rpName,
    rpID,
    userID: userID,
    userName: user.username,
    excludeCredentials: user.devices.map((dev) => ({
      id: dev.credentialID,
      type: 'public-key',
    })),
    authenticatorSelection: { userVerification: 'discouraged' },
    extensions: {
      credProps: true,
    },
  })

  user.currentChallenge = options.challenge
  response.send(options)
})

app.post('/startFingerprint', (request, response) => {
  const sessionID = request.cookies['__session']
  const { user } = sessions[sessionID]

  const options = generateAttestationOptions({
    rpName,
    rpID,
    userID: userID,
    userName: user.username,
    excludeCredentials: user.devices.map((dev) => ({
      id: dev.credentialID,
      type: 'public-key',
    })),
    extensions: {
      credProps: true,
    },
    authenticatorSelection: {
      authenticatorAttachment: 'platform',
      userVerification: 'required',
    },
  })

  options.attestation = 'direct'

  user.currentChallenge = options.challenge
  console.log('odsyłamy opcje', options)
  response.send(options)
})

app.post('/register', async (request, response) => {
  const { attestation } = request.body
  const sessionID = request.cookies['__session']
  const { user } = sessions[sessionID]

  const expectedChallenge = user.currentChallenge

  let verification
  try {
    verification = await verifyAttestationResponse({
      credential: attestation,
      expectedChallenge,
      expectedOrigin: origin,
      expectedRPID: rpID,
    })
  } catch (error) {
    console.error(error)
    return response.status(400).send({ error: error.message })
  }

  const { verified, authenticatorInfo } = verification

  const { base64PublicKey, base64CredentialID, counter } =
    authenticatorInfo

  user.authenticators[base64CredentialID] = {
    credentialID: base64CredentialID,
    publicKey: base64PublicKey,
    counter,
  }

  user.devices.push({ credentialID: base64CredentialID })

  response.send({ verified })
})

app.post('/startVerify', (request, response) => {
  const { userID } = request.body
  const user = users.find((u) => u.id === userID)

  const options = generateAssertionOptions({
    allowCredentials: user.devices.map((dev) => ({
      id: dev.credentialID,
      type: 'public-key',
    })),
    attestation: 'direct',
    extensions: {
      credProps: true,
    },
    rpID,
  })

  user.currentChallenge = options.challenge
  console.log('odsyłamy opcje', options)
  response.send(options)
})

app.post('/verify', async (request, response) => {
  const { userID, assertion } = request.body
  const user = users.find((u) => u.id === userID)

  const expectedChallenge = user.currentChallenge

  const authenticator = user.authenticators[assertion.id]

  if (!authenticator) {
    response
      .status(403)
      .send(
        `Nie udało się znaleźć danych uwierzytelniających ${assertion.id} dla użytkownika ${user.id}`
      )
  }

  let verification
  try {
    verification = await verifyAssertionResponse({
      credential: assertion,
      expectedChallenge,
      expectedOrigin: origin,
      expectedRPID: rpID,
      authenticator,
    })
  } catch (error) {
    console.error(error)
    return response.status(400).send({ error: error.message })
  }

  const { verified, authenticatorInfo } = verification

  const { counter } = authenticatorInfo

  user.authenticators[assertion.id].counter = counter

  const newSession = Math.random()
  sessions[newSession] = { user }

  response
    .status(200)
    .cookie('__session', newSession)
    .send({ verified })
})

app.listen(5000, () => console.log('🚀 Serwer uruchomiony na porcie 5000!'))
