"""Animacja księżyca autorstwa Erica T. Mortensona."""
import math
import numpy as np
import cv2 as cv
import matplotlib.pyplot as plt

IMG_HT = 500
IMG_WIDTH = 500
BLACK_IMG = np.zeros((IMG_HT, IMG_WIDTH, 1), dtype='uint8')
STAR_RADIUS = 200
EXO_RADIUS = 20
MOON_RADIUS = 5
EXO_START_X = 20
EXO_START_Y = 250
NUM_DAYS = 200  # Liczba dni w roku


def main():
    intensity_samples = record_transit(EXO_START_X, EXO_START_Y)
    relative_brightness = calc_rel_brightness(intensity_samples)
    print('\nszacunkowy promien egzoplanety = {:.2f}\n'
          .format(STAR_RADIUS * math.sqrt(max(relative_brightness)
                                          - min(relative_brightness))))
    plot_light_curve(relative_brightness)


def record_transit(exo_x, exo_y):
    """
    Rysuje tranzyt planety na tle gwiazdy
    i zwraca listę zmian intensywności.
    """
    intensity_samples = []
    for dt in range(NUM_DAYS):
        temp_img = BLACK_IMG.copy()
        # Rysuje gwiazdę.
        cv.circle(temp_img, (int(IMG_WIDTH / 2), int(IMG_HT / 2)),
                  STAR_RADIUS, 255, -1)
        # Rysuje egzoplanetę.
        cv.circle(temp_img, (int(exo_x), int(exo_y)), EXO_RADIUS, 0, -1)
        # Rysuje księżyc.
        if dt != 0:
            cv.circle(temp_img, (int(moon_x), int(moon_y)), MOON_RADIUS, 0, -1)
        intensity = temp_img.mean()
        cv.putText(temp_img, 'Srednia intensywnosc = {}'.format(intensity),
                   (5, 10), cv.FONT_HERSHEY_PLAIN, 1, 255)
        cv.imshow('Tranzyt', temp_img)
        cv.waitKey(10)
        intensity_samples.append(intensity)
        exo_x = IMG_WIDTH / 2 - (IMG_WIDTH / 2 - 20) * \
                math.cos(2 * math.pi * dt / (NUM_DAYS) * (1 / 2))
        moon_x = exo_x + \
                 3 * EXO_RADIUS * math.sin(2 * math.pi * dt / NUM_DAYS * (5))
        moon_y = IMG_HT / 2 - \
                 0.25 * EXO_RADIUS * \
                 math.sin(2 * math.pi * dt / NUM_DAYS * (5))
    cv.destroyAllWindows()

    return intensity_samples


def calc_rel_brightness(intensity_samples):
    """Zwraca listę wyników względnej jasności na podstawie intensywności."""
    rel_brightness = []
    max_brightness = max(intensity_samples)
    for intensity in intensity_samples:
        rel_brightness.append(intensity / max_brightness)
    return rel_brightness


def plot_light_curve(rel_brightness):
    """Rysuje wykres zmian względnej jasności w czasie."""
    plt.plot(rel_brightness, color='red', linestyle='dashed',
             linewidth=2, label='Względna jasność')
    plt.legend(loc='upper center')
    plt.title('Względna jasność w czasie')
    plt.show()


if __name__ == '__main__':
    main()
