import numpy as np

# Funkcja wyliczająca błąd średniokwadratowy dla zbioru wartości ciągłych
def mse(targets):
    # Jeżeli zbiór jest pusty
    if targets.size == 0:
        return 0
    return np.var(targets)

def weighted_mse(groups):
    """
    Funkcja wyliczająca ważony błąd średniokwadratowy po podziale węzła
    """
    total = sum(len(group) for group in groups)
    weighted_sum = 0.0
    for group in groups:
        weighted_sum += len(group) / float(total) * mse(group)
    return weighted_sum

print(f'{mse(np.array([1, 2, 3])):.4f}')
print(f'{weighted_mse([np.array([1, 2, 3]), np.array([1, 2])]):.4f}')
print(f'rodzaj-bliźniak: {weighted_mse([np.array([600, 400, 700]), np.array([700, 800])]):.4f}')
print(f'łazienki-2: {weighted_mse([np.array([700, 400]), np.array([600, 800, 700])]):.4f}')
print(f'łazienki-3: {weighted_mse([np.array([600, 800]), np.array([700, 400, 700])]):.4f}')
print(f'łazienki-4: {weighted_mse([np.array([700]), np.array([600, 700, 800, 400])]):.4f}')
print(f'łazienki-2: {weighted_mse([np.array([]), np.array([600, 400, 700])]):.4f}')
print(f'łazienki-3: {weighted_mse([np.array([400]), np.array([600, 700])]):.4f}')
print(f'łazienki-4: {weighted_mse([np.array([400, 600]), np.array([700])]):.4f}')

def split_node(X, y, index, value):
    """
    Funkcja dzieląca zbiór X na podstawie cechy i wartości
    @param index: indeks cechy wykorzystywanej do dzielenia
    @param value: wartość cechy wykorzystywanej do dzielenia
    @return: dwie listy w formacie [X, y] reprezentujące węzły potomne lewy i prawy
    """
    x_index = X[:, index]
    # Jeżeli cecha jest liczbowa
    if type(X[0, index]) in [int, float]:
        mask = x_index >= value
    # Jeżeli cecha jest kategorialna
    else:
        mask = x_index == value
    # Podział na węzły potomne lewy i prawy
    left = [X[~mask, :], y[~mask]]
    right = [X[mask, :], y[mask]]
    return left, right

def get_best_split(X, y):
    """
    Funkcja wyszukująca najlepszy punkt podziału zbioru X, y i zwracająca węzły potomne
    @return: {index: indeks cechy, value: wartość cechy, children: węzły potomne lewy i prawy }
    """
    best_index, best_value, best_score, children = None, None, 1e10, None
    for index in range(len(X[0])):
        for value in np.sort(np.unique(X[:, index])):
            groups = split_node(X, y, index, value)
            impurity = weighted_mse([groups[0][1], groups[1][1]])
            if impurity < best_score:
                best_index, best_value, best_score, children = index, value, impurity, groups
    return {'index': best_index, 'value': best_value, 'children': children}

def get_leaf(targets):
    # Zwrócenie liścia z uśrednioną wartością docelową
    return np.mean(targets)

def split(node, max_depth, min_size, depth):
    """
    Funkcja dzieląca węzeł lub przypisująca mu wartość końcową
    @param node: słownik z informacjami o węźle
    @param max_depth: maksymalna głębokość drzewa
    @param min_size: minimalna liczba próbek wymagana do podziału węzła
    @param depth: głębokość aktualnego węzła
    """
    left, right = node['children']
    del (node['children'])
    if left[1].size == 0:
        node['right'] = get_leaf(right[1])
        return
    if right[1].size == 0:
        node['left'] = get_leaf(left[1])
        return
    # Sprawdzenie, czy aktualna głębokość nie przekracza maksymalnej
    if depth >= max_depth:
        node['left'], node['right'] = get_leaf(left[1]), get_leaf(right[1])
        return
    # Sprawdzenie, czy lewy węzeł potomny zawiera wystarczającą liczbę próbek
    if left[1].size <= min_size:
        node['left'] = get_leaf(left[1])
    else:
        # Jeżeli tak, dzielimy go dalej
        result = get_best_split(left[0], left[1])
        result_left, result_right = result['children']
        if result_left[1].size == 0:
            node['left'] = get_leaf(result_right[1])
        elif result_right[1].size == 0:
            node['left'] = get_leaf(result_left[1])
        else:
            node['left'] = result
            split(node['left'], max_depth, min_size, depth + 1)
    # Sprawdzenie, czy prawy węzeł potomny zawiera wystarczającą liczbę próbek
    if right[1].size <= min_size:
        node['right'] = get_leaf(right[1])
    else:
        # Jeżeli tak, dzielimy go dalej
        result = get_best_split(right[0], right[1])
        result_left, result_right = result['children']
        if result_left[1].size == 0:
            node['right'] = get_leaf(result_right[1])
        elif result_right[1].size == 0:
            node['right'] = get_leaf(result_left[1])
        else:
            node['right'] = result
            split(node['right'], max_depth, min_size, depth + 1)

def train_tree(X_train, y_train, max_depth, min_size):
    root = get_best_split(X_train, y_train)
    split(root, max_depth, min_size, 1)
    return root

CONDITION = {'numerical': {'yes': '>=', 'no': '<'},
             'categorical': {'yes': 'to', 'no': 'to nie'}}
def visualize_tree(node, depth=0):
    if isinstance(node, dict):
        if type(node['value']) in [int, float]:
            condition = CONDITION['numerical']
        else:
            condition = CONDITION['categorical']
        print('{}|- X{} {} {}'.format(depth * '  ', node['index'] + 1, condition['no'], node['value']))
        if 'left' in node:
            visualize_tree(node['left'], depth + 1)
        print('{}|- X{} {} {}'.format(depth * '  ', node['index'] + 1, condition['yes'], node['value']))
        if 'right' in node:
            visualize_tree(node['right'], depth + 1)
    else:
        print('{}[{}]'.format(depth * '  ', node))


X_train = np.array([['bliźniak', 3],
                    ['jednorodzinny', 2],
                    ['jednorodzinny', 3],
                    ['bliźniak', 2],
                    ['bliźniak', 4]], dtype=object)

y_train = np.array([600, 700, 800, 400, 700])

tree = train_tree(X_train, y_train, 2, 2)
visualize_tree(tree)

# Bezpośrednie użycie modułu DecisionTreeRegressor z pakietu scikit-learn
from sklearn import datasets
boston = datasets.load_boston()

num_test = 10    # 10 ostatnich próbek tworzy zbiór testowy
X_train = boston.data[:-num_test, :]
y_train = boston.target[:-num_test]
X_test = boston.data[-num_test:, :]
y_test = boston.target[-num_test:]

from sklearn.tree import DecisionTreeRegressor
regressor = DecisionTreeRegressor(max_depth=10, min_samples_split=3)

regressor.fit(X_train, y_train)
predictions = regressor.predict(X_test)
print(predictions)
print(y_test)

from sklearn.ensemble import RandomForestRegressor
regressor = RandomForestRegressor(n_estimators=100, max_depth=10, min_samples_split=3)
regressor.fit(X_train, y_train)
predictions = regressor.predict(X_test)
print(predictions)