﻿# starttweetstream.py

"""Przechwytywanie tweetów na temat określony przez argument(y)
   skryptu i przesyłanie ich treści do gniazda w celu przetworzenia
   przez Spark"""   
import keys
import socket
import sys
import tweepy

class TweetListener(tweepy.StreamListener):
    """Obsługuje strumień nadchodzących tweetów."""

    def __init__(self, api, connection, limit=10000):
        """Utworzenie zmiennych obiektu kontrolujących liczbę tweetow."""
        self.connection = connection
        self.tweet_count = 0
        self.TWEET_LIMIT = limit  
        super().__init__(api)  # wywołanie __init__ z nadklasy

    def on_connect(self):
        """Metoda wywoływana, gdy połączenie zostaje pomyślnie
           nawiązane. Możliwe staje się wykonywanie zadań wymagających 
           tego połączenia"""
        print('Nawiązano połączenie\n')

    def on_status(self, status):
        """Metoda wywoływana, gdy Twitter dostarcza nowy tweet"""
        # pobierz hashtagi
        hashtags = []

        for hashtag_dict in status.entities['hashtags']:
            hashtags.append(hashtag_dict['text'].lower())

        hashtags_string = ' '.join(hashtags) + '\n'
        print(f'Nazwa ekranowa: {status.user.screen_name}:')
        print(f'      Hashtagi: {hashtags_string}')
        self.tweet_count += 1  # kontrola liczby przetworzonych tweetów
                
        try:
            # wysłanie wymaganych bajtów zakodowanych w formacie utf-8
            self.connection.send(hashtags_string.encode('utf-8'))  
        except Exception as e:
            print(f'Błąd: {e}')

        # gdy osiągnięty zostanie limit przetworzonych tweetów
        # (TWEET_LIMIT) zwróć False w celu przerwania strumieniowania,
        # w przeciwnym razie zwróć True.
        return self.tweet_count != self.TWEET_LIMIT
    
    def on_error(self, status):
        print(status)
        return True
        
if __name__ == '__main__':
    tweet_limit = int(sys.argv[1])  # pobierz maksymalną liczbę tweetów
    client_socket = socket.socket() # utworz gniazdo 
    
    # aplikacja wykorzystuje port 9876 na komputerze lokalnym
    client_socket.bind(('localhost', 9876))  
 
    print('Oczekiwanie na połączenie')
    client_socket.listen()  # poczekaj na klienta
    
    # pobierz adres klienta
    connection, address = client_socket.accept()  
    print(f'Połączenie z {address}')
 
    # konfiguruj dostęp do Twittera
    auth = tweepy.OAuthHandler(keys.consumer_key, keys.consumer_secret)
    auth.set_access_token(keys.access_token, keys.access_token_secret)
    
    #  skonfiguruj Tweepy w kontekście limitu wywolań metod
    api = tweepy.API(auth, wait_on_rate_limit=True, 
                     wait_on_rate_limit_notify=True)               
 
    # utworzenie strumienia
    twitter_stream = tweepy.Stream(api.auth, 
        TweetListener(api, connection, tweet_limit))

    # sys.argv[2] od argmunetu [2] rozpozyna się lista tematów 
    twitter_stream.filter(track=sys.argv[2:]) 

    connection.close()
    client_socket.close()

##########################################################################
# (C) Copyright 2019 by Deitel & Associates, Inc. and                    #
# Pearson Education, Inc. All Rights Reserved.                           #
#                                                                        #
# DISCLAIMER: The authors and publisher of this book have used their     #
# best efforts in preparing the book. These efforts include the          #
# development, research, and testing of the theories and programs        #
# to determine their effectiveness. The authors and publisher make       #
# no warranty of any kind, expressed or implied, with regard to these    #
# programs or to the documentation contained in these books. The authors #
# and publisher shall not be liable in any event for incidental or       #
# consequential damages in connection with, or arising out of, the       #
# furnishing, performance, or use of these programs.                     #
##########################################################################
